#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <errno.h>
#include <sys/time.h>
#include <sys/ioctl.h>
#include <sys/poll.h>

#include <pp/features.h>
#include <pp/base.h>

#include "driver.h"
#include "comm_proto_pppp.h"

int send_data_i2c(comm_proto_data_t* context, int slen, const char* s);
int read_data_i2c(comm_proto_data_t* context, int* nn, char* buf);

#define I2C_DEBUG	0
#define KME_DEBUG	0

#if I2C_DEBUG
#define DBG(fmt, x...)	printf(fmt, ##x)
#else
#define DBG(fmt, x...)
#endif

int init_comm_proto_pppp_i2c(driver_t* obj, int reset_ioctl_fd) {
    int r;
    comm_proto_data_t* context;

    assert(obj);

    obj->comm_proto.id = __FUNCTION__;
    r = init_comm_proto_pppp(obj, reset_ioctl_fd);
    if (r) return r;
    context = (comm_proto_data_t*)obj->comm_proto.data;
    context->i2c_addr = 0x56;
    context->read_data = read_data_i2c;
    context->send_data = send_data_i2c;
    return r;
}

int send_data_i2c(comm_proto_data_t* context, int slen, const char* s) {
    int ret;

    assert(context);

#ifdef PP_KM_KME_DEBUG
    printf(">>>%s", s);
#endif /* PP_KM_KME_DEBUG */
        
    while ((ret = write(context->kmefd, s, slen)) != slen) {
	if (ret < 0) {
	    if (errno == EINTR) {
		continue;
	    } else {
		pp_log_err("%s: write failed", ___F);
		return 1;
	    }
	} else {
	    pp_log("write: obscure error: only %d of %d bytes sent\n",
		   ret, slen);
	    errno = EIO;
	    return 1;
	}
    }
    DBG("write successful: %d.\n", ret);
    return 0;
}

int read_data_i2c(comm_proto_data_t* context, int* nn, char* buf) {
    int ret;
    struct timeval to;
    fd_set kmeset;
#if KME_DEBUG
    char *dbg_string;
#endif
    
    assert(context);

    to.tv_sec = KME_RECV_THREAD_TIMEOUT;
    to.tv_usec = 0;
    FD_ZERO(&kmeset);
    FD_SET(context->kmefd, &kmeset);

    /* wait that something is going to happen */
    if((ret = select(context->kmefd+1, &kmeset, NULL, NULL, &to)) < 0) {
	perror("THREAD: kme_wait_for_prompt: select");
	if(EINTR == errno) return 1;
	return 0;
    }

    if(!FD_ISSET(context->kmefd, &kmeset)) { // timeout
	/* just try it again */
	DBG("read_data_i2c: no data available\n");
	return 0;
    }

    // OK, we've really got something...
    if((*nn = read(context->kmefd, buf, MAX_BUF_LEN-1)) < 0) {
	perror("THREAD: kme_wait_for_prompt: read");
	if(EINTR == errno) return 1;
	return 0;
    }

    DBG("read_data_i2c: read %d bytes.\n", *nn);
    
#if KME_DEBUG
    dbg_string = malloc((*nn) + 1);
    memcpy(dbg_string, buf, *nn);
    dbg_string[*nn] = '\0';
    
    printf(dbg_string);
    
    free(dbg_string);
#endif

    if(0 == *nn) return 1; // nothing to do
    return 2;
}
