/* layer: comm_proto_belkin
   data path: send_pdu() -> library function
*/

/* system includes */
#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <malloc.h>
#include <sched.h>

/* firmware includes */
#include <pp/base.h>
#include <pp/usb.h> /* to call into the USB lib */

/* local includes */
#include "driver.h"

/* object functions */
static int cleanup(driver_t* obj);
static int reconfigure(driver_t* obj);
static int send_pdu(driver_t*, const void* pdu, const int length, const input_t input);
static int suspend(driver_t*);
static int resume(driver_t*);
static int ping(driver_t*, char** info);
static int is_equal(driver_t* obj, driver_t* d);
static int reset_driver(driver_t* obj);

typedef struct {
    usb_device_t dev;
    usb_device_mouse_type_t mouse_type;
} usb_data_t;

/* local object functions */
int init_comm_proto_usb(driver_t* obj, usb_device_t dev, usb_device_mouse_type_t mouse_type) {
    /* create the driver */
    usb_data_t *usb_data;

    assert(obj);
    
    obj->comm_proto.id = __FUNCTION__;
    obj->comm_proto.data = malloc(sizeof(usb_data_t));
    usb_data = (usb_data_t*)obj->comm_proto.data;
    assert(usb_data);
    usb_data->dev = dev;
    usb_data->mouse_type = mouse_type;
    obj->comm_proto.is_equal = is_equal;
    obj->comm_proto.cleanup = cleanup;
    obj->comm_proto.reconfigure = reconfigure;
    obj->comm_proto.send_pdu = send_pdu;
    obj->comm_proto.suspend = suspend;
    obj->comm_proto.resume = resume;
    obj->comm_proto.ping = ping;
    obj->comm_proto.reset_driver = reset_driver;
    return 0;
}

static int is_equal(driver_t* obj, driver_t* d) {
    usb_data_t* usb_data;
    usb_data_t* usb_data_d;

    assert(obj);
    usb_data = (usb_data_t*)obj->comm_proto.data;    
    assert(usb_data);

    usb_data_d = (usb_data_t*)d->comm_proto.data;    
    assert(usb_data_d);

    
    if (!strcmp(obj->comm_proto.id, d->comm_proto.id)) {
	// check additional parameters
	if ((usb_data->dev == usb_data_d->dev) &&
	    (usb_data->mouse_type == usb_data_d->mouse_type)) {
	    // driver identically
	    return 1;
	} else {
	    return 0;
	}
    } else {
	return 0;
    }
}

static int reset_driver(driver_t * /* obj */ ) {
    return pp_usb_reset_device();
}

static int cleanup(driver_t * /* obj */ ) {
    return 0;
}

static int ping(driver_t * /* obj */, char** info) {
    if (info) {
	*info = strdup("pong");
    }
    return 1;
}

static int suspend (driver_t * /* obj */) {
    return 0;
}

static int resume (driver_t* obj) {
    usb_data_t* usb_data;
    struct timeval now;
    struct timespec to;

    assert(obj);
    usb_data = (usb_data_t*)obj->comm_proto.data;    
    assert(usb_data);

    pthread_mutex_lock(&usb_propchange_mutex);
    pp_usb_set_km(usb_device_kbd_enabled, usb_data->mouse_type);

    /* This is a rather awkward hack to keep the propchange handler from
       calling pp_km_reconfigure() again as a reaction to potential
       suspend/resume events caused by pp_usb_set_device():
          1. Wait up to two seconds for a suspend event
          2. If the device is suspended give the OS a few more seconds to
             resume it
       Any suspend/resume events during that interval are ignored, i.e.
       don't cause a reconfiguration. */
    ignore_usb_propchange = 1;
    gettimeofday(&now, NULL);
    to.tv_sec = now.tv_sec + 2;
    to.tv_nsec = now.tv_usec * 1000;
    if (pthread_cond_timedwait(&usb_suspend_cond, &usb_propchange_mutex, &to) == 0) {
	to.tv_sec = now.tv_sec + 5;
	to.tv_nsec = now.tv_usec * 1000;
	pthread_cond_timedwait(&usb_resume_cond, &usb_propchange_mutex, &to);
    }
    ignore_usb_propchange = 0;

    pthread_mutex_unlock(&usb_propchange_mutex);

    return 0;
}

static int reconfigure(driver_t * /* obj */ ) {
    return 0;
}
    
static int send_pdu(driver_t* obj, const void* pdu, const int length, const input_t input) {
    /* obj blocks until data is sent */
    pp_usb_send_data(obj->data_link, input, pdu, length);

    return 0;
}
