/* layer: data_conv_mouse
   data path: data_conv->send_ptrmove() -> data_unit_mouse->send_mouse_data()
*/

/* system includes */
#include <stdlib.h>
#include <stdio.h>
#include <sys/ioctl.h>

/* firmware includes */
#include <lara.h>
#include <pp/base.h>
#include <pp/grab.h>
#include <pp/km.h>
#include <pp/propchange.h>

/* local includes */
#include "driver.h"

/* data converter for absolute USB mouse */

static int cleanup(driver_t* obj);
static int reconfigure(driver_t* obj);
static int send_ptrmove(driver_t* obj,
			const int x, const int y, const int z, 
			const unsigned char buttonmask,
			const unsigned char abs_rel);
static int sync_mouse (driver_t* obj, const int sync_type);
static int resume(driver_t* obj);
static int suspend(driver_t* obj);
static int is_equal(driver_t* obj, driver_t* d);


static void propchange_handler(pp_propchange_listener_t * listener, 
			       unsigned short prop, unsigned short prop_flags);

typedef struct {
    pp_propchange_listener_t propchange_listener;
    driver_t* obj;
} local_propchange_listener_t;


typedef struct {
    int min_x, max_x, min_y, max_y;
    int x,y;
    local_propchange_listener_t propchange_listener;
    pthread_mutex_t xyres_mtx;
} ausb_mouse_data_t;



int init_data_conv_mouse_ausb(driver_t* obj) {
    ausb_mouse_data_t *md;
    assert(obj);

    /* create the driver */
    obj->data_conv_mouse.id = __FUNCTION__;
    obj->data_conv_mouse.is_equal = is_equal;
    
    obj->data_conv_mouse.resume = resume;
    obj->data_conv_mouse.suspend = suspend;
    obj->data_conv_mouse.cleanup = cleanup;
    obj->data_conv_mouse.reconfigure = reconfigure;
    obj->data_conv_mouse.send_ptrmove = send_ptrmove;
    obj->data_conv_mouse.sync_mouse = sync_mouse;
    obj->data_conv_mouse.data = malloc(sizeof(ausb_mouse_data_t));
    
    /* initializes our private data */
    md = (ausb_mouse_data_t*) obj->data_conv_mouse.data;
    assert(md);

    pthread_mutex_init(&md->xyres_mtx, NULL);
    pp_propchange_init_listener((pp_propchange_listener_t *)&md->propchange_listener, propchange_handler);
    md->propchange_listener.obj = obj;
    pp_propchange_add((pp_propchange_listener_t *)&md->propchange_listener, PP_PROP_VIDEO_MODE_CHANGED);

    MUTEX_LOCK(&md->xyres_mtx);
    /* some default values */
    md->min_x = 0;
    md->max_x = 800;
    md->min_y = 0;
    md->max_y = 600;
    MUTEX_UNLOCK(&md->xyres_mtx);
    
    md->x = 0;
    md->y = 0;
    return 0;
}

static int resume(driver_t * /* obj */) {
    return 0;
}

static int suspend(driver_t * /* obj */) {
    return 0;
}

static int is_equal(driver_t* obj, driver_t* d) {
    assert(obj);
    assert(d);
    
    if (!strcmp(obj->data_conv_mouse.id, d->data_conv_mouse.id)) {
	return 1;
    } else {
	return 0;
    }
}

static int cleanup(driver_t* obj) {
    ausb_mouse_data_t *mouse_data;

    assert(obj);
    mouse_data = (ausb_mouse_data_t*)obj->data_conv_mouse.data;
    assert(mouse_data);
    
    pp_propchange_remove_all((pp_propchange_listener_t *)&mouse_data->propchange_listener);
    free(obj->data_conv_mouse.data);
    
    return 0;
}

static void
propchange_handler(pp_propchange_listener_t * listener, 
		   unsigned short prop, unsigned short /* prop_flags */)
{
    if (prop == PP_PROP_VIDEO_MODE_CHANGED) {
	local_propchange_listener_t *mylistener = (local_propchange_listener_t *)listener;
	driver_t* obj = mylistener->obj;
	obj->data_conv_mouse.reconfigure(obj);
    } else {
	pp_log("data_conv_mouse_absolute_usb: %s(): Unhandled property %hu received.\n", ___F, prop);
    }
}

static int reconfigure(driver_t* obj) {
    ausb_mouse_data_t* mouse_data;
    fb_format_info_t fb_f_info;
    assert(obj);
    mouse_data = (ausb_mouse_data_t*)obj->data_conv_mouse.data;
    assert(mouse_data);

    if (pp_grab_get_format_info(obj->video_link, &fb_f_info) != 0) {
	    return -1;
    }

    if (fb_f_info.g_w > 0 && fb_f_info.g_h > 0) {
	MUTEX_LOCK(&mouse_data->xyres_mtx);
	mouse_data->max_x = fb_f_info.g_w;
	mouse_data->max_y = fb_f_info.g_h;
	MUTEX_UNLOCK(&mouse_data->xyres_mtx);
	return 0;
    }

    return -1;
}

static int send_ptrmove(driver_t* obj,
			const int x, const int y, const int z, 
			const unsigned char buttonmask,
			const unsigned char abs_rel) {
    ausb_mouse_data_t *md;
    assert(obj);
    md = (ausb_mouse_data_t*) obj->data_conv_mouse.data;
    assert(md);

    /* no conversion necessary */
    if (abs_rel == PP_KM_PTR_MOVE_RELATIVE) {
	if (!z) {
	    md->x += x;
	    md->y += y;
	    MUTEX_LOCK(&md->xyres_mtx);
	    if (md->x < md->min_x) md->x = md->min_x;
	    if (md->x > md->max_x) md->x = md->max_x;
	    if (md->y < md->min_y) md->y = md->min_y;
	    if (md->y > md->max_y) md->y = md->max_y;
	    KD("data_conv ptrmove relativ: rx: %d, ry: %d, ax: %d, ay: %d\n", x, y, md->x, md->y);
	    MUTEX_UNLOCK(&md->xyres_mtx);
	    return obj->data_unit_mouse.send_mouse_data(obj, md->x, md->y, 0, buttonmask);
	} else {
	    // wheel event set x,y implicit to zero
	    KD("data_conv ptrmove: wheel event\n");
	    return obj->data_unit_mouse.send_mouse_data(obj, 0, 0, z, buttonmask);
	}
    }
    
    return obj->data_unit_mouse.send_mouse_data(obj, x, y, z, buttonmask);
}

static int sync_mouse(driver_t * /* obj */, const int /* sync_type */) {
    /* the absolute USB mouse is always synchron  - do nothing */
    return 0;
}
