/*****************************************************************************
*                                                                            *
* Implements internal functions to move the mouse with the help of	     *
* the LARA Mouse Universe[tm]						     *
*                                                                            *
*****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <pthread.h>
#include <fcntl.h>
#include <math.h>

#include <pp/base.h>
#include <pp/intl.h>
#include <pp/cfg.h>
#include <liberic_config.h>
#include <pp/grab.h>
#include <pp/xdefs.h>
#include <pp/rfb.h>

#include "iipptr_internal.h"
#include "iipptr_help.h"
#include "iipptr_cursor.h"
#include "driver.h"

#if defined(PP_FEAT_KITTY_CAT)
# include "cat_internal.h"
# include "cat_debug.h"
#endif /* PP_FEAT_KITTY_CAT */

/*
 * PPDU_MAX_LENGTH		low level pdu size
 * BUILD_INITIAL_SLEEP		detection: waiting time after move to corner (us)
 * MOUSE_SEGMENT_MIN_SIZE	detection: if a segment is smaller, give warning
 * REGRESSION_COUNT		detection: number of samples
 * BUILD_SLEEP			detection: waiting time after move to position (us)
 * MOUSE_MAX_DIFF_COUNT		maximum number of consecutive moves during a real move
 * MAX_M                       maximum sane value for an acceleration segment
 */
#define PPDU_MAX_LENGTH		4
#define BUILD_INITIAL_SLEEP	10000
#define MOUSE_SEGMENT_MIN_SIZE  3
#define REGRESSION_COUNT	20
#define BUILD_SLEEP		100000
#define MOUSE_MAX_DIFF_COUNT    5
#define MAX_M                   15.0

/* obj comes from ptrbase.c
   it is needed here because the it is used as low-level routine here */

static int detect_mouse_universe(driver_t* obj, int debug);
static int check_mouse_error(driver_t* obj);

static long  mouse_virt_in2out(driver_t* obj, long x);
static long  mouse_virt_out2in(driver_t* obj, long y);
#ifdef PP_FEAT_KITTY_CAT
static float fabs(float x);
#endif
static long  sign(long x);

// 1:1 correction funtion, i.e. no equalization
static long  non_fm_corr(long i);

// Guntermann & Drunck equalization funktions
static int gud_find_eqz_fkt(int eqz_fkt_id);
static int gud_detect_mouse_universe(driver_t* obj,
				     int eqzfkt, int debug);
static long  gud3_fm_corr_in2out(long i);
static long  gud3_fm_corr_out2in(long o);
static long  gud2_fm_corr_in2out(long i);
static long  gud2_fm_corr_out2in(long o);
static long  gud1_fm_corr_in2out(long i);
static long  gud1_fm_corr_out2in(long o);
static long  gud0_fm_corr_in2out(long i);
static long  gud0_fm_corr_out2in(long o);

// mouse correction function ids
#define NON_FM_CORR_ID 0

// initialize the default mouse correction function
t_mouse_func non_mouse_corr = {
    NON_FM_CORR_ID, non_fm_corr, non_fm_corr
};

// initialize all G&D correction functions, that we need to try
#define GUD_FM_CORR_NR 4
t_mouse_func gud_mouse_corr[GUD_FM_CORR_NR] = {
    { 8, gud3_fm_corr_in2out, gud3_fm_corr_out2in },
    { 4, gud2_fm_corr_in2out, gud2_fm_corr_out2in },
    { 2, gud1_fm_corr_in2out, gud1_fm_corr_out2in },
    { 1, gud0_fm_corr_in2out, gud0_fm_corr_out2in },
};

static const char * unit_port_mouse_key		= "unit[%u].port[%u].mouse.%s";
static const char * unit_port_accel_key		= "unit[%u].port[%u].mouse.accel[%u].%s";
static const char * unit_port_accel_count_key	= "unit[%u].port[%u].mouse.accel._s_";
static const char * accel_m_key		= "m";
static const char * accel_x0_key	= "x0";
static const char * accel_xmax_key	= "xmax";
static const char * accel_ymax_key	= "ymax";
static const char * gud_eqz_key		= "gud_eqz";
static const char * gud_eqz_id_key	= "gud_eqz_id";
static const char * gud_eqz_preset_key	= "gud_eqz_preset";

/* save the mouse universe to config fs */
int
save_mouse_universe(driver_t* obj)
{
    data_imouse_t* mouse_data = (data_imouse_t*) obj->data_conv_mouse.data;
    km_kvm_entry_t * port = mouse_data->current_kvm_port;
    t_mouse_data* md = &port->mouse_data;
    u_char kvm_unit = port->u_index;
    u_short kvm_port = port->index;
    int i, ret = PP_SUC;
    
    if (md->fm_corr.id != NON_FM_CORR_ID) {
	pp_cfg_set_int(md->fm_corr.id, unit_port_mouse_key, kvm_unit, kvm_port, gud_eqz_id_key);
    }    
    pp_cfg_set_int(md->fm_corr.id, unit_port_mouse_key, kvm_unit, kvm_port, gud_eqz_id_key);
    pp_cfg_set_int(md->accel_steps, unit_port_accel_count_key, kvm_unit, kvm_port, md->accel_steps);

    KD("%s(): unit %d, port %d, %d accel_steps\n", ___F, kvm_unit, kvm_port, md->accel_steps);
    for (i = 0; i < md->accel_steps; i++) {
	KD("%s(): m=%f, x0=%d, xmax=%d, ymax=%d\n", ___F, md->m[i], md->x0[i], md->xmax[i], md->ymax[i]);

	if ((pp_cfg_set_float(md->m[i], unit_port_accel_key, kvm_unit, kvm_port, i, accel_m_key) != PP_SUC) |
	    (pp_cfg_set_int(md->x0[i], unit_port_accel_key, kvm_unit, kvm_port, i, accel_x0_key) != PP_SUC) |
	    (pp_cfg_set_int(md->xmax[i], unit_port_accel_key, kvm_unit, kvm_port, i, accel_xmax_key) != PP_SUC) |
	    (pp_cfg_set_int(md->ymax[i], unit_port_accel_key, kvm_unit, kvm_port, i, accel_ymax_key) != PP_SUC)) {

	    KD("%s(): errors saving config\n", ___F);
	    ret = PP_ERR;
	    goto bail;
	}
    }

    ret = pp_cfg_save(DONT_FLUSH);
    
 bail:
    return ret;
}

int
load_mouse_universe(driver_t * /* obj */, km_kvm_entry_t* port)
{
    t_mouse_data* md = &port->mouse_data;
    u_char kvm_unit = port->u_index;
    u_short kvm_port = port->index;
    int eqz_id, idx, i, ret = PP_SUC;

    if (PP_FAILED(pp_cfg_get_int_nodflt(&md->accel_steps,
                                        unit_port_accel_count_key,
					kvm_unit, kvm_port))) {
	ret = PP_ERR;
	goto bail;
    }

    KD("%s: unit %d, port %d, %d accel_steps\n", ___F, kvm_unit, kvm_port, md->accel_steps);
    for (i = 0; i < md->accel_steps; i++) {
	if ((pp_cfg_get_float(&md->m[i], unit_port_accel_key, kvm_unit, kvm_port, i, accel_m_key) != PP_SUC) |
	    (pp_cfg_get_int(&md->x0[i], unit_port_accel_key, kvm_unit, kvm_port, i, accel_x0_key) != PP_SUC) |
	    (pp_cfg_get_int(&md->xmax[i], unit_port_accel_key, kvm_unit, kvm_port, i, accel_xmax_key) != PP_SUC) |
	    (pp_cfg_get_int(&md->ymax[i], unit_port_accel_key, kvm_unit, kvm_port, i, accel_ymax_key) != PP_SUC)) {

	    KD("%s: error loading config\n", ___F);
	    goto bail;
	}

	KD("%s: m=%f, x0=%d, xmax=%d, ymax=%d\n", ___F, md->m[i], md->x0[i], md->xmax[i], md->ymax[i]);
    }
    
    if (PP_SUCCED(pp_cfg_get_int(&eqz_id, unit_port_mouse_key, kvm_unit, kvm_port, gud_eqz_id_key))
	&& eqz_id > 0 && (idx = gud_find_eqz_fkt(eqz_id)) >= 0) {
	KD("load_mouse_universe: gud_mouse_corr[%u]\n", idx);
	md->fm_corr = gud_mouse_corr[idx];
    } else {
	KD("load_mouse_universe: non_mouse_corr!\n");
	md->fm_corr = non_mouse_corr;
    }

 bail:
    return ret;
}

int
get_mouse_universe(driver_t* obj)
{
    int debug = 0, ret = 0, gud_eqz;
    data_imouse_t* mouse_data = (data_imouse_t*) obj->data_conv_mouse.data;
    km_kvm_entry_t * port = mouse_data->current_kvm_port;
    u_char kvm_unit = port->u_index;
    u_short kvm_port = port->index;

    pp_cfg_is_enabled(&debug, "iipptr_debug");
#if !defined(PP_FEAT_KITTY_CAT)
    /* add grab client for grabbing mouse sync video data, has to succeed */
    if ((mouse_data->grab_client = pp_grab_new_client(obj->video_link, GRAB_FIX_MEM_DESC_MOUSESYNC)) == NULL) {
	goto bailout;
    }
#endif /* PP_FEAT_KITTY_CAT */

    if (get_mousecursor_shape(obj) != 0) {
	ret = ERIC_MOUSE_UNIVERSE_FAILED;
	goto bailout;
    }

    pp_cfg_is_enabled(&gud_eqz, unit_port_mouse_key, kvm_unit, kvm_port, gud_eqz_key);
    if (!gud_eqz) { /* not GuD */
	mouse_data->current_kvm_port->mouse_data.fm_corr = non_mouse_corr;
	ret = detect_mouse_universe(obj, debug);
    } else { /* GuD: detect GuD Equalizer function via "try and error" */
	int eqzfkt = 0;
	pp_cfg_get_int(&eqzfkt, unit_port_mouse_key, kvm_unit, kvm_port, gud_eqz_preset_key);
	ret = gud_detect_mouse_universe(obj, eqzfkt, debug);
    }

 bailout:
#if !defined(PP_FEAT_KITTY_CAT)
    if (mouse_data->grab_client) pp_grab_remove_client(mouse_data->grab_client);
#endif /* PP_FEAT_KITTY_CAT */
    return ret;
}

/* find GuD Equalizer function by id and returns its index in GuD table*/
static int
gud_find_eqz_fkt(int eqz_fkt_id) {
    int i;
    for (i = 0; i < GUD_FM_CORR_NR; ++i) {
	if (gud_mouse_corr[i].id == eqz_fkt_id) {
	    return i;
	}
    }
    return -1;
}

/* GuD: detect GuD Equalizer function */
static int
gud_detect_mouse_universe(driver_t* obj, int eqzfkt, int debug)
{
    int i;
    int res[GUD_FM_CORR_NR][2];
    data_imouse_t* mouse_data = (data_imouse_t*) obj->data_conv_mouse.data;
    t_mouse_data* md = &mouse_data->current_kvm_port->mouse_data;

#define RET 0
#define ERR 1
    if (eqzfkt != 0 &&
	(i = gud_find_eqz_fkt(eqzfkt)) >= 0) { // use a preselected function
	mouse_data->current_kvm_port->mouse_data.fm_corr = gud_mouse_corr[i];
	res[i][RET] = detect_mouse_universe(obj, debug);
	pp_log("gud_detect: preselected GuD no %d: ret=%d\n",
		 gud_mouse_corr[i].id, res[i][RET]);
    } else {     // auto or eqzfkt was wrong
	for (i = 0; i < GUD_FM_CORR_NR; ++i) {
	    mouse_data->current_kvm_port->mouse_data.fm_corr = gud_mouse_corr[i];
	    pp_log("gud_detect: try GuD no %d\n", gud_mouse_corr[i].id);
	    res[i][RET] = detect_mouse_universe(obj, debug);
	    if (res[i][RET] == ERIC_MOUSE_UNIVERSE_OK
		|| res[i][RET] == ERIC_MOUSE_UNIVERSE_WARN_GOOD) {
		if(8 >= (res[i][ERR] = check_mouse_error(obj))) {
		    break;
		}
	    } else {
		res[i][ERR] = -1;
	    }
	    pp_log("gud_detect: GuD no %d: ret=%d err=%d\n",
	      gud_mouse_corr[i].id, res[i][RET], res[i][ERR]);
	}

	// in case we have no perfect result,
	// take that one with the smallest error
	if(!(i < GUD_FM_CORR_NR)) {
	    int minidx = 0;
	    for (i = 1; i < GUD_FM_CORR_NR; ++i) {
		if (res[i][ERR] < res[minidx][ERR])
		    minidx = i;
	    }
	    i = minidx;
	}
	pp_log("gud_detect: selected GuD no %d: ret=%d err=%d\n",
	  gud_mouse_corr[i].id, res[i][RET], res[i][ERR]);
    }
    md->fm_corr = gud_mouse_corr[i];
    return res[i][RET];
}
	
/* wrap KD to CDMSG for KITTY_CAT */
#if !defined(CAT_DEBUG)
# define _KD KD
#else /* CAT_DEBUG */
# if defined(KBD_DEBUG)
#  define _KD_DEBUG 1
# else /* KBD_DEBUG */
#  define _KD_DEBUG 0
# endif /* KBD_DEBUG */
# define _KD(args...) CDMSG(_KD_DEBUG || CAT_CDMSG_SYNC_INFO, ##args)
#endif /* CAT_DEBUG */

static int
detect_mouse_universe(driver_t* obj, int debug)
{
    int xi[REGRESSION_COUNT], yi[REGRESSION_COUNT];
    t_point mouse_cursor_point={0,0}, mouse_cursor_expect={0,0};
    double speed=.0, snapped=.0, diff=.0, maxdiff=.0, snap=.0, x0=.0,
	speeds[MAX_ACCEL_STEPS];
    double mulsum=.0, xsum=.0, ysum=.0, xsqrsum=.0,
	est=.0, m_fac_real=.0, m_fac=.0;
    int	i, j, step, count_all, count=0, start=0, m_fac_i;
    int ret = ERIC_MOUSE_UNIVERSE_OK;
    char debug_msg[100];
    data_imouse_t* mouse_data = (data_imouse_t*) obj->data_conv_mouse.data;
    t_mouse_data* md = &mouse_data->current_kvm_port->mouse_data;
    km_event_t event;
    struct list_head *ptr;
    km_encoder_desc_data_t *enc_data;
#if defined(PP_FEAT_KITTY_CAT)
    u_int16_t *shape_fb_rgb = NULL;

#if defined(CAT_DEBUG)
    /* wrap debug */
    debug |= CAT_CDMSG_SYNC_INFO;
#endif /* CAT_DEBUG */
    
    if(PP_ERR == cat_bmp2fb_rgb(&shape_fb_rgb,
                                &mouse_data->current_cursor_shape)) {
        _KD("could not convert shape to frame buffer RGB!\n");
        abort();
        goto bailout;
    }
#endif /* PP_FEAT_KITTY_CAT */
    
    md->state = ERIC_MOUSE_GETTING;
    md->accel_steps = 0;

    /* get mouse movements */
    if(0 != move_mouse_corner(obj, 0))
	goto bailout;
	
    mouse_cursor_expect.pos_x = 0;

    /* we need to make sure, that we are collecting at least
     * REGRESSION_COUNT steps at the host's mouse input indepent
     * of the mouse correction function */
    for (i = step = 0; i < REGRESSION_COUNT; step++) {

	// for the regression funktion we need to take the value
	// after the correction introduced by some other device
	xi[i] = md->fm_corr.fm_in2out(step);

	// check whether we really have the next step,
	// if not, just jump over obj step and try the next one
	if (i > 0 && xi[i] == xi[i - 1]) continue;

	// make step 
	if(0 != move_mouse_direct(obj, step, 0, 0, 0, 0))
	    goto bailout;
	usleep(BUILD_SLEEP);

	// determine step 
	if (i>0) mouse_cursor_expect.pos_x = yi[i-1];
	mouse_cursor_expect.pos_y = 0;
#if defined(PP_FEAT_KITTY_CAT)
        cat_find_cursor_for_table_fb(obj, shape_fb_rgb,
                                     &mouse_cursor_expect, &mouse_cursor_point);
#else /* PP_FEAT_KITTY_CAT */
        /* FIXME: find_cursor_for_table() does not work for cropped cursors */
	find_cursor_for_table(obj,&mouse_cursor_expect,
			      &mouse_cursor_point);
#endif /* PP_FEAT_KITTY_CAT */

	yi[i] = mouse_cursor_point.pos_x;

	// back to the roots
	if(0 != move_mouse_direct(obj, mouse_cursor_point.pos_x * -2,
				  0, 0, 0, 0))
	    goto bailout;

	snprintf(debug_msg, sizeof(debug_msg), "Step=%d Corr=%d Dist=%d",
		 step, xi[i], mouse_cursor_point.pos_x);
	_KD("%s\n", debug_msg);
	if ((debug)) {
            event.type = KM_EVENT_DEBUG_MSG;
            event.event.debug_msg.msg = debug_msg;
            MUTEX_LOCK(&km_encoder_hook_mtx);
            list_for_each(ptr, &km_encoder_hook_list) {
            	enc_data = list_entry(ptr, km_encoder_desc_data_t, listnode);
            	if (enc_data->encoder_hook && enc_data->encoder_hook->km_event) {
            	    enc_data->encoder_hook->km_event(enc_data->encoder_hook, obj->data_link, &event);
            	}
            }
            MUTEX_UNLOCK(&km_encoder_hook_mtx);
	}
	i++;
    }

    /* calculate linear regression for movements */
    count_all = REGRESSION_COUNT;
    if (count_all > 2 && yi[1] > yi[0] && xi[1] > xi[0]) {
        _KD("Hard coding step 1.\n");
        
	speeds[md->accel_steps]		= ((double)(yi[1] - yi[0])) / ((double)(xi[1] - xi[0]));
	md->xmax[md->accel_steps]	= xi[1] * MOUSE_RASTER;
	md->ymax[md->accel_steps]	= yi[1] * MOUSE_RASTER;
	md->x0[md->accel_steps]		= (int) x0 * MOUSE_RASTER;
	md->accel_steps++;

        _KD("  yi[1]=%d yi[0]=%d xi[1]=%d xi[0]=%d --> speed=%02.04f\n", yi[1], yi[0], xi[1], xi[0], speeds[md->accel_steps]);
	start = 2;
    }
    for (i = start; i < count_all; i++) {
	mulsum	+= xi[i] * yi[i];
	xsum	+= xi[i];
	ysum	+= yi[i];
	xsqrsum += xi[i] * xi[i];

	snprintf(debug_msg, sizeof(debug_msg),
		 "i=%d, m/x0 now=%02.04f,%02.04f", i, speed, x0);
	_KD("%s\n", debug_msg);
	if ((debug)) {
            event.type = KM_EVENT_DEBUG_MSG;
            event.event.debug_msg.msg = debug_msg;
            MUTEX_LOCK(&km_encoder_hook_mtx);
            list_for_each(ptr, &km_encoder_hook_list) {
            	enc_data = list_entry(ptr, km_encoder_desc_data_t, listnode);
            	if (enc_data->encoder_hook && enc_data->encoder_hook->km_event) {
            	    enc_data->encoder_hook->km_event(enc_data->encoder_hook, obj->data_link, &event);
            	}
            }
            MUTEX_UNLOCK(&km_encoder_hook_mtx);
	}

	if ((speed > 0) && (i-start > 1)) {

	    maxdiff = .0;
	    for (j = start; j < i; j++) {
		est = speed * xi[j] + x0;
		KD("  j=%d est=%02.04f speed=%02.04f xi[%d]=%d x0=%02.04f\n", j, est, speed, j, xi[j], x0);
		KD("  yi[%d]=%d fabs(yi[%d]-est)=%02.04f maxdiff=%02.04f\n", j, yi[j], j, fabs(yi[j]-est), maxdiff);
		if (fabs(yi[j]-est) > maxdiff) {
		    maxdiff = fabs(yi[j]-est);
		    KD("    new maxdiff=%02.04f\n", maxdiff);
		}
	    }
		
	    est	 = speed * xi[i] + x0;
	    diff = fabs(yi[i]-est);
	    snprintf(debug_msg, sizeof(debug_msg),
		     "est=%02.04f, diff=%02.04f, maxdiff=%02.04f",
		     est, diff, maxdiff);
	    _KD("%s\n", debug_msg);
            if ((debug)) {
                event.type = KM_EVENT_DEBUG_MSG;
                event.event.debug_msg.msg = debug_msg;
            	MUTEX_LOCK(&km_encoder_hook_mtx);
            	list_for_each(ptr, &km_encoder_hook_list) {
            	    enc_data = list_entry(ptr, km_encoder_desc_data_t, listnode);
            	    if (enc_data->encoder_hook && enc_data->encoder_hook->km_event) {
            	    	enc_data->encoder_hook->km_event(enc_data->encoder_hook, obj->data_link, &event);
            	    }
            	}
            	MUTEX_UNLOCK(&km_encoder_hook_mtx);
            }

	    if (diff > (maxdiff+1)) {
		snprintf(debug_msg, sizeof(debug_msg),
			 "New Accel (old Speed,x0=%02.04f,%02.04f)",
			 speed, x0);
		_KD("%s\n", debug_msg);
                if ((debug)) {
                    event.type = KM_EVENT_DEBUG_MSG;
                    event.event.debug_msg.msg = debug_msg;
            	    MUTEX_LOCK(&km_encoder_hook_mtx);
            	    list_for_each(ptr, &km_encoder_hook_list) {
            	    	enc_data = list_entry(ptr, km_encoder_desc_data_t, listnode);
            	    	if (enc_data->encoder_hook && enc_data->encoder_hook->km_event) {
            	    	    enc_data->encoder_hook->km_event(enc_data->encoder_hook, obj->data_link, &event);
            	    	}
            	    }
            	    MUTEX_UNLOCK(&km_encoder_hook_mtx);
                }

		speeds[md->accel_steps]		= speed;
		md->xmax[md->accel_steps]	= xi[i] * MOUSE_RASTER;
		md->ymax[md->accel_steps]	= yi[i] * MOUSE_RASTER;
		md->x0[md->accel_steps]		= (int) x0 * MOUSE_RASTER;
		md->accel_steps++;

		start = i;
		mulsum = xsum = ysum = xsqrsum = count = 0;
	    }
	}

	count++;
	if (xsqrsum-(xsum*xsum)/count <= 0) {
	    speed = 0.0;
	} else {
	    speed = (mulsum - (xsum * ysum)/count)/(xsqrsum-(xsum*xsum)/count);
	}
	if (speed == 0.0) {
	    x0 = 0.0;
	} else {
	    x0 = (1/count) * (xsum - (1/speed)*ysum);
	}

	/* too many acceleration steps */
/* FIXME: how to get maximum accel steps from CD to avoid redundancy? */
	if (md->accel_steps > MAX_ACCEL_STEPS -1) {
	    ret = ERIC_MOUSE_UNIVERSE_WARN_BAD;
	    break;
	}
    }

    snprintf(debug_msg, sizeof(debug_msg),
	     "last m,x0=%02.04f,%02.04f", speed, x0);
    _KD("%s\n", debug_msg);

    if ((debug)) {
        event.type = KM_EVENT_DEBUG_MSG;
        event.event.debug_msg.msg = debug_msg;
        MUTEX_LOCK(&km_encoder_hook_mtx);
        list_for_each(ptr, &km_encoder_hook_list) {
            enc_data = list_entry(ptr, km_encoder_desc_data_t, listnode);
            if (enc_data->encoder_hook && enc_data->encoder_hook->km_event) {
            	enc_data->encoder_hook->km_event(enc_data->encoder_hook, obj->data_link, &event);
            }
        }
        MUTEX_UNLOCK(&km_encoder_hook_mtx);
    }

    speeds[md->accel_steps]	= speed;
    md->x0[md->accel_steps]	= (int) x0 * MOUSE_RASTER;
    md->accel_steps++;

    /* snap in accelerations to 0.25 steps */
    for (i = 0; i < md->accel_steps; i++) {
	snapped = speeds[i];
	maxdiff = 100.0;
	for (snap=0.25; snap<15.0; snap+=0.25) {
	    diff = fabs(snap - speeds[i]);
	    if (diff < maxdiff) {
		snapped = snap;
		maxdiff = diff;
	    }
	}

	md->m[i] = snapped;
    }   

    /* sanity check for impossible m values */
    for (i = 0; i < md->accel_steps; i++) {
       if (md->m[i] > MAX_M) {
           _KD("Impossible value for m found (%f for step %d)\n", md->m[i], i);
           goto bailout;
       }
    }
    
    /* acceleration steps must be whole-numbered multiples */
    if (md->accel_steps > 1) {
	for (i=md->accel_steps; i>0; i--) {
	    m_fac = md->m[i]/md->m[i-1];
	    m_fac_i = (int)m_fac;
	    if ((m_fac_i - m_fac) > 0.1) {
		if (!ret) ret = ERIC_MOUSE_UNIVERSE_WARN_GOOD;
		snprintf(debug_msg, sizeof(debug_msg),
			 "Corrected multiply for i=%d\n -> Warn", i);
		_KD("%s\n", debug_msg);
                if ((debug)) {
                    event.type = KM_EVENT_DEBUG_MSG;
                    event.event.debug_msg.msg = debug_msg;
            	    MUTEX_LOCK(&km_encoder_hook_mtx);
            	    list_for_each(ptr, &km_encoder_hook_list) {
            	    	enc_data = list_entry(ptr, km_encoder_desc_data_t, listnode);
            	    	if (enc_data->encoder_hook && enc_data->encoder_hook->km_event) {
            	    	    enc_data->encoder_hook->km_event(enc_data->encoder_hook, obj->data_link, &event);
            	    	}
            	    }
            	    MUTEX_UNLOCK(&km_encoder_hook_mtx);
                }
		md->m[i-1] = md->m[i]/m_fac_i;
	    }
	}
    }
    
    /* if we got more than 2 steps,
       try to correct them using their differences */
    if (md->accel_steps > 2) {
	m_fac_real = md->m[md->accel_steps-1]/md->m[md->accel_steps-2];

	for (i=md->accel_steps-2; i>0; i--) {
	    m_fac = md->m[i]/md->m[i-1];
	    if (m_fac != m_fac_real) {
		if (!ret) ret = ERIC_MOUSE_UNIVERSE_WARN_GOOD;
		snprintf(debug_msg, sizeof(debug_msg),
			 "%d-%d:m_real=%02.02f, m=fac=%02.02f -> correcting",
			 i, i-1, m_fac_real, m_fac);
		_KD("%s\n", debug_msg);
                if ((debug)) {
                    event.type = KM_EVENT_DEBUG_MSG;
                    event.event.debug_msg.msg = debug_msg;
            	    MUTEX_LOCK(&km_encoder_hook_mtx);
            	    list_for_each(ptr, &km_encoder_hook_list) {
            	    	enc_data = list_entry(ptr, km_encoder_desc_data_t, listnode);
            	    	if (enc_data->encoder_hook && enc_data->encoder_hook->km_event) {
            	    	    enc_data->encoder_hook->km_event(enc_data->encoder_hook, obj->data_link, &event);
            	    	}
            	    }
            	    MUTEX_UNLOCK(&km_encoder_hook_mtx);
                }
		md->m[i-1] = md->m[i]/m_fac_real;
	    }
	}
    }

    md->xmax[md->accel_steps-1]	= 254 * MOUSE_RASTER;
    md->ymax[md->accel_steps-1]	= (int)((double)md->xmax[md->accel_steps-1]
	* (double)(md->m[md->accel_steps-1]));

    /* warn if acceleration segments are to small */
    if (!ret) {
	for (i=0; i<md->accel_steps-1; i++) {
	    if (md->xmax[i+1] - md->xmax[i] < MOUSE_SEGMENT_MIN_SIZE) {
		_KD("Segments small -> warn\n");
		ret = ERIC_MOUSE_UNIVERSE_WARN_GOOD;
		break;
	    }
	}
    }

    if(0 != move_mouse_corner(obj, 0))
	goto bailout;

    for(i=0;i<md->accel_steps;i++) {
	snprintf(debug_msg, sizeof(debug_msg),
		 "%d:m=%02.02f,x0=%2d,xmax=%8d,ymax=%8d",
		 i, md->m[i], md->x0[i], md->xmax[i], md->ymax[i]);
	_KD("%s\n", debug_msg);
        if ((debug)) {
            event.type = KM_EVENT_DEBUG_MSG;
            event.event.debug_msg.msg = debug_msg;
            MUTEX_LOCK(&km_encoder_hook_mtx);
            list_for_each(ptr, &km_encoder_hook_list) {
            	enc_data = list_entry(ptr, km_encoder_desc_data_t, listnode);
            	if (enc_data->encoder_hook && enc_data->encoder_hook->km_event) {
            	    enc_data->encoder_hook->km_event(enc_data->encoder_hook, obj->data_link, &event);
            	}
            }
            MUTEX_UNLOCK(&km_encoder_hook_mtx);
        }
    }
    
    md->cur_x	=  MOUSE_RASTER/2;
    md->cur_y	= -MOUSE_RASTER/2;

    md->state = ERIC_MOUSE_IS_VALID;
    return ret;
    
 bailout:
#if defined(PP_FEAT_KITTY_CAT)
    free(shape_fb_rgb);
#endif /* PP_FEAT_KITTY_CAT */
    return ERIC_MOUSE_UNIVERSE_FAILED;
}

int
move_mouse_diff(driver_t* obj, long diff_x, long diff_y, const unsigned char buttonmask,
		int* moved)
{
    long virt_mov_x, virt_mov_y, real_moved;
    long real_mov_x, real_mov_y, virt_moved_x, virt_moved_y;
    int count = 0, ret = 0;
    data_imouse_t* mouse_data = (data_imouse_t*) obj->data_conv_mouse.data;
    t_mouse_data* md = &mouse_data->current_kvm_port->mouse_data;
    long diff_orig_x = diff_x, diff_orig_y = diff_y;

    *moved = 0;

    while ((diff_x || diff_y) && (++count < MOUSE_MAX_DIFF_COUNT)) {
        /* X direction */
        virt_mov_x	= mouse_virt_out2in(obj, diff_x);

        /* Y direction */
        virt_mov_y	= mouse_virt_out2in(obj, diff_y);

        real_mov_x	= md->fm_corr.fm_out2in(virt_mov_x / MOUSE_RASTER);
        real_mov_y	= md->fm_corr.fm_out2in(virt_mov_y / MOUSE_RASTER);

        /* avoid sending 0xAA to host, because it may lead to a mouse reset */
	if	(real_mov_x == -86) real_mov_x = -80;
	if	(real_mov_y == -86) real_mov_y = -80;

	real_moved = md->fm_corr.fm_in2out(real_mov_x);
	virt_moved_x = mouse_virt_in2out(obj, real_moved * MOUSE_RASTER);

	real_moved = md->fm_corr.fm_in2out(real_mov_y);
	virt_moved_y = mouse_virt_in2out(obj, real_moved * MOUSE_RASTER);

	/* return, we cannot reach the target exactly from our position */
	if (real_mov_x == 0 && real_mov_y == 0) {
	    break;
	}

	if (md->accel_steps == 1) {
	    if (0 != (ret = obj->data_unit_mouse.send_mouse_data(obj, real_mov_x, real_mov_y * -1,
								  0, buttonmask)))
		break;
	} else {
	    if (0 != (ret = obj->data_unit_mouse.send_mouse_data(obj, real_mov_x, 0,
								  0, buttonmask)))
		break;
	    
	    if (0 != (ret = obj->data_unit_mouse.send_mouse_data(obj, 0, real_mov_y * -1,
								  0, buttonmask)))
		break;
	}
	
	md->cur_x = md->cur_x + virt_moved_x;
	md->cur_y = md->cur_y + virt_moved_y;
	
	diff_x = diff_x - virt_moved_x;
	diff_y = diff_y - virt_moved_y;

	/* we reached our original diff value again, so
	   pointer is oscillating, set to 0 to stop movement */
	if (diff_x == diff_orig_x) diff_x = 0;
	if (diff_y == diff_orig_y) diff_y = 0;
	
	*moved = 1;
    }
    return ret;
}

/* obj function moves the mouse pointer using our GUT and
   the virtual mouse universe */
int
move_mouse(driver_t *obj,
	   int x, int y, int /* z */,
	   const unsigned char buttonmask)
{
    int ret, moved;
    long target_diff_x, target_diff_y;
    data_imouse_t* mouse_data = (data_imouse_t*) obj->data_conv_mouse.data;
    t_mouse_data* md = &mouse_data->current_kvm_port->mouse_data;

#if defined(PP_FEAT_KITTY_CAT)
    cat_update_mouse_position(obj);
#endif /* PP_FEAT_KITTY_CAT */

    target_diff_x = (x * MOUSE_RASTER) - md->cur_x;
    target_diff_y = (y * MOUSE_RASTER) - md->cur_y;

    KD("move_mouse: (%ld,%ld)->(%ld,%ld)\n", md->cur_x, md->cur_y, (long)x*MOUSE_RASTER, (long)y*MOUSE_RASTER);
    
#if defined(PP_FEAT_KITTY_CAT)
    cat_enqueue_ptr_move(obj, target_diff_x / MOUSE_RASTER, 
                               target_diff_y / MOUSE_RASTER);
#endif /* PP_FEAT_KITTY_CAT */

    if (0 != (ret = move_mouse_diff(obj, target_diff_x, target_diff_y,
				    buttonmask, &moved)))
	goto bailout;

    /* now send buttons
       this is necessary when no movement is made but the
       buttons must be transmitted anyway */

    if (!moved)
	ret = obj->data_unit_mouse.send_mouse_data(obj, 0, 0, 0, buttonmask);
    
 bailout:
    return ret;
}


/* obj function moves the mouse directly (relative),
   we dont care about its current and future position.
   It is used e.g. to determine movements */
int
move_mouse_direct(driver_t* obj,
		  const int dx, const int dy, const int dz, 
		  const unsigned char buttonmask, 
		  const unsigned char scale)
{
    int diff_x, diff_y, sign_x, sign_y;
    int x_single_mov, y_single_mov;
    int ret = 0, moved = 0;
    data_imouse_t* mouse_data = (data_imouse_t*) obj->data_conv_mouse.data;
    t_mouse_data* md = &mouse_data->current_kvm_port->mouse_data;

    KD("mm_direct: x: %d,y %d,z %d,b %d,s %f\n", dx,dy,dz,buttonmask,md->scaling);
    
    diff_x = abs(dx);
    diff_y = abs(dy);
    
    sign_x = sign(dx);
    sign_y = sign(dy);

    if (scale) {
	diff_x = (int) diff_x * (int)md->scaling;
	diff_y = (int) diff_y * (int)md->scaling;
    }

    while (diff_x > 0 || diff_y > 0) {
	x_single_mov = diff_x;
	y_single_mov = diff_y;

	/* avoid sending 0xAA to host
	   because it may lead to a mouse reset
	thomas: we don't need to check obj anymore, as we
	shrink the max movement distance to 63 */
	
	if (x_single_mov > 63)	      x_single_mov = 63;
	if (y_single_mov > 63)	      y_single_mov = 63;

	KD("xsm:%d, sx:%d, ysm:%d, sy:%d\n", x_single_mov, sign_x, y_single_mov, sign_y);
	
	if(0 != (obj->data_unit_mouse.send_mouse_data(obj,
						  x_single_mov * sign_x,
						  y_single_mov * sign_y * (-1),
						  dz,
						  buttonmask)))
	    goto bailout;
	
	diff_x -= x_single_mov;
	diff_y -= y_single_mov;

	moved = 1;
    }

    md->cur_x += dx * MOUSE_RASTER;
    md->cur_y += dy * MOUSE_RASTER;

    if (md->cur_x < 0) md->cur_x = 0;
    if (md->cur_y < 0) md->cur_y = 0;

    if (!moved) {
	if(0 != (obj->data_unit_mouse.send_mouse_data(obj, 0, 0, dz,
						       buttonmask)))
	    goto bailout;
    }
    
 bailout:
    return ret;
}

/* obj function moves the mouse into the left top 
   corner, used for syncing */
int move_mouse_corner(driver_t* obj, 
		      const unsigned char buttonmask)
{
    int ret;
    
    ret = move_mouse_direct(obj, -2100, -2100, 0, buttonmask, 0);

    return ret;
}

/* F:	returns the number of ticks the mouse moves
	for the given input count */
static long
mouse_virt_in2out(driver_t* obj, long x)
{
    data_imouse_t* mouse_data = (data_imouse_t*) obj->data_conv_mouse.data;
    t_mouse_data* md = &mouse_data->current_kvm_port->mouse_data;

    double m;
    long x0, y;
    int i;
    
    m = md->m[md->accel_steps-1];
    x0= md->x0[md->accel_steps-1];
    for (i = 0; i < md->accel_steps;i++) {
	if (abs(x) < md->xmax[i]) {
	    m	= md->m[i];
	    x0	= md->x0[i];
	    break;
	}
    }

    y = (long int)(x*m + x0);
    
    return y;
}

/* F':	returns the necessary input value to move the mouse
	the requested number of ticks */
static long
mouse_virt_out2in(driver_t* obj, long y)
{
    data_imouse_t* mouse_data = (data_imouse_t*) obj->data_conv_mouse.data;
    t_mouse_data* md = &mouse_data->current_kvm_port->mouse_data;

    double m;
    long x0, x;
    int i;

    m = md->m[md->accel_steps - 1];
    x0= md->x0[md->accel_steps - 1];
    for (i = 0; i < md->accel_steps - 1; i++) {
	if (abs(y) < md->ymax[i]) {
	    m	= md->m[i];
	    x0	= md->x0[i];
	    break;
	}
    }

    x = (long int)((double)(y - x0) / m);

    if (abs(x) > (md->xmax[i] - MOUSE_RASTER)) {
	x = (md->xmax[i] - MOUSE_RASTER) * sign(x);
    }

    /* avoid sending too big values to host */
    if (x > 64*MOUSE_RASTER)  x = 64*MOUSE_RASTER;
    else if (x < -64*MOUSE_RASTER) x = -64*MOUSE_RASTER;

    return x;
}

/* moves the mouse a little forth and back and checks
   whether at the end the mouse can be found at the expected
   position. The quadratic error will be returned
   or -1 in case the mouse is not found at all */
static int
check_mouse_error(driver_t* obj) {
    t_point mouse_cursor_point;
    t_point mouse_cursor_expect;
    int dx, dy, sqerr;
    
    move_mouse_corner(obj, 0);
    move_mouse(obj, 2, 0, 0, 0);
    move_mouse(obj, 5, 0, 0, 0);
    move_mouse(obj, 7, 0, 0, 0);
    move_mouse(obj, 12, 0, 0, 0);
    move_mouse(obj, 13, 0, 0, 0);
    move_mouse(obj, 17, 0, 0, 0);
    move_mouse(obj, 20, 0, 0, 0);
    move_mouse(obj, 30, 0, 0, 0);
    move_mouse(obj, 32, 0, 0, 0);
    move_mouse(obj, 33, 0, 0, 0);
    move_mouse(obj, 38, 0, 0, 0);
    move_mouse(obj, 41, 0, 0, 0);
    usleep(BUILD_SLEEP);
    mouse_cursor_expect.pos_x = 41;
    mouse_cursor_expect.pos_y = 0;
    
    if (0 > find_cursor_for_table(obj, &mouse_cursor_expect,
				  &mouse_cursor_point)) {
	sqerr = -1;
	KD("check_mouse_error: -1");
    } else {
	dx = mouse_cursor_expect.pos_x - mouse_cursor_point.pos_x;
	dy = mouse_cursor_expect.pos_y - mouse_cursor_point.pos_y;
	sqerr = dx * dx + dy * dy;
	KD("check_mouse_error: ex=%d mx=%d ey=%d my=%d\n",
	   mouse_cursor_expect.pos_x, mouse_cursor_point.pos_x,
	   mouse_cursor_expect.pos_y, mouse_cursor_point.pos_y);
    }
    return sqerr;
}


/* misc helper functions */
#ifdef PP_FEAT_KITTY_CAT
static float
fabs(float x)
{
    if (x >= 0) return x;
    else return x*(-1);
}
#endif

static long
sign(long x)
{
    if (x >= 0) return 1;
    else return -1;
}

static long non_fm_corr(long i) {
    return i;
}

/*
 * resolution param is 3 (8 ticks / mm)
 */
static long gud3_fm_corr_in2out(long i) {
    long o;
    // make sure we never call with an overflow condition
    assert(i <= 127);
    assert(i >= -128);
    if (i == 0) {
	o = 0;
    } else if (i > 0) {
	o = (i * 2) - 1;
    } else {
	o = (i * 2) + 1;
    }
    return o;
}

static long gud3_fm_corr_out2in(long o) {
    long i;
    if (o == 0) {
	i = 0;
    } else if (o > 0) {
	i = (o + 1) / 2;
    } else {
	i = (o - 1) / 2;
    }
    // avoid overflow, depends on original in2out function
    if (i > 127) i = 127;
    if (i < -128) i = -128;
    return i;
}

/*
 * resolution param is 2 (4 ticks / mm)
 * there seems to be some overflow condition in the G&D switch
 * so we need these 2 functions too
 */
static long  gud2_fm_corr_in2out(long i) {
    return i;
}

static long  gud2_fm_corr_out2in(long o) {
    long i = o;
    // avoid overflow, depends on original in2out function
    if (i > 127) i = 127;
    if (i < -128) i = -128;
    return i;
}

/*
 * resolution param is 1 (2 ticks / mm)
 */
static long  gud1_fm_corr_in2out(long i) {
    long o;
    // make sure we never call with an overflow condition
    assert(i <= 127);
    assert(i >= -128);
    if (i >= 0) {
	o = i / 2;
    } else {
	o = (i - 1) / 2;
    }
    return o;
}

static long  gud1_fm_corr_out2in(long o) {
    long i;
    i = 2 * o;
    // avoid overflow, depends on original in2out function
    if (i > 127) i = 127;
    if (i < -128) i = -128;
    return i;
}

/*
 * resolution param is 0 (1 ticks / mm)
 */
static long  gud0_fm_corr_in2out(long i) {
    long o;
    // make sure we never call with an overflow condition
    assert(i <= 127);
    assert(i >= -128);
    if (i >= 0) {
	o = i / 4;
    } else {
	o = (i - 3) / 4;
    }
    return o;
}

static long  gud0_fm_corr_out2in(long o) {
    long i;
    i = 4 * o;
    // avoid overflow, depends on original in2out function
    if (i > 127) i = 127;
    if (i < -128) i = -128;
    return i;
}
