/*****************************************************************************
* iipptr_bitmap.c                                                            *
*                                                                            *
* Copyright 2002 Peppercon AG                                                *
* Author: Thomas Rendelmann                                                  *
*                                                                            *
* Part of libpp_km                                                           *
* mouse driver iipps2 for Intelligent Mouse cursor Tracking                  *
*                                                                            *
* Contains functions for importing and exporting Windows Bitmap files (.BMP) *
* this module is not very flexible but this is not necessary here            *
*                                                                            *
* this is only needed for testing purposes !!                                *
*****************************************************************************/

#include "iipptr_bitmap.h"


#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include <pp/base.h>


/* read bitmap data (bytes) */
static unsigned char bitmap_bytes[2000 * 1500 * 3];



/* obj function reads a bitmap file into a t_bitmap structure
   the bitmap read function currently only works with bitmaps having
   a color depth of 24 bit */

int
read_bitmap(t_bitmap* bitmap, const char* filename) {
	int fd_bitmap;
	int npad;
	int nindex;
	
	unsigned int i, j;
	long r;
	
	t_bitmap_info_header r_info;
	t_bitmap_file_header r_file_info;
		
	/* open bitmap file */
	if ((fd_bitmap = open(filename, O_RDONLY)) < 0) {
	    pp_log("Error: can't open screen file \n");
	    return -1;
	}

	/* read information from bitmap */
	if (read(fd_bitmap, (void *)&r_file_info, sizeof(t_bitmap_file_header)) != sizeof(t_bitmap_file_header)) {
		pp_log("Error: can't read file header from screen file \n");
		close(fd_bitmap);
		return -1;
	}
	
	if (read(fd_bitmap, (void *)&r_info, sizeof(t_bitmap_info_header)) != sizeof(t_bitmap_info_header)) {
		pp_log("Error: can't read info header from screen file \n");
		close(fd_bitmap);
		return -1;
	}

	/* only 24 bit bitmaps ! */
	if(le16_to_cpu(r_info.bi_bitcount) != 24) {
		pp_log("Dieses Programm funktioniert nur mit Bitmaps mit 24 Bit Farbtiefe !\n");
		close(fd_bitmap);
		return -1;
	}

	/* saving bitmap info */
	bitmap->width = le32_to_cpu(r_info.bi_width);
	bitmap->height = le32_to_cpu(r_info.bi_height);
	

	/* read bitmap data */
	if(bitmap->rgb != NULL) {
		free(bitmap->rgb);
		bitmap->rgb = NULL;
	}

	bitmap->rgb = (t_rgb *)malloc(bitmap->height * bitmap->width * sizeof(t_rgb));
        
	npad = (le32_to_cpu(r_info.bi_size_image) /
                (le32_to_cpu(r_info.bi_height) * 3)) - bitmap->width;
        
	if((r = read(fd_bitmap, (void *)bitmap_bytes, (long)le32_to_cpu(r_info.bi_size_image))) != (long)le32_to_cpu(r_info.bi_size_image)) {
		pp_log("Error: can't read picture\n");
		pp_log("  read: %li   should: %i\n", r, le32_to_cpu(r_info.bi_size_image));
		close(fd_bitmap);
		return -1;
	}
        
	nindex = 0;
	for (j = 0; j < bitmap->height; j++) {
		for (i = 0; i < bitmap->width; i++) {
			bitmap->rgb[bitmap->width * (bitmap->height - j - 1) + i].b = *(bitmap_bytes + nindex);
			bitmap->rgb[bitmap->width * (bitmap->height - j - 1) + i].g = *(bitmap_bytes + nindex + 1);
			bitmap->rgb[bitmap->width * (bitmap->height - j - 1) + i].r = *(bitmap_bytes + nindex + 2);
			nindex += 3;
		}
		nindex += npad;
	}

	/* close bitmap file */
	close(fd_bitmap);
	return 0;
}



/* obj function writes a t_bitmap structure into a bitmap file
   the bitmap write function only works when the bitmap has a width
   dividable by 4 because otherwise additional bytes must be inserted;
   but that is not the task here */
   
int
write_bitmap(t_bitmap* bitmap, const char* filename) {
	int fd_bitmap;
	int nindex;

	unsigned int i, j;

	long r;
	
	long bi_size_image;
	
	t_bitmap_info_header wr_info;
	t_bitmap_file_header wr_file_info;
        
	bi_size_image = bitmap->width ? 
                        ((bitmap->width - 1) / 4 + 1) * 4 * bitmap->height * 3 :
                        0;

        /* preparing bitmap headers */
	wr_file_info.bf_type = cpu_to_le16(19778);
	wr_file_info.bf_off_bits = cpu_to_le32(sizeof(t_bitmap_file_header) + sizeof(t_bitmap_info_header));
	wr_file_info.bf_size = cpu_to_le32(bi_size_image + wr_file_info.bf_off_bits);
	wr_file_info.bf_reserved1 = cpu_to_le16(0);
	wr_file_info.bf_reserved2 = cpu_to_le16(0);
		
	wr_info.bi_size = cpu_to_le32(sizeof(t_bitmap_info_header));
	wr_info.bi_width = cpu_to_le32(bitmap->width);
	wr_info.bi_height = cpu_to_le32(bitmap->height);
	wr_info.bi_planes = cpu_to_le16(1);
	wr_info.bi_bitcount = cpu_to_le16(24);
	wr_info.bi_compression = cpu_to_le32(0);
	wr_info.bi_size_image = cpu_to_le32(bi_size_image);
	wr_info.bi_x_per_meter = cpu_to_le32(3780);
	wr_info.bi_y_per_meter = cpu_to_le32(3780);
	wr_info.bi_clr_used = cpu_to_le32(0);
	wr_info.bi_clr_important = cpu_to_le32(0);

	/* open bitmap file */
	if ((fd_bitmap = open(filename, O_RDWR | O_CREAT | O_TRUNC, 0777)) < 0) {
		pp_log("Error: can't open output file \n");
		return -1;
	}

	/* write information to bitmap file */
	if (write(fd_bitmap, (void *)&wr_file_info, sizeof(t_bitmap_file_header)) != sizeof(t_bitmap_file_header)) {
		pp_log("Error: can't read file header from screen file \n");
		close(fd_bitmap);
		return -1;
	}
	
	if (write(fd_bitmap, (void *)&wr_info, sizeof(t_bitmap_info_header)) != sizeof(t_bitmap_info_header)) {
		pp_log("Error: can't read info header from screen file \n");
		close(fd_bitmap);
		return -1;
	}

	nindex = 0;
	for (j = 0; j < bitmap->height; j++) {
		for (i = 0; i < bitmap->width; i++) {
			*(bitmap_bytes + nindex) = (unsigned char)bitmap->rgb[bitmap->width * (bitmap->height - j - 1) + i].b;
			*(bitmap_bytes + nindex + 1) = (unsigned char)bitmap->rgb[bitmap->width * (bitmap->height - j - 1) + i].g;
			*(bitmap_bytes + nindex + 2) = (unsigned char)bitmap->rgb[bitmap->width * (bitmap->height - j - 1) + i].r;
			nindex += 3;
		}
                while (nindex % 4) {
                    /* fill each row to a multiply of four bytes */
                    bitmap_bytes[nindex++] = (unsigned char)0x0;
                }
	}

	if((r = write(fd_bitmap, (void *)bitmap_bytes, bi_size_image)) != bi_size_image) {
		pp_log("Error: can't write picture\n");
		pp_log("  written: %li   should: %li\n", r, bi_size_image);
		close(fd_bitmap);
		return -1;
	}

	close(fd_bitmap);

	return 0;
}



/* obj function prints info and RGB values of the bitmap
   on the console */

int
show_bitmap(t_bitmap * bitmap) {
	unsigned int i, j;
	
	/* show infos */
	pp_log("  Width: %i\n", bitmap->width);
	pp_log("  Height: %i\n", bitmap->height);
	
	/* show bitmap rgb data */
	pp_log("\nBitmap:\n");
	for (j = 0; j < bitmap->height; j++) {
		for (i = 0; i < bitmap->width; i++) {
			pp_log("%2x", bitmap->rgb[j * bitmap->width + i].b);
			pp_log("%2x", bitmap->rgb[j * bitmap->width + i].g);
			pp_log("%2x ", bitmap->rgb[j * bitmap->width + i].r);
		}
		pp_log("\n");
        } 
	
	return 0;
}

#if defined(PP_FEAT_CAT) || defined(PP_FEAT_KITTY_CAT)
/* returns bitmap within given borders */
int crop_bitmap(t_bitmap *cropped, t_bitmap *bitmap,
                u_int16_t left, u_int16_t right, 
                u_int16_t top, u_int16_t bottom) {
    u_int16_t width, height, row;
    
    assert(cropped);
    assert(bitmap);
    assert(bitmap != cropped);
    assert(bitmap->rgb);
    assert(left <= right);
    assert(right < bitmap->width);
    assert(top <= bottom);
    assert(bottom < bitmap->height);
    
    width = (right - left) + 1;
    height = (bottom - top) + 1;
    
    cropped->width = width;
    cropped->height = height;
    if(cropped->rgb != NULL) {
	free(cropped->rgb);
    }
    cropped->rgb = (t_rgb*)malloc(height * width * sizeof(t_rgb));
    
    for(row = 0; row < height; ++row) {
        memcpy(&cropped->rgb[row * /* cropped-> */ width], 
               &bitmap->rgb[(row + top) * bitmap->width + left],
               width * sizeof(t_rgb));
    }
               
    return 0;
}

/* returns bitmap cropped by frame of given color */
int crop_bitmap_by_color(t_bitmap *cropped,
                         u_int16_t *left_p, u_int16_t *right_p,
                         u_int16_t *top_p, u_int16_t *bottom_p,
                         t_bitmap *bitmap, t_rgb color) {
    u_int16_t left = -1, right = 0, top = 0, bottom = -1, row, col, offset;
    char dirty;
    
    assert(cropped);
    assert(bitmap);
    assert(bitmap != cropped);
    assert(bitmap->rgb);
    
    /* find first "dirty" row */
    for(row = 0, dirty = 0; !dirty && row < bitmap->height; ++row) {
        offset = row * bitmap->width;
        for(col = 0; !dirty && col < bitmap->width; ++col) {
            if(memcmp(&bitmap->rgb[offset + col], &color, sizeof(t_rgb))) {
                /* (row, col) pixel of bitmap is not of color color */
                dirty = 1;
                top = row;
            }
        }
    }
    
    /* find last "dirty" row */
    for(row = bitmap->height - 1, dirty = 0; !dirty && row > top; --row) {
        offset = row * bitmap->width;
        for(col = 0; !dirty && col < bitmap->width; ++col) {
            if(memcmp(&bitmap->rgb[offset + col], &color, sizeof(t_rgb))) {
                /* (row, col) pixel of bitmap is not of color color */
                dirty = 1;
                bottom = row;
            }
        }
    }
    
    /* find first "dirty" column */
    for(col = 0, dirty = 0; !dirty && col < bitmap->width; ++col) {
        for(row = top; !dirty && row <= bottom; ++row) {
            if(memcmp(&bitmap->rgb[row * bitmap->width + col], 
                      &color, sizeof(t_rgb))) {
                /* (row, col) pixel of bitmap is not of color color */
                dirty = 1;
                left = col;
            }
        }
    }
    
    /* find last "dirty" column */
    for(col = bitmap->width - 1, dirty = 0; !dirty && col > left; --col) {
        for(row = top; !dirty && row <= bottom; ++row) {
            if(memcmp(&bitmap->rgb[row * bitmap->width + col],
                      &color, sizeof(t_rgb))) {
                /* (row, col) pixel of bitmap is not of color color */
                dirty = 1;
                right = col;
            }
        }
    }
    
    if(left_p) *left_p = left;
    if(right_p) *right_p = right;
    if(top_p) *top_p = top;
    if(bottom_p) *bottom_p = bottom;

    return crop_bitmap(cropped, bitmap, left, right, top, bottom);
}

/* returns 1bit threshold bitmap
   mode 0 is all channels larger threshold, mode 1 is any */
int threshold_bitmap_by_color(t_bitmap *new, t_bitmap *bitmap, 
                              t_rgb threshold, int mode) {
    u_int16_t row, col, offset, ret = PP_ERR;
    
    assert(new);
    assert(bitmap);
    assert(bitmap != new);
    assert(bitmap->rgb);
    
    if(new->rgb) {
        free(new->rgb);
    }

    new->rgb = (t_rgb*)malloc(bitmap->height * bitmap->width * sizeof(t_rgb));
    new->height = bitmap->height;
    new->width = bitmap->width;
    
    if(mode == 1) {
        /* any channel has to be larger than threshold... */
        for(row = 0; row < bitmap->height; ++row) {
            offset = row * bitmap->width;
            for(col = 0; col < bitmap->width; ++col) {
                if(bitmap->rgb[offset + col].r >= threshold.r ||
                   bitmap->rgb[offset + col].g >= threshold.g ||
                   bitmap->rgb[offset + col].b >= threshold.b) {
                    new->rgb[offset + col].r = 255;
                    new->rgb[offset + col].g = 255;
                    new->rgb[offset + col].b = 255;
                    ret = PP_SUC;
                } else {
                    new->rgb[offset + col].r = 0;
                    new->rgb[offset + col].g = 0;
                    new->rgb[offset + col].b = 0;
                }
            }
        }
    } else if(mode == 0) {
        /* all channels have to be larger than threshold... */
        for(row = 0; row < bitmap->height; ++row) {
            offset = row * bitmap->width;
            for(col = 0; col < bitmap->width; ++col) {
                if(bitmap->rgb[offset + col].r < threshold.r ||
                   bitmap->rgb[offset + col].g < threshold.g ||
                   bitmap->rgb[offset + col].b < threshold.b) {
                    new->rgb[offset + col].r = 0;
                    new->rgb[offset + col].g = 0;
                    new->rgb[offset + col].b = 0;
                } else {
                    new->rgb[offset + col].r = 255;
                    new->rgb[offset + col].g = 255;
                    new->rgb[offset + col].b = 255;
                    ret = PP_SUC;
                }
            }
        }
    } else {
        /* unknown mode */
        abort();
    }
    
    return PP_SUC;
}
#endif /* PP_FEAT_CAT || PP_FEAT_KITTY_CAT */


