/*****************************************************************************
* iipptr_cursor.c                                                            *
*                                                                            *
* Copyright 2002 Peppercon AG                                                *
* Author: Thomas Rendelmann                                                  *
*                                                                            *
* Part of liberic_km                                                         *
* mouse driver iipps2 for Intelligent Mouse cursor Tracking                  *
*                                                                            *
* Contains some operations on the mouse cursor                               *
*****************************************************************************/

#include "iipptr_cursor.h"
#include "iipptr_bitmap.h"
#include "iipptr_help.h"
#include "iipptr_internal.h"

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <pthread.h>
#include <fcntl.h>

#include <pp/grab.h>
#include <pp/base.h>

// use flag in Makefile!
// #define noDEBUG_CURSORSHAPE_BMP

#define CURSOR_SNAP_SLEEP 50000
#define CURSOR_MASK_MIN	  1024

/* obj function tries to find the mouse cursor on the screen in a
   horizontal stripe of cursor size at the top of the screen,
   provided for building the acceleration table */

int
find_cursor_for_table(driver_t* obj, t_point *center, t_point *found)
{
	int diff_result;
	int search_diff = 200;
	int max_diff;
	fb_format_info_t fb_f_info;

	data_imouse_t* mouse_data = (data_imouse_t*) obj->data_conv_mouse.data;
	
	if(mouse_data->current_cursor_shape.rgb == NULL) {
		return -1;
	}
	
	if(pp_grab_get_format_info(obj->video_link, &fb_f_info) != 0) {
	    return -1;
	}
   
	max_diff = fb_f_info.g_w;
	
	while(((diff_result =
		difference(mouse_data->grab_client, &mouse_data->current_cursor_shape,
			   &mouse_data->current_cursor_mask,
			   center, search_diff,
			   mouse_data->current_cursor_shape.height, found)) == -1)
	      && (search_diff < max_diff)) {
	    search_diff *=2;
	}
	
	return diff_result;
}

/* -this is the most important function in this module
   it takes a screenshot of the current mouse cursor
   and removes unnecessary rows and columns
   -first the cursor is moved _away_ from the top left corner
    (32,32) and a screenshot is taken
   -after obj the cursor is moved _in_ the corner and another
    shot is taken
   -then a difference mask between the shots is created
    (as a base for the acc table recognition) */

int
get_mousecursor_shape(driver_t* obj)
{
    data_imouse_t* mouse_data = (data_imouse_t*) obj->data_conv_mouse.data;
    t_mouse_data* md = &mouse_data->current_kvm_port->mouse_data;
    assert(mouse_data->grab_client);

    md->cursor_state = ERIC_MOUSE_CURSOR_GETTING;

    KD("liberic_km: Getting mouse cursor shape\n");
    move_mouse_direct(obj, 2000, 2000, 0, 0, 0);	
    usleep(CURSOR_SNAP_SLEEP);

    if((screenshot(mouse_data->grab_client, &mouse_data->current_cursor_bg_shape,
		   0, 0, 32, 32)) != 0) {
	pp_log_err("Failure during screenshot of background\n");
	goto error;
    }

    move_mouse_corner(obj, 0);
    usleep(CURSOR_SNAP_SLEEP);
    
    if((screenshot(mouse_data->grab_client, &mouse_data->current_cursor_shape,
		   0, 0, 32, 32)) != 0) {
	pp_log("Failure during screenshot of mouse cursor\n");
	goto error;
    }

    /* create mask bitmap */
    mouse_data->current_cursor_mask.width  = mouse_data->current_cursor_shape.width;
    mouse_data->current_cursor_mask.height = mouse_data->current_cursor_shape.height;
    
    if (mouse_data->current_cursor_mask.rgb) {
	free(mouse_data->current_cursor_mask.rgb);
	mouse_data->current_cursor_mask.rgb = NULL;
    }
    mouse_data->current_cursor_mask.rgb    = (t_rgb *)malloc(mouse_data->current_cursor_mask.height *
							  mouse_data->current_cursor_mask.width  * sizeof(t_rgb));

    /* Diff both screenshots to get current cursor-mask */
    if (diff_mousecursor_shape(&mouse_data->current_cursor_shape,
			       &mouse_data->current_cursor_bg_shape,
			       &mouse_data->current_cursor_mask) != 0) {
	pp_log("liberic_km: Could not detect mouse cursor");
        goto error;
    }
    
#if defined(PP_FEAT_KITTY_CAT)
    {
        t_bitmap kc_tmp = {0, 0, NULL};
        u_int16_t left, right, top, bottom;
        
        if(PP_ERR == threshold_bitmap_by_color(&kc_tmp,
                                               &mouse_data->current_cursor_mask, 
                                      (t_rgb){CAT_CROP_CHANNEL_THRESHOLD, 
                                              CAT_CROP_CHANNEL_THRESHOLD, 
                                              CAT_CROP_CHANNEL_THRESHOLD}, 1)) {
            pp_log("could not threshold mask");
            free(kc_tmp.rgb);
            goto error;
        }
        if(PP_ERR == crop_bitmap_by_color(&mouse_data->current_cursor_mask,
                                          &left, &right, &top, &bottom, 
                                          &kc_tmp, (t_rgb){0, 0, 0})) {
            pp_log("could not crop mask");
            free(kc_tmp.rgb);
            goto error;
        }
        memcpy(kc_tmp.rgb, mouse_data->current_cursor_shape.rgb,
               kc_tmp.width * kc_tmp.height * sizeof(t_rgb));
        if(PP_ERR == crop_bitmap(&mouse_data->current_cursor_shape, &kc_tmp,
                                 left, right, top, bottom)) {
            pp_log("could not crop shape");
            free(kc_tmp.rgb);
            goto error;
        }
        free(kc_tmp.rgb);

#ifdef DEBUG_CURSORSHAPE_BMP
        write_bitmap(&mouse_data->current_cursor_shape,
                     "cropped_cursor_shape.bmp");
        write_bitmap(&mouse_data->current_cursor_mask,
                     "cropped_cursor_mask.bmp");
#endif
    }
#endif /* PP_FEAT_KITTY_CAT */

    md->cursor_state = ERIC_MOUSE_CURSOR_IS_VALID;
    
    return 0;
    
 error:
    md->cursor_state = ERIC_MOUSE_CURSOR_NOT_VALID;
    return -1;
}

/* obj takes two screenshot-bitmaps (one for the background and one hopefully
   including the cursor) and creates a difference mask */
int
diff_mousecursor_shape(t_bitmap *cursor, t_bitmap *background, t_bitmap *mask)
{
    unsigned long offset;
    unsigned long mask_val, mask_sum=0;
    unsigned int  row, col;
#if 0
    t_hsv hsv_bg, hsv_cursor;
    unsigned char hdiff, sdiff, vdiff;
#endif
  
    for (row=0; row < cursor->height; row++) {
	for (col=0; col < cursor->width; col++) {
	    offset = row * cursor->width + col;

#if 0
	    /* convert to hsv model */
	    rgb_to_hsv(&cursor->rgb[offset], &hsv_cursor);
	    rgb_to_hsv(&background->rgb[offset], &hsv_bg);
	    hdiff = abs(hsv_cursor.h - hsv_bg.h);
	    sdiff = abs(hsv_cursor.s - hsv_bg.s);
	    vdiff = abs(hsv_cursor.v - hsv_bg.v);
#endif

	    mask_val = (abs(cursor->rgb[offset].r - background->rgb[offset].r) +
			abs(cursor->rgb[offset].g - background->rgb[offset].g) +
			abs(cursor->rgb[offset].b - background->rgb[offset].b))/3;
	    mask_sum+=mask_val;
	    
	    mask->rgb[offset].r = mask->rgb[offset].g = mask->rgb[offset].b = mask_val;
	}
    }

#ifdef DEBUG_CURSORSHAPE_BMP
    write_bitmap(cursor, "cursor_shape.bmp");
    write_bitmap(background, "cursor_bg.bmp");
    write_bitmap(mask, "cursor_mask.bmp");
#endif

    
    if (mask_sum < CURSOR_MASK_MIN) {
	return -1;
    }
    
    return 0;
}
