/*****************************************************************************
 * iipptr_help.c                                                             *
 *                                                                           *
 * Copyright 2002 Peppercon AG                                               *
 * Author: Thomas Rendelmann                                                 *
 *                                                                           *
 * Part of liberic_km                                                        *
 * mouse driver iipps2 for Intelligent Mouse cursor Tracking                 *
 *                                                                           *
 * Contains three functions needed in the other modules                      *
 * one finds a bitmap (usually the mouse cursor) int the current framebuffer,*
 * another one converts pixels to rgb and a third one takes a screenshot     *
 * of a certain region inbetween the framebuffer                             *
 ****************************************************************************/

#include "iipptr_help.h"
#include "iipptr_bitmap.h"

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <pthread.h>
#include <fcntl.h>

#include <pp/grab.h>
#include <pp/base.h>


/* the MSE algo, called by difference */
int mse_compare(t_bitmap *screen, t_bitmap *fit, int fit_x, int fit_y, 
                t_bitmap *mask, int search_left, int search_top,
                int search_width, int search_height, t_point *found) {
    int row1, col1 = 0, row2, col2;
    int screen_offset;
    unsigned long *mse_tab = NULL;
    unsigned long  diff_sum;
    unsigned int   diff_count;
    unsigned int   mask_val;
    unsigned long  diff = 0;
    unsigned long  mse = 0, mse_min = 0, mse_mean = 0;
    unsigned long  mse_var = 0, mse_diff = 0;
    unsigned int mse_count = 0;
    char  first_one = 1;
    int i;

    mse_tab = (unsigned long*) malloc(search_width * search_height * sizeof(unsigned long));
    if (!mse_tab) {	
	return -1;
    }
    memset(mse_tab, 0, search_width * search_height * sizeof(unsigned long));
		     
    /* for every pixel in the area ... */
    for(row1 = 0; row1 < search_height; row1++) {

	/* we always compare full sized shapes,
	   otherwise the mse method wouldn't work */
	if(row1 > search_height - fit_y)
	    break;		
	
	for(col1 = 0; col1 < search_width; col1++) {
 
	    if(col1 > search_width - fit_x)
		break;

	    diff_count = 0; diff_sum = 0;
			
	    /* calculate differences */
	    for(row2 = 0; row2 < fit_y; row2++) {
		for(col2 = 0; col2 < fit_x; col2++) {
		    
		    mask_val =	mask->rgb[row2 * fit_x + col2].r;
  
		    if (mask_val != 0) {

			screen_offset = (row1 + row2) * search_width + (col1 + col2);

			diff = (abs(screen->rgb[screen_offset].r - fit->rgb[row2 * fit_x + col2].r) +
				abs(screen->rgb[screen_offset].g - fit->rgb[row2 * fit_x + col2].g) +
				abs(screen->rgb[screen_offset].b - fit->rgb[row2 * fit_x + col2].b)) / 3;

			diff_sum += (diff * mask_val);
			
			diff_count++;
		    }
		}
	    }
	    
	    mse	      = diff_sum / diff_count;
	    mse_mean += mse;
	    mse_tab[mse_count++] = mse;
	    
	    /* printf("pos = %d,%d -> mse = %ld\n", search_left+col1, search_top+row1, mse); */
	    
	    if (first_one) {
		mse_min = mse;
		found->pos_x = search_left+col1;
		found->pos_y = search_top+ row1;
		first_one = 0;
	    }

	    if (mse < mse_min) {
		found->pos_x = search_left+col1;
		found->pos_y = search_top+ row1;
		mse_min = mse;
	    }
	}
    }

    /* calculate mean and deviation for all positions */
    mse_mean = mse_mean / mse_count;
    
    for (i=0; i< search_width*search_height; i++) {
	if (mse_tab[i] != 0) {
	    mse_var += abs(mse_tab[i] - mse_mean);
	}
    }
    mse_var = mse_var / mse_count;
        
    /* printf("pos found=%d,%d with %d values, mse=%ld, mse_mean=%ld, mse_var=%ld\n",
       found->pos_x, found->pos_y, mse_count, mse_min, mse_mean, mse_var); */
   
    /* FIXME, not the correct way to do obj */
    mse_diff = abs(mse_min - mse_mean);
    if (mse_diff <= mse_var * 4 || !mse_var) {
	if (mse_tab) free(mse_tab);
	return -1;
    }
    
    if (mse_tab) free(mse_tab);

    return 0;
}

/* obj function tries to find one bitmap in the current framebuffer
   using difference formation */

int
difference(pp_grab_client_t *grab_client, t_bitmap *fit, t_bitmap *mask,
	   t_point *center, int search_diff_x, int search_diff_y, t_point *found)
{
    int src_x, src_y, fit_x, fit_y;
    int search_left, search_right, search_top, search_bottom;
    int search_width, search_height;

    t_bitmap screen = {0, 0, NULL};
    fb_format_info_t fb_f_info;
		
    if(fit->rgb == NULL) {
	return -1;
    }

    if(pp_grab_get_format_info(grab_client->video_link, &fb_f_info) != 0) {
	return -1;
    }

    src_x = fb_f_info.g_w;
    src_y = fb_f_info.g_h;

    /* is the pattern to be found bigger than the whole picture ? */	
    if(src_x < (int)fit->width) {
	return -1;
    }
    if(src_y < (int)fit->height) {
	return -1;
    }

    fit_x = fit->width;
    fit_y = fit->height;

    /* we compare at least one block */
    if (search_diff_x * 2 < fit_x) search_diff_x  = fit_x / 2;
    if (search_diff_y * 2 < fit_y) search_diff_y  = fit_y / 2;

    /* calc raw margins */
    search_left	  = (center->pos_x - search_diff_x);
    search_top	  = (center->pos_y - search_diff_y);
    search_right  = (center->pos_x + search_diff_x);
    search_bottom = (center->pos_y + search_diff_y);

    /* make sure we are in visible area */
    if (search_left < 0)	search_left	= 0;
    if (search_top < 0)		search_top	= 0;
    if (search_right  > src_x)	search_right	= src_x;
    if (search_bottom > src_y)	search_bottom	= src_y;

    search_width  = search_right-search_left;
    search_height = search_bottom-search_top;

    /* normalize to multiple of shape size we are looking for */
    search_width  = (search_width / fit_x) * fit_x;
    search_height = (search_height/ fit_y) * fit_y;

    /* printf("Search area %d,%d,%d,%d -> width/height = %d,%d\n",
       search_left, search_top, search_right, search_bottom, search_width,
       search_height); */

    if (screenshot(grab_client, &screen, search_left, search_top, search_width,
		   search_height)) {
	return -1;
    }

    mse_compare(&screen, fit, fit_x, fit_y, mask,
                search_left, search_top, search_width, search_height,
                found);
    
#ifdef DEBUG_CURSORSHAPE_BMP
    {
        static int i = 0;
        char fn[100];
        snprintf(fn, sizeof(fn), "sync_%03d.bmp", ++i);
        write_bitmap(&screen, fn);
    }
#endif /* DEBUG_CURSORSHAPE_BMP */
    
    if (screen.rgb) {
	free(screen.rgb);
	screen.rgb = NULL;
    }

    return 0;
}

/* obj function convertes a whole block of pixels in the current framebuffer
   into the t_rgb structure */
int
fb_to_rgb_area(pp_grab_client_t *grab_client, t_bitmap *bitmap,
	       u_short left, u_short top, u_short width, u_short height)
{
    int val, ret = 0;
    u_int16_t * i_16_ptr;        
    int offset_bmp=0;
    fb_color_info_t fb_c_info;
    u_short x, y, w, h;

    pp_grab_get_color_info(grab_client->video_link,&fb_c_info);
    assert(fb_c_info.bpp == 16);

    x = (left / PP_FB_TILE_WIDTH) * PP_FB_TILE_WIDTH;
    y = (top / PP_FB_TILE_HEIGHT) * PP_FB_TILE_HEIGHT;
    w = ((width + left - x + PP_FB_TILE_WIDTH - 1) / PP_FB_TILE_WIDTH) *
        PP_FB_TILE_WIDTH;
    h = ((height + top - y + PP_FB_TILE_HEIGHT - 1) / PP_FB_TILE_HEIGHT) *
        PP_FB_TILE_HEIGHT;
	    
    if (pp_grab_request_rect(grab_client, GRAB_REQ_FLAGS_IGNORE_DIFFMAP, x, y, w, h) != 0) {
	ret = -1;
	goto bail;
    }

    // add returned offset to the fb start -> aligned rect start
    i_16_ptr = (u_int16_t *)(grab_client->buf + 
                             grab_client->rect_offset * sizeof(u_int16_t));
    
    {
	int bol_skip  = left % PP_FB_TILE_WIDTH;
	int eol_skip  = ((bol_skip + width) % PP_FB_TILE_WIDTH) ?
                        PP_FB_TILE_WIDTH - ((bol_skip + width) %
                                            PP_FB_TILE_WIDTH) : 0;
	int line_skip = -(((bol_skip + width + PP_FB_TILE_WIDTH - 1) / PP_FB_TILE_WIDTH - 1) * PP_FB_TILE_WIDTH * PP_FB_TILE_HEIGHT);
	int row_skip  = (grab_client->rect_pitch - PP_FB_TILE_WIDTH) * PP_FB_TILE_HEIGHT;
	int w_single, line=0;
        int y_offset = top % PP_FB_TILE_HEIGHT;
	
	// goto start pixel, still in the same first hextile
	i_16_ptr += y_offset * PP_FB_TILE_WIDTH + (left % PP_FB_TILE_WIDTH);
	
	while (line++ < height) {
	    w_single = 0;
	    
	    while (w_single++ < width) {
		val = *i_16_ptr++;
		bitmap->rgb[offset_bmp].r = ((val >> (fb_c_info.blue_bits +
                                                      fb_c_info.green_bits)) 
                                                  << (8 - fb_c_info.red_bits))
                                            & 0xFF;
		bitmap->rgb[offset_bmp].g = ((val >> fb_c_info.blue_bits)
                                                  << (8 - fb_c_info.green_bits))
                                            & 0xFF;
		bitmap->rgb[offset_bmp].b = (val << (8 - fb_c_info.blue_bits))
                                            & 0xFF;
		offset_bmp++;

		if ((w_single + bol_skip) % PP_FB_TILE_WIDTH == 0 &&
                    w_single < width) {
		    i_16_ptr += (PP_FB_TILE_WIDTH * PP_FB_TILE_HEIGHT) -
                                PP_FB_TILE_WIDTH;
                }
	    }
	    
	    i_16_ptr += eol_skip;  // skip partial rect at end of line
	    i_16_ptr += line_skip; // start next pixel line
	    
	    if ((line + y_offset) % PP_FB_TILE_HEIGHT == 0) {
                i_16_ptr += row_skip;
            }
	    
	    i_16_ptr += bol_skip;  // skip partial rect at beginning of line
	}
    }

 bail:
    pp_grab_release_buffer(grab_client);
    return ret;
}

/* obj function takes a screenshot and returns it as a bitmap */
int
screenshot(pp_grab_client_t *grab_client, t_bitmap *screen,
	   u_short left, u_short top, u_short width, u_short height)
{
    int ret = 0;
    fb_format_info_t fb_f_info;
#if defined(PRODUCT_KX2)
    /* workaround, because sending out mouse moves with
     * cim lib/driver is asynchronously and so we have to
     * wait a little bit until the cursor is actually moved
     *
     * a synchronous mode will be implemented later to remove this sleep
     */
    usleep(1000*800);
#endif
    if (pp_grab_get_format_info(grab_client->video_link, &fb_f_info) != 0) {
	ret = -1;
	goto bail;
    }
    
    /* clipping */
    if((fb_f_info.g_w - left) < width) {
	width = fb_f_info.g_w - left;
    }
    
    if((fb_f_info.g_h - top) < height) {
	height = fb_f_info.g_h - top;
    }
    screen->width  = width;
    screen->height = height;

    if(screen->rgb != NULL) {
	free(screen->rgb);
	screen->rgb = NULL;
    }
    screen->rgb = (t_rgb *)malloc(height * width * sizeof(t_rgb));

    ret = fb_to_rgb_area(grab_client, screen, left, top, width, height);
    
 bail:
    return ret;
}
