/**
 * \file km.h
 *
 * API to talk to the various types of keyboard and mouse emulations.
 *
 * (c) 2004 Peppercon AG
 */


#ifndef _PP_KM_H
#define _PP_KM_H

#ifdef __cplusplus
extern "C" {
#endif

#include <sys/types.h>
#include <kernel_usb.h>

/* NOTE: definition of input_t moved to kernel_usb.h */

/* used in libpp_rfb */
#define PP_KM_PTR_MOVE_ABSOLUTE 0
#define PP_KM_PTR_MOVE_RELATIVE 1

/* used in frontend e.g. liberic_webs */
#define PP_KM_MOUSE_SYNC_NORM	        0x00    // resync fast or hard, whatever is necessary
#define PP_KM_MOUSE_SYNC_HARD           0x01    // complete resync, (re)read mouse parameters
#define PP_KM_MOUSE_SYNC_FAST           0x02    // fast resync, keep mouse parameters
#define PP_KM_MOUSE_SYNC_ACCEL_STATE    0x03    // internal use only

/* mouse buttons of the internal protocol */
#define BUTTON_LEFT_MASK   0x1
#define BUTTON_MIDDLE_MASK 0x2
#define BUTTON_RIGHT_MASK  0x4

/* keyboard LEDs */
#define PP_KM_KEYBOARD_LED_NUM  0x01
#define PP_KM_KEYBOARD_LED_CAPS 0x02
#define PP_KM_KEYBOARD_LED_SCRL 0x04



/**
 * Mouse sync event types
 */
typedef enum {
    KM_MOUSE_SYNC_IN_PROGRESS,    // mouse sync in progress
    KM_MOUSE_SYNC_SUCCESS,        // mouse sync successful
    KM_MOUSE_SYNC_FAILED,         // mouse sync failed
    KM_MOUSE_SYNC_AMBIGUOUS_GOOD, // mouse sync ambiguous but should work
    KM_MOUSE_SYNC_AMBIGUOUS_BAD,  // mouse sync ambiguous, most likely wont work
} km_mouse_sync_t;

/**
 * Mouse sync event structure
 */
typedef struct {
    km_mouse_sync_t type;
} km_event_mouse_sync_t;

/**
 * Debug message event structure
 */
typedef struct {
    char * msg;                 // the debug message, valid during call only, copy if needed
} km_event_debug_msg_t;

/**
 * Type of the km event
 */
typedef enum {
    KM_EVENT_MOUSE_SYNC,        // mouse sync event
    KM_EVENT_DEBUG_MSG          // debug message, can savely be ignored
} km_event_type_t;

/**
 * km event, sent to the encoder using a callback
 */
typedef struct {
    km_event_type_t type;

    union {
        km_event_mouse_sync_t mouse_sync;
        km_event_debug_msg_t  debug_msg;
    } event;
} km_event_t;


/**
 * Encoder callback structure. Defines callbacks that the encoder
 * must provide to connect to the kme. Callback functions may be NULL.
 */
typedef struct _pp_km_encoder_desc_t pp_km_encoder_desc_t;

struct _pp_km_encoder_desc_t {
    /**
     * Send events from the km to the encoder.
     * 
     * @param data_link The logical instance of the km that sent the event.
     *                  Usually this corresponds to a vsc instance.
     * @param event     A ptr to the event, valid during call only, copy if needed
     */
    void(*km_event)(pp_km_encoder_desc_t *ppthis, unsigned char data_link, km_event_t* event);
    
    /**
     * Notify the encoder that a new keyboard layout has been applied.
     * The client might be interested in the keyboard layout to display
     * a correct soft keyboard.
     * 
     * @param data_link The logical instance of the km that sent the message.
     *                  Usually this corresponds to a vsc instance.
     * @param kbd       The type of keyboard as defined in kbd_model in 
     *                  types_n_configs.cdl; for example pc101, pc104, ...
     */
    void(*encoder_send_kbdlayout)(pp_km_encoder_desc_t *ppthis, unsigned char data_link, char* kbd);
    
    /**
     * Notify the encoder that a local keyboard event occured.
     *
     * @param key      The virtual key number
     */
    void (*encoder_send_local_key_event)(pp_km_encoder_desc_t *ppthis, unsigned char key);
    
    /** 
     * Notify the encoder that a local mouse event occured.
     * Please note that in our current implementation, x and y
     * are relative positions, and we only capture wheel and
     * button events (so x and y are always set to 0).
     *
     * @param x        The x position of the mouse event
     * @param y        The y position of the mouse event
     * @param z        The z position of the mouse event (wheel)
     * @param bm       The current button mask (see above)
     */
    void (*encoder_send_local_mouse_event)(pp_km_encoder_desc_t *ppthis, int x, int y, int z, unsigned char bm);
};


/**
 * Initialize km lib.
 * @return 0 if successful
 */
int  pp_km_init(void);

/**
 * Cleans up the library and releases all the resources.
 * @return 0 if successful
 */
void pp_km_cleanup(void);

/**
 * Connect an encoder implementation to the km. Several encoders
 * can be registered in parallel by calling this function multiple
 * times. Encoders can be disconnected.
 * 
 * @param encoder  encoder decription structure containing
 *                 the information and callbacks of the actual encoder
 * @return         PP_SUC if successful
 *                 PP_ERR on error
 */
int pp_km_connect_encoder(pp_km_encoder_desc_t* encoder);

/**
 * Disconnect an encoder implementation from the km.
 *
 * @param encoder  encoder description structure to remove
 */
void pp_km_disconnect_encoder(pp_km_encoder_desc_t* encoder);


/**
 * Reset the currently active emulation.
 *
 * @param data_link emulation data_link to reset, in case it's a single
 *                  ported hardware, it is always 0
 * @return 0 if successful
 */
int  pp_km_reset_driver(unsigned char data_link);

/**
 * Test if this driver offers a reset function.
 *
 * @return 1 if driver is implemented, 0 otherwise
 */
int  pp_km_reset_driver_is_implemented(unsigned char data_link);


/**
 * (Re)synchronise the mousepointer.
 * 
 * The pointer can be kept in sync in two different modes.
 * The different modes are choosen via the according configuration settings.
 *
 * 1:1 mode:
 * This modes assumes the the host pointer is exactly moving like the
 * the relative pointer device tells him to move, i.e. if the pointer
 * device sends the command to move the pointer 1 pixel towards the left
 * the pointer will move 1 pixel towards the left.
 * Synchronization in this mode is done by forcefully moving the pointer
 * to the upper left corner and then back to the current position.
 *
 * intelligent mode:
 * This mode tries to determine the mouse translation function between
 * the pointer device commands and the actual movements of the pointer
 * on the screen. It is done by forcefully moving the pointer
 * to the upper left corner and probing different pointer commands in
 * horizontal direction. The result of the commands will be examined 
 * on the screen using image processing. This way, different pointer
 * speeds and acceleration steps will be recognized.
 *
 * Currently, the intelligent pointer sync algorithm works satisfactorily
 * for the different flavors of windows and graphical environments
 * based on X11.
 *
 * Nevertheless, there are the following issues:
 * - the pointer shape in the upper left region of the screen must not
 *   change during sync (shape changes could be caused by icons laying on
 *   the desctop, or due to an activated 'Active Desktop' in Windows.
 * - the pointer movement parameters must not be adjusted to extrem
 *   values, like very fast or very slow or very much accelerated.
 * - On WinXP the option 'Improve Mouse Movement' of the pointer dialog
 *   must be unchecked.
 * 
 * @param data_link  emulation data_link to use, in case it's a single
 *                   ported hardware, it is always 0
 * @param sync_type  synchronisation type, valid values are defined in
 *                   PP_KM_MOUSE_SYNC_*
 */
int  pp_km_sync_mouse(unsigned char data_link, int sync_type);


/**
 * Sends keycodes to the currently activated emulation.
 * The emulation will forward that in an emulation depended way
 * to its connected host. The keycodes used comply to an internal
 * and proprietary keymap, shown in kbd104.png.
 *
 * @param data_link emulation data_link to use, in case it's a single
 *                  ported hardware, it is always 0
 * @param keys      arrey of keycodes to send
 * @param length    length of keys array
 * 
 * @return         0 if successful
 *
 * \image html kbd104.png
 */
int  pp_km_send_keycodes(unsigned char data_link, u_char keys[], size_t length);

/**
 * Moves the pointer to the given position using the currently
 * activated emulation.
 *
 * @param data_link emulation data_link to use, in case it's a single
 *                 ported hardware, it is always 0
 * @param x        x coordinate of ptr movement 
 * @param y        y coordinate of ptr movement 
 * @param z        z coordinate of ptr movement 
 * @param buttonmask  state of the buttons as defined with the
 *                 BUTTON_*_MASK macros
 * @param abs_rel  specifies whether x, y, z are absolute or relative
 *                 coordinates
 * 
 * @return         0 if successful
 */
int  pp_km_send_ptrmove(unsigned char data_link, int x, int y, int z, u_char buttonmask, u_char abs_rel);


/**
 * Reconfigures the library according to the current
 * configuration settings.
 *
 * @param  data_link  emulation data_link to reconfigure, in case it's a single
 *                    ported hardware, it is always 0
 * @return 0 if successful
 */
int  pp_km_reconfigure(unsigned char data_link);


/**
 * Reconfigures all KM drivers according to the current
 * configuration settings.
 */
void pp_km_reconfigure_all(void);


/**
 * Test if the library can communicate with the underlying
 * emulation device (PS2, USB, ..)
 * 
 * @param data_link  emulation data_link to use, in case it's a single
 *                 ported hardware, it is always 0
 * @param  info    info/version of the specific emulation driver
 * @return         0 if successful
 */
int  pp_km_comm_proto_ping(unsigned char data_link, char ** info);

/**
 * For kimtester - directly reads/writes a KME PS/2 pin value
 * 
 * @param is_write    1 for writing, 0 for reading
 * @param  pin        number of the pin to access
 * @param  *val       where to store the value to (or read it from)
 * @param  data_link  emulation data_link to use, in case it's a single
 *                    ported hardware, it is always 0
 * @return            0 if successful
 */
int  pp_km_comm_proto_rw_pin(int is_write, int pin, int *val, unsigned char data_link);

/**
 * Tests if PS2 is available
 *
 * @param data_link  emulation data_link to use, in case it's a single
 *                   ported hardware, it is always 0
 * @return           1 if PS2 is supported, 0 otherwise
 */
int  pp_km_is_ps2_possible(unsigned char data_link);

/**
 * Tests if USB is available
 *
 * @param data_link  emulation data_link to use, in case it's a single
 *                   ported hardware, it is always 0
 * @return           1 if USB is supported, 0 otherwise
 */
int  pp_km_is_usb_possible(unsigned char data_link);

/**
 * Tests is USB emulation is currently active
 *
 * @param data_link  emulation data_link to use, in case it's a single
 *                   ported hardware, it is always 0
 * @return           1 if USB is active, false otherwise
 */
int  pp_km_is_usb_active(unsigned char data_link);

/**
 * Get the state of the keyboard leds of the selected data_link.
 * 
 * @param data_link  emulation data_link to use, in case it's a single
 *                   ported hardware, it is always 0
 * @return           keyboard led state
 */
int pp_km_get_keyboard_led_state(unsigned char data_link);

#ifdef __cplusplus
}
#endif 
#endif /* _PP_KM_H */


