/**
 * Copyright 2001 Peppercon AG
 * Author: Thomas Breitfeld <thomas@peppercon.de>
 *
 * Description: scancode 1 to scancode 2 mapping function
 */
#include <stdio.h>
#include <malloc.h>
#include <assert.h>
#include "sc1to2conv.h"

/**
 * obj table maps scan code 1 to 2
 * scancode 1 is considered as the index in 
 * the table, under the value for scancode 2 can be found. 
 * only the first 128 (make) codes are written down
 * the rest can be calculated using the known rules
 */
static unsigned char sc2tab[] = {
  /* 00 */  0xFF, /* -1 */
  /* 01 */  0x76,
  /* 02 */  0x16,
  /* 03 */  0x1E,
  /* 04 */  0x26,
  /* 05 */  0x25,
  /* 06 */  0x2E,
  /* 07 */  0x36,
  /* 08 */  0x3D,
  /* 09 */  0x3E,
  /* 0A */  0x46,
  /* 0B */  0x45,
  /* 0C */  0x4E,
  /* 0D */  0x55,
  /* 0E */  0x66,
  /* 0F */  0x0D,
  /* 10 */  0x15,
  /* 11 */  0x1D,
  /* 12 */  0x24,
  /* 13 */  0x2D,
  /* 14 */  0x2C,
  /* 15 */  0x35,
  /* 16 */  0x3C,
  /* 17 */  0x43,
  /* 18 */  0x44,
  /* 19 */  0x4D,
  /* 1A */  0x54,
  /* 1B */  0x5B,
  /* 1C */  0x5A,
  /* 1D */  0x14,
  /* 1E */  0x1C,
  /* 1F */  0x1B,
  /* 20 */  0x23,
  /* 21 */  0x2B,
  /* 22 */  0x34,
  /* 23 */  0x33,
  /* 24 */  0x3B,
  /* 25 */  0x42,
  /* 26 */  0x4B,
  /* 27 */  0x4C,
  /* 28 */  0x52,
  /* 29 */  0x0E,
  /* 2A */  0x12,
  /* 2B */  0x5D,
  /* 2C */  0x1A,
  /* 2D */  0x22,
  /* 2E */  0x21,
  /* 2F */  0x2A,
  /* 30 */  0x32,
  /* 31 */  0x31,
  /* 32 */  0x3A,
  /* 33 */  0x41,
  /* 34 */  0x49,
  /* 35 */  0x4A,
  /* 36 */  0x59,
  /* 37 */  0x7C,
  /* 38 */  0x11,
  /* 39 */  0x29,
  /* 3A */  0x58,
  /* 3B */  0x05,
  /* 3C */  0x06,
  /* 3D */  0x04,
  /* 3E */  0x0C,
  /* 3F */  0x03,
  /* 40 */  0x0B,
  /* 41 */  0x83,
  /* 42 */  0x0A,
  /* 43 */  0x01,
  /* 44 */  0x09,
  /* 45 */  0x77,
  /* 46 */  0x7E,
  /* 47 */  0x6C,
  /* 48 */  0x75,
  /* 49 */  0x7D,
  /* 4A */  0x7B,
  /* 4B */  0x6B,
  /* 4C */  0x73,
  /* 4D */  0x74,
  /* 4E */  0x79,
  /* 4F */  0x69,
  /* 50 */  0x72,
  /* 51 */  0x7A,
  /* 52 */  0x70,
  /* 53 */  0x71,
  /* 54 */  0x84,	/* special! */
  /* 55 */  0xFF,	/* -1 */
  /* 56 */  0x61,
  /* 57 */  0x78,
  /* 58 */  0x07,
  /* 59 */  0x0F,
  /* 5A */  0xFF,	/* -1 */
  /* 5B */  0x1F,
  /* 5C */  0x27,
  /* 5D */  0x2F,
  /* 5E */  0x37,
  /* 5F */  0x3F,
  /* 60 */  0x47,
  /* 61 */  0x4F,
  /* 62 */  0x56,
  /* 63 */  0x5E,
  /* 64 */  0x08,
  /* 65 */  0x10,
  /* 66 */  0x18,
  /* 67 */  0x20,
  /* 68 */  0x28,
  /* 69 */  0x30,
  /* 6A */  0x38,
  /* 6B */  0x40,
  /* 6C */  0x48,
  /* 6D */  0x50,
  /* 6E */  0x57,
  /* 6F */  0x6F,
  /* 70 */  0x13,
  /* 71 */  0x19,
  /* 72 */  0x39,
  /* 73 */  0x51,
  /* 74 */  0x53,
  /* 75 */  0x5C,
  /* 76 */  0x5F,
  /* 77 */  0x62,
  /* 78 */  0x63,
  /* 79 */  0x64,
  /* 7A */  0x65,
  /* 7B */  0x67,
  /* 7C */  0x68,
  /* 7D */  0x6A,
  /* 7E */  0x6D,
  /* 7F */  0x6E
};

/**
 * the scancode 2 will be returned in a newly allocated array
 * the caller is reponsible for freeing obj array after it 
 * has been used
 */
int conv_sc1_sc2(const unsigned char scs1[], const int  sc1length,
		 unsigned char* scs2[], int* sc2length)
{
  // double is far enough, perfect would be 3/2
  *scs2 = (unsigned char*)malloc(2*sc1length);
  *sc2length = 2*sc1length;
  conv_sc1_sc2_core(scs1, sc1length, *scs2, sc2length);
  return 0;
}

int conv_sc1_sc2_core(const unsigned char scs1[], const int  sc1length,
		      unsigned char scs2[], int* sc2length)
{
  unsigned char cc;
  int i1, i2 = 0, brk;
  assert(*sc2length >= (2 * sc1length));
  for(i1=0; i1<sc1length; ++i1) {
    cc = scs1[i1]; // cc is the current code
    // check for special codes E0, E1
    if(0xe0 == cc || 0xe1 == cc) {
      scs2[i2++] = cc;
      continue;
    }
    /* FIXME: hack from thre - DOCUMENT THIS, PLEASE! */
    if(cc == 0x54) {
      brk = 0;
    } else {
      brk = cc & 0x80; // make or break
      cc = cc & 0x7f;    // base code
    }
    
    if(brk) scs2[i2++] = 0xf0;
    scs2[i2++] = sc2tab[cc];
  }
  *sc2length = i2;
  return 0;
}


#ifdef TEST
/**
 * littel standalone test, to see if scan code conversion is 
 * really working, and also correectly, ofcourse.. ;-)
 */
int main(int argc, char** argv)
{
  int i, sc2length, count = argc - 1;
  unsigned char* sc = (unsigned char*)malloc(count);
  unsigned char* sc2;
  for(i=0; i<count; ++i) {
    sc[i] = strtol(argv[i+1], NULL, 16);
  }
  conv_sc1_sc2(sc, count, &sc2, &sc2length);
  return 0;
}

#endif /* TEST */
