#include <pthread.h>
#include <sys/time.h>
#include <pp/base.h>
#include <pp/cons.h>
#include <pp/kvm.h>
#include <liberic_misc.h>
#include "distribute.h"
#include "kvm_defaults.h"
#include "kvm_driver.h"

using namespace pp;
int pp_cons_cluster_active(void);

typedef enum {
    KVM_CLMSG_TYPE_SWITCH,
    KVM_CLMSG_TYPE_SWITCH_ACK,
} kvm_clmsg_type_t;
    
typedef struct {
    kvm_clmsg_type_t type;
    u_int id;
    u_char channel;
    u_char unit;
    u_short port;
    u_int error;
} switch_ack_kvm_clmsg_t;

typedef struct {
    kvm_clmsg_type_t type;
    u_int id;
    u_char channel;
    u_char unit;
    u_short port;
} switch_kvm_clmsg_t;

typedef union { 
    kvm_clmsg_type_t type;
    switch_ack_kvm_clmsg_t ack_msg;
    switch_kvm_clmsg_t switch_msg;
} kvm_clmsg_t;

extern KvmDriverBase *kvm_driver;

static u_int kvm_switch_ack_id = 0;
static pthread_cond_t kvm_msg_cond = PTHREAD_COND_INITIALIZER;
static pthread_mutex_t kvm_msg_mtx = PTHREAD_MUTEX_INITIALIZER;

PP_SYM_HIDDEN int
kvm_distribute_switch_port(u_char channel, u_char unit, u_short port)
{
    if (pp_cons_cluster_active()) {
        static u_int clmsg_id = 0;
        switch_kvm_clmsg_t switch_msg;
        u_int trials = 3;
        if (clmsg_id == 0) clmsg_id = channel + 1;
        switch_msg.type = KVM_CLMSG_TYPE_SWITCH;
        switch_msg.id = clmsg_id;
        switch_msg.channel = channel;
	switch_msg.unit = unit;
        switch_msg.port = port;
        clmsg_id += PP_KVM_CHANNEL_COUNT;

        MUTEX_LOCK(&kvm_msg_mtx);
        kvm_switch_ack_id = switch_msg.id;
        while (kvm_switch_ack_id > 0 && trials--) {
            struct timeval now;
            struct timespec timeout = { 0, 0 };

            pp_cons_bcast_clmsg(PP_CONS_CLMSG_TYPE_KVM, &switch_msg, sizeof(switch_msg));

            gettimeofday(&now, NULL);
            timeout.tv_sec  = now.tv_sec + 1;
            timeout.tv_nsec = now.tv_usec * 1000;
            pthread_cond_timedwait(&kvm_msg_cond, &kvm_msg_mtx, &timeout);
        }
        MUTEX_UNLOCK(&kvm_msg_mtx);
    }
    return 0;
}

PP_SYM_HIDDEN int
kvm_clmsg_hook(void * data, size_t data_len)
{
    pp_kvm_link_client_t client = NULL;
    kvm_clmsg_t * msg = (kvm_clmsg_t *)data;
    if (data_len == sizeof(switch_kvm_clmsg_t) && msg->type == KVM_CLMSG_TYPE_SWITCH) {
        switch_kvm_clmsg_t * switch_msg_p = (switch_kvm_clmsg_t *)msg;
        if (switch_msg_p->channel == (u_int)pp_misc_get_channel()) {
            switch_ack_kvm_clmsg_t switch_ack_msg;
            switch_ack_msg.type = KVM_CLMSG_TYPE_SWITCH_ACK;
            switch_ack_msg.id = switch_msg_p->id;
            switch_ack_msg.channel = switch_msg_p->channel;
            switch_ack_msg.unit = switch_msg_p->unit;
            switch_ack_msg.port = switch_msg_p->port;
#if 0 //TODO:
            switch_ack_msg.error = kvm_driver->DoSwitchPort( switch_msg_p->channel, 
                                                    switch_msg_p->unit, 
                                                    switch_msg_p->port) ? 1 : 0;
#endif
	    pp_cons_bcast_clmsg(PP_CONS_CLMSG_TYPE_KVM, &switch_ack_msg, sizeof(switch_ack_msg));
        }
        return 0;
    } else if (data_len == sizeof(switch_ack_kvm_clmsg_t) && msg->type == KVM_CLMSG_TYPE_SWITCH_ACK) {
        switch_ack_kvm_clmsg_t * switch_ack_msg_p = (switch_ack_kvm_clmsg_t *)msg;
        if (!switch_ack_msg_p->error) {
            if (switch_ack_msg_p->channel < PP_KVM_CHANNEL_COUNT) {
                if (switch_ack_msg_p->channel != (u_char)pp_misc_get_channel()) {
					kvm_driver->PortChanged ( switch_ack_msg_p->channel, 
								switch_ack_msg_p->unit, switch_ack_msg_p->port);
                }
		MUTEX_LOCK(&kvm_msg_mtx);
                if (switch_ack_msg_p->id == kvm_switch_ack_id) {
                    kvm_switch_ack_id = 0;
                    pthread_cond_signal(&kvm_msg_cond);
                }
                MUTEX_UNLOCK(&kvm_msg_mtx);
            }
        }
        return 0;
    } else {
        return -1;
    }
}
