#include <unistd.h>
#include <liberic_misc.h>
#include <pp/km.h>
#include <pp/um.h>

#include "kbd_switch.h"
#include "debug.h"

using namespace pp;

/* static method decls */

static void ReleaseStackKeys(u_char channel, vector_t *stack, u_char releasemode);

/* method implementations */

KvmKbdSwitch::KvmKbdSwitch()
{
    DP_VERBOSE("%s called\n", ___F);
}

int
KvmKbdSwitch::Init()
{
    DP_VERBOSE("%s called\n", ___F);

    return 0;
}

KvmKbdSwitch::~KvmKbdSwitch()
{
    /* destroy non-inherited members */

    /* destroy superclass */
}

int
KvmKbdSwitch::SendKeySeq(u_char channel, char * key_seq)
{
    vector_t *stack = NULL;
    vector_t *key_vec = NULL;
    int i, key, count;
    int ret = PP_ERR;
    u_int pause;
    u_char vkey;
    char *errortok;

    DP_NOTICE("%s: sending key sequence '%s' via channel %d\n", ___F, key_seq, channel);

    /* get pause duration */
    if (PP_FAILED(pp_cfg_get_uint(&pause, "kvm.key_pause_duration"))) {
	DP_ERROR("%s: kvm.key_pause_duration invalid\n", ___F);
	goto bail;
    }

    DP_NOTICE("%s: pause is %ums\n", ___F, pause);
    pause *= 1000;

    /* turn key_seq into vector of press/release keycodes */
    key_vec = eric_misc_key_get_keysequence(key_seq, &errortok);
    if (!key_vec) {
	DP_ERROR("%s: error while parsing port switching key seq %s\n", ___F, key_seq);
	goto bail;
    }
    
    count = vector_size(key_vec);
    stack = vector_new(NULL, count, NULL);
	    
    for (i = 0; i < count; ++i) {
	key = (int)vector_get_const(key_vec, i);
	
	switch(key) {
	  case KEYDELI_ADD:
	      break;
	  case KEYDELI_SEPERATE:
	      ReleaseStackKeys(channel, stack, KVM_KEY_RELEASE_ALL);
	      stack = vector_new(NULL, count, NULL);
	      break;
	  case KEYDELI_RELEASE_LAST:
	      ReleaseStackKeys(channel, stack, KVM_KEY_RELEASE_LAST);
	      break;
	  case KEYDELI_PAUSE:
	      usleep(pause);
	      break;
	  default:
	      vkey = (u_char) key;
	      vkey = key | 0x80;
	      if (PP_FAILED(pp_km_send_keycodes(0, &vkey, 1))) {
		  DP_ERROR("%s: error sending magic keys\n", ___F);
		  goto bail;
	      }
	      vector_add(stack, (void*)key);
	}
    }
    /* stack is released in ReleaseStackKeys() */
    ReleaseStackKeys(channel, stack, KVM_KEY_RELEASE_ALL);

    ret = PP_SUC;
 bail:
    free(key_vec);
    return ret;
}

/* Releases keys (integer keycodes) on STACK, either last or all.  Private. */
static void
ReleaseStackKeys( u_char NOTUSED(channel), vector_t * stack, u_char releasemode)
{
    int i, key;
    size_t size;
    u_char vkey;

    if (stack) {
	size = vector_size(stack);
	    
	if (releasemode == KVM_KEY_RELEASE_ALL) {
	    for (i = size - 1; i >= 0; --i) {
		key = (int)vector_get(stack, i);
		vkey = key;
		pp_km_send_keycodes(0, &vkey, 1);
	    }
	    vector_delete(stack);
	} else if (releasemode == KVM_KEY_RELEASE_LAST) {
	    if (size > 0) {
		key = (int) vector_get(stack, size-1);
		vkey = key;
		pp_km_send_keycodes(0, &vkey, 1);
		vector_remove(stack, size - 1);
	    }
	}
    }
}

