#include <stdlib.h>
#include <pp/cfg.h>
#include "kvm_defaults.h"

/* forward decls */
#if defined(PRODUCT_LARA) && defined(OEM_DAXTEN)
static int kvm_def_lara_daxten_init(kvm_default_t* def);
#endif

/* initializer for kvm_defaults */
kvm_default_t kvm_defaults[] = {
#if defined(PRODUCT_RIPCKIMXN)
    {"Belkin RIPC", 16, 500, 64, {}, {}, {}, kvm_def_ripc_init },
#endif
#if defined(PRODUCT_1601IP)
    {"", 16, 500, 64, {}, {}, {}, kvm_def_xx01ip_init },
#endif
#if defined(PRODUCT_0801IP)
    {"", 8, 500, 64, {}, {}, {}, kvm_def_xx01ip_init },
#endif
#if defined(PRODUCT_0801IPTT)
    {"", 8, 500, 64, {}, {}, {}, NULL },
#endif
#if ( defined(PRODUCT_LARA) && defined(OEM_MINICOM) ) || defined(OEM_SENTINEL)
    {"Minicom KVM-Switch", 64, 800, 64, {}, {}, {}, kvm_def_minicom_init },
#endif
#if defined(PRODUCT_LARA) && defined(OEM_XCEEDIUM)
     {"NET - KVM Switch", 64, 800, 64, {}, {}, {}, kvm_def_minicom_init },
#endif     
#if defined(PRODUCT_LARA) && defined(OEM_DAXTEN)
     {"SCOUTutp", 64, 800, 64, {}, {}, {}, kvm_def_lara_daxten_init },
#endif     
#if defined(PRODUCT_LARA) && defined(OEM_ADDER)
    {"SmartView World", 16, 500, 64, {}, {}, {}, kvm_def_adder_init },
#endif
#if defined(PRODUCT_LARA) && defined(OEM_LINDY)
    {"Matrix CPU Switch IP", 16, 500, 64, {}, {}, {}, kvm_def_adder_init },
#endif
    {NULL, 0, 0, 0, {}, {}, {}, NULL } /* keep this */
};

static int kvm_defaults_count = 0;

/**
 * various KVM default initialization functions
 */
int
kvm_defaults_init(void)
{
    kvm_default_t* def = kvm_defaults;
    
    while (def && def->name) {
	if (def->init) {
	    if (def->init(def)) {
		return -1;
	    }
	}
	kvm_defaults_count++; def++;
    }

    return 0;
}

void
kvm_defaults_cleanup(void)
{
    kvm_default_t* def = kvm_defaults;
    int count;

    while (def && def->name) {
	if (def->keys) {
	    count = def->count - 1;
	    while (count-- >= 0) {
		free(def->names[count]);
		free(def->keys[count]);
	    }
	}
	def++;
    }    
}

#if defined(PRODUCT_RIPCKIMXN)
int
kvm_def_ripc_init(kvm_default_t* def)
{
    int i;
    
    for (i = 0; i < def->count; ++i) {  
	if ((def->keys[i] = (char *)malloc(def->max_size)) == NULL) goto error;
	snprintf(def->keys[i], def->max_size, "CTRL-CTRL-0-0-%u-%u", (i+1)/10, (i+1)%10);
    }

    return 0;

 error:
    return -1;
}
#endif /* PRODUCT_RIPCKIMXN */

#if defined(PRODUCT_XX01IP_ANY)
int
kvm_def_xx01ip_init(kvm_default_t* def)
{
    int i;
    char* ports = "12345678ABCDEFGH";

    for (i=0; i<def->count; i++) {
	if ((def->keys[i] = (char*) malloc(def->max_size * sizeof(char))) == NULL) {
	    goto error;
	}
	snprintf(def->keys[i], def->max_size, "CTRL-CTRL-*%c", ports[i]);
    }
    
    return 0;
    
 error:
    return -1;
}
#endif /* PRODUCT_XX01IP_ANY */

#if defined(PRODUCT_LARA) && defined(OEM_DAXTEN)
static int
kvm_def_lara_daxten_init (kvm_default_t* def)
{
    int i;
    char key[4];

    assert(def);
    
    for (i = 0; i < def->count; i++) {
	/* set name (PC1-PC9)*/
	def->names[i] = (char*) malloc(def->max_size * sizeof(char));
	if (i < 9) {
	    snprintf(def->names[i], def->max_size, "PC%d", i+1);
	} else {
	    snprintf(def->names[i], def->max_size, "%s", "");
	}

	/* set key */
	def->keys[i] = (char*) malloc(def->max_size * sizeof(char));
	if (i<9) {
	    snprintf(key, 4, "%d", i+1);
	} else {
	    snprintf(key, 4, "%d-%d", (i+1)/10, (i+1)%10);
	}
	snprintf(def->keys[i], def->max_size, "CTRL-CTRL-*%s-ENTER", key);

	/* set show_in_rc (only show PC1-PC9) */
	def->show_in_rc[i] = (i < 9) ? 1 : 0; /* ensure positive values! */
    }
    /* end markers */
    def->keys[def->count] = NULL; 
    def->names[def->count] = NULL;
    def->show_in_rc[def->count] = -1;

    return 0;
}
#endif /* PRODUCT_LARA && OEM_DAXTEN */

#if ( defined(PRODUCT_LARA) && (defined(OEM_MINICOM) || defined(OEM_PEPPERCON) || defined(OEM_DAXTEN)) ) || defined(OEM_SENTINEL) || defined(OEM_XCEEDIUM)
int
kvm_def_minicom_init(kvm_default_t* def)
{
    char key[4];
    int i;

    for (i=0; i<def->count; i++) {
	def->keys[i] = (char*) malloc(def->max_size * sizeof(char));
	if (i<9) {
	    snprintf(key, 4, "%d", i+1);
	} else {
	    snprintf(key, 4, "%d-%d", (i+1)/10, (i+1)%10);
	}
	snprintf(def->keys[i], def->max_size, "LSHIFT-LSHIFT-*%s-ENTER", key);
    }
    def->keys[def->count] = NULL; /* end marker */

    return 0;
}
#endif /* ( PRODUCT_LARA && (OEM_MINICOM || OEM_PEPPERCON || OEM_DAXTEN) ) || OEM_SENTINEL || OEM_XCEEDIUM */

#if defined(PRODUCT_LARA) && (defined(OEM_ADDER) || defined(OEM_LINDY))
int
kvm_def_adder_init(kvm_default_t* def)
{
    char key[4];
    int i;

    for (i=0; i < def->count; i++) {
	/* set name */
	if ((def->names[i] = (char*) malloc(def->max_size * sizeof(char))) == NULL) {
	    goto error;
	}
	snprintf(def->names[i], def->max_size, "Computer %d", i+1);

	/* set key */
	if ((def->keys[i] = (char*) malloc(def->max_size * sizeof(char))) == NULL) {
	    goto error;
	}
	if (i < 9) {
	    snprintf(key, 4, "%d", i+1);
	} else {
	    snprintf(key, 4, "%d>%d", (i+1)/10, (i+1)%10);
	}
	snprintf(def->keys[i], def->max_size, "CTRL+ALT+%s", key);
    }

    return 0;
}
#endif /* PRODUCT_LARA && (OEM_ADDER || OEM_LINDY) */

/**
 * get the number of available defaults
 */
int
pp_kvm_get_defaultcount()
{
    return kvm_defaults_count;
}

/**
 * get the default KVM key duration
 */
int
pp_kvm_get_defaultduration(int index)
{
    return kvm_defaults[index].pause;
}

/**
 * get the names of a default entry
 */
const char*
pp_kvm_get_defaultname(int index)
{
    return kvm_defaults[index].name;
}

/**
 * get the names for a requested default
 */
char**
pp_kvm_get_defaultnames(int index)
{
    return kvm_defaults[index].names;
}

/**
 * get the hotkey entries for a requested default
 */
char**
pp_kvm_get_defaultkeys(int index)
{
    return kvm_defaults[index].keys;
}

/**
 * get the hotkey entries for a requested default
 */
int8_t*
pp_kvm_get_default_show_in_rc_vals(int index)
{
    return kvm_defaults[index].show_in_rc;
}

#ifdef PRODUCT_XX01IP_ANY
/**
 * put the defaults directly to the hotkey entires
 */
int
pp_kvm_put_defaults_to_keys(int index)
{
    int i;
    char** keys;

    keys = pp_kvm_get_defaultkeys(index);
    for (i = 0; keys && keys[i]; i++) {
	pp_cfg_set(keys[i], "unit[0].port[%u].hotkey.key", i);
    }

    pp_cfg_save(DONT_FLUSH);
    
    return 0;
}
#endif /* PRODUCT_XX01IP_ANY */

