/*
 * KvmDriver:  The root class for KVM.
 */

#ifndef _KVM_DRIVER_H
#define _KVM_DRIVER_H

/* global includes */
//#include <pp/oop_object.h>
#include <pp/vector.h>
#include <pp/kvm.h>
#include <liberic_pthread.h>

/* local includes */
#include "kvm_internals.h"
#include "kvm_driver_base.h"

namespace pp
{
class  KvmDriver : public KvmDriverBase
{
public:
	// Constructor
	KvmDriver();

	// Destructor
	virtual ~KvmDriver();

    /* Post-initialization after all constructors are finished.  Can be
     * used when inits are to do VCALLs.  Returns 0 on success.  */
    int Init();

    /**** ports, channels, and switching ****/

    /* Tells switch to switch to a new host UNIT/PORT on CHANNEL.
     * For switches not supporting multiple units, UNIT must be zero.
     */
    int Switch( LinkClient* client, u_char unit, u_short  target_port,
			    u_char* data_link, u_char* video_link )=0;

    virtual int setLinkClientLinks(LinkClient* client,
	    u_char data_link, u_char video_link);

    u_char GetCurrUnit(u_char channel);

    /* Get the currently active UNIT and PORT on KVM_NODE. Returns 0 on success. */
    virtual int getUnitAndPortForKvmNode(u_char kvm_node, int *unit_out, int *port_out);

    /* see superclass */
    virtual int getUnitAndPortForDataLink(int data_link_id,
	    int *unit_out, int *port_out);

    /* see superclass */
    virtual int getUnitAndPortForVideoLink(int video_link_id,
	    int *unit_out, int *port_out);

    /* Returns the number of currently available ports on the switch.  */
    u_short GetUnitPortCount (u_char unit);

    /* Tells the driver when kvm switch units are added or removed (Belkin-specific).
     * UNIT is 0-15, COUNT = number of ports in this unit (0 = removed).
     */
    void SetUnitPortCount(u_char unit, u_short count);

    /* Notifies driver that the port on CHANNEL has changed to UNIT/PORT,
     * The driver will update itself and pass info on to listeners.
     */
    void PortChanged(u_char channel, u_char unit, u_short port);

    virtual int getAnyVscVideoLinkForDataLink(int data_link_id, int *video_link_id);

    /* Is USER permitted to switch to UNIT/PORT ? */
    int IsPortAllowedForUser(const char * user, u_char unit, u_short port);

    /* Has an active host been detected at UNIT/PORT ? */
    int IsHostAtPort(u_char unit, u_short port);

    /* Is switching to a specific port possible and allowed for this user? */
    bool isSwitchToPortAllowed(const char *user, u_char unit, u_short port);

    /**** misc ****/

    /* Sets a new IP and mask in the config fs and in network device. */
    void ChangeIpAddress( uint32_t ip, int ip_valid, uint32_t mask, int mask_valid, uint32_t gw, int gw_valid);

#if defined(PP_FEAT_MASTERCONSOLE_FW_UPDATE)
    /* Updates the firmware on the KVM switch. */
    int UpdateMasterConsoleFirmware( char * firmware, size_t len);
#endif

protected:
    /* the current unit per channel */
    u_char _currUnit[PP_KVM_MAX_CHANNEL_COUNT];

    /* the currently active port within the current unit per channel */
    u_short _currPort[PP_KVM_MAX_CHANNEL_COUNT];

    /* protects _currPort */
    pthread_mutex_t _currPortMtx;

    /* protects port switching */
    pthread_mutex_t _switchPortMtx;

    /* Caches number of ports in unit i.  Needed for switches that only send
     * notifications of changes, but cannot be queried (e.g. belkin ripc).  */
    u_short _portCountInUnit[PP_KVM_MAX_UNIT_COUNT];

    /* protects _portsInUnit */
    pthread_mutex_t _portsInUnitMtx;
};

} // namespace



#endif /* _KVM_DRIVER_H */
