/*
 * KvmDriverBase:  The root class for KVM.
 */

#ifndef _KVM_DRIVER_BASE_H
#define _KVM_DRIVER_BASE_H

/* global includes */
#include <liberic_pthread.h>
#include <liberic_session.h>
#include <pp/vector.h>
#include <pp/kvm.h>
#include <pp/kvmdefs.h>
#include <link_client.h>

/* local includes */
#include "kvm_internals.h"

namespace pp
{

class  KvmDriverBase
{
public:
    // Constructor
    KvmDriverBase();

    // Destructor
    virtual ~KvmDriverBase();

    //-------------------------------------------------------------
    //                    PURE VIRTUAL FUNCTIONS
    //-------------------------------------------------------------

    /** 
     *  @brief	Creates a new LinkClient and registers it in this instance.
     *
     *  This LinkClient can later be used with Switch() for getting access to a
     *  specific target.
     * 
     *  @param  session_id     Client ID requesting the connection to the target 
     *  @param  session_type   This can be PP_KVM_LOCAL, PP_KVM_REMOTE, or
     *                         PP_KVM_DUMMY
     *
     *  @return  On success, returns a ptr to the new LinkClient, otherwise NULL
     *
     */
    virtual LinkClient* NewLinkClient(
	    eric_session_int_id_t session_id, pp_kvm_session_type_t session_type);

    /*
     * Switches the user to a requested target_port, the request
     * could be for "new connection" or temporary "switch" to the
     * target. Returns the data and video link for easy use.
     */
    virtual int Switch( LinkClient* client, u_char unit, u_short  target_port,
			u_char* data_link, u_char* video_link ) = 0;

    /*
     * Makes <client> use the given <data_link> and <video_link>.
     */
    virtual int setLinkClientLinks(LinkClient* client,
	    u_char data_link, u_char video_link);

    /* 
     * Release a LinkClient
     */
    virtual int ReleaseLinkClient ( LinkClient * client );

    /*
     * Returns the data_link_id connected to port on unit, if any
     *
     *  @return PP_SUC on success, otherwise PP_ERR
     */
    virtual int getDataLinkIdAtPort(int unit, int port, int *data_link_id);

    /*
     * Is switching to a specific port possible and allowed for this user?
     */
    virtual bool isSwitchToPortAllowed(const char *user, u_char unit, u_short port) = 0;

    /* Is USER permitted to switch to UNIT/PORT ? */
    virtual int IsPortAllowedForUser( const char * user, u_char unit, u_short port ) = 0;

    /** 
     *  @brief	Returns the unit and port number data_link_id is currently on.
     * 
     *  @param  data_link_id	data link ID to look at
     *  @param  unit_out	ptr for returning unit number
     *  @param  port_out	ptr for returning port number
     * 
     *  @return PP_SUC on success, otherwise PP_ERR
     *
     */
    virtual int getUnitAndPortForDataLink(int data_link_id,
	    int *unit_out, int *port_out) = 0;

    /** 
     *  @brief	Returns the unit and port number video_link_id is currently on.
     * 
     *  @param  video_link_id	video link ID to look at
     *  @param  unit_out	ptr for returning unit number
     *  @param  port_out	ptr for returning port number
     * 
     *  @return PP_SUC on success, otherwise PP_ERR
     *
     */
    virtual int getUnitAndPortForVideoLink(int video_link_id,
	    int *unit_out, int *port_out) = 0;

    /** 
     *  @brief	Returns the unit and port number of the given link client.
     * 
     *  @return PP_SUC and unit/port on success, otherwise PP_ERR
     *
     */
    virtual int getUnitAndPort(LinkClient* client,
	    int *unit_out, int *port_out);

    /*
     * Returns the number of currently available ports on the switch (on a specific unit).
     */
    virtual u_short GetUnitPortCount (u_char unit) = 0;

    /*
     * checks (if possible) whether a host is connected to a port
     * if not supported by HW, return true
     */
    virtual int IsHostAtPort(u_char unit, u_short port) = 0;

    //----------------------------------------------------------
    //                    VIRTUAL FUNCTIONS
    //----------------------------------------------------------

    /* Notifies driver that the port on CHANNEL has changed to UNIT/PORT,
     * The driver will update itself and pass info on to listeners.
     */
    virtual void PortChanged(u_char channel, u_char unit, u_short port);

    /*
     * Sets a new IP and mask in the config fs and in network device.
     */
    virtual void ChangeIpAddress( uint32_t ip, int ip_valid, uint32_t mask, int mask_valid, uint32_t gw, int gw_valid);

    /* Tells the kvm lib to enable/disable passing through the video of CHANNEL
     * to the local console port of the kvm switch.
     */
    virtual void SetLocalVideoState( u_char channel, int status);

    /* Tells the driver when kvm switch units are added or removed (Belkin-specific).
     * UNIT is 0-15, COUNT = number of ports in this unit (0 = removed).
     */
    virtual void SetUnitPortCount(u_char unit, u_short count);

    virtual int PortStatusChanged ( int, int, int );

    virtual int getAnyVscVideoLinkForDataLink(int data_link_id, int *video_link_id) = 0;

    virtual int getClientCntOnDataLink(int data_link_id, int *cnt_out);

    virtual int getClientCntOnVideoLink(int video_link_id);

protected:
    /* protects all members of this object */
    pthread_mutex_t m_mtx;
					
    /* the link clients registered with this driver */
    vector_t *m_link_clients;
};

} //namespace

#endif /* _KVM_DRIVER_H */
