/**	
 *	@file	pp_kvm.cpp
 *	@brief	Implementation of class KvmDriverKx2 class
 *
 *	
 */
/* system includes */
#include <stdio.h>
#include <unistd.h>
#include <arpa/inet.h>
#include <sys/time.h>

/* firmware includes */
#include <pp/base.h>
#include <pp/cfg.h>
#include <pp/um.h>
#include <pp/rfb.h>
#include <pp/OS_Port.h>
#include "kvm_driver_kx2.h"
#include <liberic_misc.h>
#include <pp/vsc.h>
#include <liberic_misc.h>
#include <liberic_pthread.h>
#include <pp_kernel_common.h>
#include "link_client.h"
#include <pp/cim_proto_lib.h>
#include <pp/km.h>
#include <pp/hal_common.h>
#include "debug.h"
#include <pp/acl.h>
#include <pp/pmdef.h>

/* local includes */

using namespace pp;

#define DEBUGLEVEL D_VERBOSE

/*----------------------------------------
 *	Equates
 *--------------------------------------*/

/*----------------------------------------
 *	Data Types
 *--------------------------------------*/

/*----------------------------------------
 *	Function Prototypes
 *--------------------------------------*/

/*----------------------------------------
 *	Forward References
 *--------------------------------------*/

/*----------------------------------------
 *	Static Data
 *--------------------------------------*/



extern DATALINK_OBJECTS * datalink;
extern CPortMgr_Device *pKVMBaseDevice;

/*----------------------------------------
 *	Code
 *--------------------------------------*/

/* -------------------------------------------------------------------- */

KvmDriverKx2::KvmDriverKx2()
{
    DP_VERBOSE("%s called\n", ___F);
    
    // set m_unit to 0 for kx2
    //-------------------------
    m_unit = 0;
    
    // Get max number of targets  supported
    // ---------------------------------------
    max_target_port_count = pp_hal_common_get_target_port_cnt();
    
    // Get max number of data links supported
    // ---------------------------------------
    max_data_link_count = pp_hal_common_get_data_link_cnt();

    // Get max number of video links supported
    // = number of VSCs + local port
    // ---------------------------------------
    max_video_link_count = pp_hal_common_get_vsc_cnt() + 1;

    // Initialize read write lock
    // --------------------------
    rwl = OS_CreateRWLock( OS_RWLOCK_NORMAL );
    assert (rwl != NULL);
}

/* -------------------------------------------------------------------- */
KvmDriverKx2::~KvmDriverKx2()
{
    // Delete the read/write lock
    // --------------------------
    if (rwl != NULL)
    {
	OS_DeleteRWLock( rwl );
	rwl = NULL;
    }
}

KvmDriverKx2*
KvmDriverBase::NewLinkClient(
	    eric_session_int_id_t session_id, pp_kvm_session_type_t session_type)
{
    LinkClient *client;
    
    client = new LinkClient(session_id, session_type, -1, -1);

    MUTEX_LOCK(&m_mtx);
    vector_add(m_link_clients, client);
    MUTEX_UNLOCK(&m_mtx);

    return client;
}

/* -------------------------------------------------------------------- */
void
KvmDriverKx2::ReadLock()
{
    OS_ReadLock( rwl );
}


/* -------------------------------------------------------------------- */
void
KvmDriverKx2::ReadUnlock()
{
    OS_ReadUnlock( rwl );
}


/* -------------------------------------------------------------------- */
void
KvmDriverKx2::WriteLock()
{
    OS_WriteLock( rwl );
}


/* -------------------------------------------------------------------- */
void
KvmDriverKx2::WriteUnlock()
{
    OS_WriteUnlock( rwl );
}

/* -------------------------------------------------------------------- */
int
KvmDriverKx2::getUnitAndPortForDataLink(int data_link_id, int *unit_out, int *port_out)
{
    int ret = PP_ERR;
    int i, cnt;
    LinkClient *client_i;
    int port = -1;
    bool found = false;

    MUTEX_LOCK(&m_mtx);

    cnt = vector_size(m_link_clients);
    for (i = 0; i < cnt; i++) {
	client_i = (LinkClient*)vector_get(m_link_clients, i);
	if (client_i->GetDataLinkId() == data_link_id) {
	    if (!found) {
		port = client_i->GetTargetPort();
		found = true;
	    } else {
		/* all clients with same data_link_id must have same port */
		assert(port == client_i->GetTargetPort());
	    }
	}
    }
    if (!found) goto bail;

    *unit_out = 0;
    *port_out = port;
    ret = PP_SUC;
bail:
    MUTEX_UNLOCK(&m_mtx);
    return ret;
}

/* -------------------------------------------------------------------- */
int
KvmDriverKx2::getUnitAndPortForVideoLink(int video_link_id, int *unit_out, int *port_out)
{
    int ret = PP_ERR;
    int i, cnt;
    LinkClient *client_i;
    int port = -1;
    bool found = false;

    MUTEX_LOCK(&m_mtx);

    cnt = vector_size(m_link_clients);
    for (i = 0; i < cnt; i++) {
	client_i = (LinkClient*)vector_get(m_link_clients, i);
	if (client_i->GetVideoLinkId() == video_link_id) {
	    if (!found) {
		port = client_i->GetTargetPort();
		found = true;
	    } else {
		/* all clients with same video_link_id must have same port */
		assert(port == client_i->GetTargetPort());
	    }
	}
    }
    if (!found) goto bail;

    *unit_out = 0;
    *port_out = port;
    ret = PP_SUC;
bail:
    MUTEX_UNLOCK(&m_mtx);
    return ret;
}

/* -------------------------------------------------------------------- */
int
KvmDriverKx2::AssignDataLink(LinkClient *client, u_short target_port,
	bool pc_share, int *data_link_id, int *err)
{
    int ret = PP_ERR;
    int i, cnt;
    LinkClient *client_i;
    int port_client_cnt;
    int ids_used[PP_KVM_DATA_LINK_CNT];
    int id_to_use = -1;

    assert(err);

    DP_VERBOSE("%s  session_id=%d, target_port=%d, session_type=%d \n",
	___F, client->GetSessionID(), target_port, client->GetSessionType());

    port_client_cnt = getClientCntAtPort(target_port);

    MUTEX_LOCK(&m_mtx);

    if (port_client_cnt == 0) { 
	/** nobody at this port -> find an unused data link **/

	memset(ids_used, '\0', sizeof(ids_used));

	cnt = vector_size(m_link_clients);
	for (i = 0; i < cnt; i++) {
	    int d_id;

	    // get data link ID
	    client_i = (LinkClient*)vector_get(m_link_clients, i);
	    d_id = client_i->GetDataLinkId();

	    // mark as used
	    assert(d_id < PP_KVM_DATA_LINK_CNT);
	    if (d_id >= 0) ids_used[d_id] = 1;
	}

	for (i = 0; i < PP_KVM_DATA_LINK_CNT; i++) {
	    if (!ids_used[i]) {
		id_to_use = i;
		break;
	    }
	}
    } else { /* port_client_cnt > 0 */
	/** sb. is at port -> find and reuse data link **/
	int client_cnt;

	if (PP_FAILED(getDataLinkIdAtPort(0 /* unit */, target_port, &id_to_use))) {
	    *err = PP_KVM_ERR_ASSIGN_DATALINK_ERROR;
	    goto bail;
	}

	// check max clients condition
	if (PP_FAILED(getClientCntOnDataLink(id_to_use, &client_cnt))) {
	    *err = PP_KVM_ERR_ASSIGN_DATALINK_ERROR;
	    goto bail;
	}
	if (client_cnt >= PP_KVM_MAX_CONNECTIONS_PER_LINK) {
	    *err = PP_KVM_ERR_MAX_CONN_REACHED_ERROR;
	    goto bail;
	}

	// check PCShare condition
	if (!pc_share )
	{
	    DP_ERROR("%s: PCShare not allowed =%d \n", ___F, pc_share);
	    *err = PP_KVM_ERR_PC_SHARE_NOT_ALLOWED_ERROR;
	    goto bail;
	}
    }

    if (id_to_use < 0) {
	*err = PP_KVM_ERR_ASSIGN_DATALINK_ERROR;
	goto bail;
    }

    // set link client vars (link ID and port)
    client->SetDataLinkId(id_to_use);

    *data_link_id = id_to_use;
    ret = PP_SUC;
bail:
    MUTEX_UNLOCK(&m_mtx);
    return ret;
}

/* -------------------------------------------------------------------- */
int
KvmDriverKx2::AssignVideoLink(LinkClient *client, u_short target_port,
	bool pc_share, int *video_link_id, int *err)
{
    int ret = PP_ERR;
    int i, cnt;
    LinkClient *client_i;
    int port_client_cnt;
    int ids_used[PP_KVM_VIDEO_LINK_CNT];
    int id_to_use = -1;

    assert(err);

    DP_VERBOSE("%s  session_id=%d, target_port=%d, session_type=%d \n",
	___F, client->GetSessionID(), target_port, client->GetSessionType());

    port_client_cnt = getClientCntAtPort(target_port);

    MUTEX_LOCK(&m_mtx);

    if (client->GetSessionType() == PP_KVM_LOCAL) {
	/** local port -> use the only available video link **/

	if (getClientCntOnVideoLink(PP_KVM_VIDEO_LINK_LOCAL_ID) > 0) {
	    *err = PP_KVM_ERR_ASSIGN_VIDEOLINK_ERROR;
	    goto bail;
	}
	id_to_use = PP_KVM_VIDEO_LINK_LOCAL_ID;
    } else if (port_client_cnt == 0) { 
	/** nobody at this port -> find an unused remote video link **/

	memset(ids_used, '\0', sizeof(ids_used));

	cnt = vector_size(m_link_clients);
	for (i = 0; i < cnt; i++) {
	    int v_id;

	    /* get video link ID */
	    client_i = (LinkClient*)vector_get(m_link_clients, i);
	    v_id = client_i->GetVideoLinkId();

	    /* mark as used */
	    assert(v_id < PP_KVM_VIDEO_LINK_CNT);
	    if (v_id >= 0) ids_used[v_id] = 1;
	}

	/* search remote IDs */
	int rid_min = PP_KVM_VIDEO_LINK_REMOTE_ID_MIN;
	int rid_after = PP_KVM_VIDEO_LINK_REMOTE_ID_MIN +
			PP_KVM_VIDEO_LINK_REMOTE_CNT;
	for (i = rid_min; i < rid_after; i++) {
	    if (!ids_used[i]) {
		id_to_use = i;
		break;
	    }
	}
    } else {
	/** remote port and sb. using it -> find and reuse video link **/
	int link_client_cnt;
	int data_link_id;

	if (PP_FAILED(getDataLinkIdAtPort(0 /* unit */, target_port, &data_link_id))) {
	    *err = PP_KVM_ERR_ASSIGN_VIDEOLINK_ERROR;
	    goto bail;
	}

	// get video link candidate
	if (PP_FAILED(getAnyVscVideoLinkForDataLink(data_link_id, &id_to_use))) {
	    *err = PP_KVM_ERR_ASSIGN_VIDEOLINK_ERROR;
	    goto bail;
	}

	/* check max clients condition */
	link_client_cnt = getClientCntOnVideoLink(id_to_use);
	if (link_client_cnt >= PP_KVM_MAX_CONNECTIONS_PER_LINK) {
	    *err = PP_KVM_ERR_MAX_CONN_REACHED_ERROR;
	    goto bail;
	}

	/* check PCShare condition */
	if (!pc_share)
	{
	    DP_ERROR("%s: PCShare not allowed =%d \n", ___F, pc_share);
	    *err = PP_KVM_ERR_PC_SHARE_NOT_ALLOWED_ERROR;
	    goto bail;
	}
    }

    if (id_to_use < 0) {
	*err = PP_KVM_ERR_ASSIGN_VIDEOLINK_ERROR;
	goto bail;
    }

    /* set link client vars (link ID and port) */
    client->SetVideoLinkId(id_to_use);

    *video_link_id = id_to_use;
    ret = PP_SUC;
bail:
    MUTEX_UNLOCK(&m_mtx);
    return ret;
}

/* -------------------------------------------------------------------- */
int
KvmDriverKx2::ReleaseDataLink ( LinkClient *client )
{
    client->SetDataLinkId(-1);

    return 0;
}

/* -------------------------------------------------------------------- */
int
KvmDriverKx2::ReleaseVideoLink ( LinkClient *client )
{
    client->SetVideoLinkId(-1);

    return 0;
}

/* -------------------------------------------------------------------- */
void
KvmDriverKx2::PortMgrPortIncBusy (int port_num )
{
    CPortMgr_Port * port = NULL;

    if ( pKVMBaseDevice != NULL )
    {
	port     = pp_portmgr->FindPort ( pKVMBaseDevice, port_num );
	port->IncBusy ();
    }
}

/* -------------------------------------------------------------------- */
void
KvmDriverKx2::PortMgrPortDecBusy (int port_num )
{
    CPortMgr_Port * port = NULL;

    if ( pKVMBaseDevice != NULL )
    {
	port     = pp_portmgr->FindPort ( pKVMBaseDevice, port_num );
	port->DecBusy ();
    }
}

/* -------------------------------------------------------------------- */
int
KvmDriverKx2::getClientCntAtPort(int port)
{
    int i, cnt;
    LinkClient *client_i;
    int found = 0;

    MUTEX_LOCK(&m_mtx);

    cnt = vector_size(m_link_clients);
    for (i = 0; i < cnt; i++) {
	client_i = (LinkClient*)vector_get(m_link_clients, i);
	if (client_i->GetTargetPort() == port) {
	    found++;
	}
    }

    MUTEX_UNLOCK(&m_mtx);

    return found;
}

/* -------------------------------------------------------------------- */
/*
 * Switches the user to a requested target_port, the request
 * could be for "new connection" or temporary "switch" to the
 * target. Returns the data and video link for easy use. 
 * If the request is from a local client, assign special video
 * link with PP_KVM_VIDEO_LINK_LOCAL_ID for local swich request
 */
int
KvmDriverKx2::Switch( LinkClient* client, u_char NOTUSED(unit), u_short target_port,
		    u_char* data_link_id_out, u_char* video_link_id_out )
{
    int video_link_id = -1; 
    int data_link_id = -1; 
    int client_cnt = -1;
    int status;
    int err;
    int ret = PP_ERR;
    int vl_done = 0;
    int dl_done = 0;
    
    WriteLock();

    // Check if user has active links 
    // connected to some targets
    //--------------------------------
    int active_links = client->hasActiveLinks();

    if (active_links == TRUE)
    {
	// Get the data_link & video_link
	//--------------------------------
	data_link_id = client->GetDataLinkId();
	video_link_id = client->GetVideoLinkId();

	// get #clients on data link
	if (PP_FAILED(getClientCntOnDataLink(data_link_id, &client_cnt))) {
	    WriteUnlock();
	    DP_ERROR("%s: getClientCntOnDataLink failed\n", ___F);
	    ret = PP_KVM_ERR_SWITCH_DATA_LINK_ERROR;
	    goto bail_err;
	}
    }

    // case 1: has active links and only one user on links
    //----------------------------------------------------
    if (active_links == TRUE && client_cnt == 1)
    {
	// Currently there is only one user on this link,
	// so there is no need to assign the link. Use
	// the one currently in use and just switch the
	// user to the new target
	// ----------------------------------
	client->SetTargetPort(0 /* unit */, target_port); 
    }
    // case 2: has active links but more than  one user on links  or
    // case 3: no active links
    //---------------------------------------------------------------
    else if ( (active_links == TRUE && client_cnt > 1) || active_links == FALSE )
    {
	//read if in PCShare mode
	const char *pc_share_key="security.pc_share_mode";
	bool pc_share=false;
	char *pc_share_str;

	if(PP_SUCCED(pp_cfg_get(&pc_share_str, pc_share_key))){
	    if(!pp_strcmp_safe(pc_share_str, "yes")) pc_share=true;
	}
	free(pc_share_str);

	DP_VERBOSE("%s: pc_share=%d\n", ___F, pc_share );

	if (PP_FAILED(AssignVideoLink(client, target_port, pc_share, &video_link_id, &err))) {
	    WriteUnlock();
	    DP_ERROR("%s: AssignVideoLink ERR: %d \n", ___F, err);
	    ret = err;
	    goto bail_err;
	}
	vl_done = 1;

	if (PP_FAILED(AssignDataLink(client, target_port, pc_share, &data_link_id, &err))) {
	    WriteUnlock();
	    DP_ERROR("%s: AssignDataLink ERR: %d \n", ___F, err);
	    ret = err;
	    goto bail_err;
	}
	dl_done = 1;

	client->SetTargetPort(0 /* unit */, target_port);
    }

    WriteUnlock();

    // get client_cnt
    if (PP_FAILED(getClientCntOnDataLink(data_link_id, &client_cnt))) {
	DP_ERROR("%s: cannot get #clients on DL=%d\n", ___F, data_link_id);
	ret = PP_KVM_ERR_SWITCH_DATA_LINK_ERROR;
	goto bail_err;
    }

    if (client->IsRemoteClient() == TRUE)
    {
	// Switch Remote video on the following condition
	// 1. when only one user on on link trying to
	//    switch to different target
	// 2. when no users are on that link.
	// -----------------------------------------------
	if (client_cnt <= 1)
	{
	    // If there are no users connected to this link 
	    // switch the remote video or else you will just 
	    // join the link and need not execute the code in
	    // this if statement
	    // FIXME: rename 'datalink' defined in libpp_cim_proto -> add lib prefix
	    status = datalink[data_link_id].object->SwitchDataLink(target_port);
	    if (status < 0)
	    {
		DP_ERROR("%s: SwitchDataLink ERR: %d \n", ___F, status);
		ret = PP_KVM_ERR_SWITCH_DATA_LINK_ERROR;
		goto bail_err;
	    }

	    status = datalink[data_link_id].object->SwitchRemoteVideo ( video_link_id, target_port );

	    if ( status < 0 )
	    {
		DP_ERROR("%s: ERR:Switch remote video failed %d \n", ___F, status);
		ret = PP_KVM_ERR_SWITCH_REMOTE_VIDEO_ERROR;
		goto bail_err;
	    }
	}

    }
    else
    {
	// Switch the data link
	//---------------------

	if (client_cnt <= 1)
	{
	    // If there are no users connected to this link 
	    // switch the remote video or else you will just 
	    // join the link and need not execute the code in
	    // this if statement
	    status = datalink[data_link_id].object->SwitchDataLink(target_port);

	    if (status < 0)
	    {
		DP_ERROR("%s: SwitchDataLink ERR: %d \n", ___F, status);
		ret = PP_KVM_ERR_SWITCH_DATA_LINK_ERROR;
		goto bail_err;
	    }

#ifdef DEBUG_LOCAL_VIDEO
	    // Switch the Local video
	    //-------------------------

	    //status = datalink[data_link_id].object->SwitchLocalVideo( data_link_id, target_port, 0);
	    status = datalink[0].object->SwitchLocalVideo( 0, 0, 0);

	    if (status < 0)
	    {
		DP_ERROR("%s: SwitchLocalVideo ERR: %d \n", ___F, status);
		ret = PP_KVM_ERR_SWITCH_LOCAL_VIDEO_ERROR;
		goto bail_err;
	    }
#endif
	}
    }

    // Reconfigure KM driver
    //-------------------------

    status = pp_km_reconfigure ( data_link_id );
    // FIXME: return value not used

    // Increment portmanager's port busy count
    // ---------------------------------------
    PortMgrPortIncBusy ( target_port) ;


    // FIXME: this cast is bad because it will shadow if *_link_id == -1
    *data_link_id_out = (u_char)data_link_id;
    *video_link_id_out = (u_char)video_link_id;

    DP_VERBOSE("%s -- Successfull\n", ___F);
    return PP_SUC;

bail_err:
    if (dl_done) ReleaseDataLink (client);
    if (vl_done) ReleaseVideoLink (client);
    if (!client->hasActiveLinks()) {
	client->SetTargetPort(0 /* unit */, target_port);
    }

    return PP_ERR;
}

/* -------------------------------------------------------------------- */
int
KvmDriverKx2::ReleaseLinkClient(LinkClient *client)
{
    //Implementation incomplete
    int status;
    int data_link_id;
    int client_cnt;
    int ret = PP_ERR;

    MUTEX_LOCK(&m_mtx);

    // allow releasing NULL
    if (client == NULL) {
	ret = PP_SUC;
	goto bail;
    }

    data_link_id = client->GetDataLinkId();

    if (PP_FAILED(getClientCntOnDataLink(data_link_id, &client_cnt))) {
	client_cnt = -1;
    }
    DP_VERBOSE("client_cnt=%d\n", client_cnt);

    ReleaseVideoLink(client);
    ReleaseDataLink(client);
	    
    // disconnect only if this was the last client on the data link
    if (client_cnt == 1) {
	status = datalink[data_link_id].object->DisconnectDataLink();

	if (status < 0){
	    DP_ERROR("cannot disconnect data link\n");
	    ret = PP_KVM_ERR_DISCONNECT_DATA_LINK_ERROR;
	    goto bail;
	}
    }

    super::ReleaseLinkClient(client);

    ret = PP_SUC;
bail:
    MUTEX_UNLOCK(&m_mtx);

    return ret;
}


/* -------------------------------------------------------------------- */
int
KvmDriverKx2::PortStatusChanged ( u_short target_port, int target_state )
{
    int ret = PP_ERR;
    int link_id = -1;

    if (target_state == PP_PORTMGR_STATE_DISCONNECTED) {
	ret = PP_SUC;
	goto bail;
    }

    if (PP_FAILED(getDataLinkIdAtPort(0 /* unit */, target_port, &link_id))) {
	goto bail;
    }

    ret = pp_km_reconfigure(link_id);

bail:
    return ret;
}

/* -------------------------------------------------------------------- */
int
KvmDriverKx2::IsHostAtPort(u_char NOTUSED(unit), u_short NOTUSED(port))
{
	/*
	* Return true for _each_ port.  Subclasses may overwrite this, e.g. if
	* they are able to detect hosts.
	*/
	return 1;
}

/* -------------------------------------------------------------------- */
int
KvmDriverKx2::IsPortAllowedForUser(const char * NOTUSED(user), u_char NOTUSED(unit), u_short NOTUSED(port))
{
    /* all users allowed on all ports (unless subclass says otherwise) */
    return 1;
}

/* -------------------------------------------------------------------- */
bool
KvmDriverKx2::isSwitchToPortAllowed(const char *user, u_char unit, u_short port)
{
    int user_ok, good_port;

    user_ok =  IsPortAllowedForUser( user, unit, port);
    good_port = IsHostAtPort ( unit, port );

    return user_ok && good_port;
}

/* -------------------------------------------------------------------- */
u_short
KvmDriverKx2::GetUnitPortCount(unsigned char unit)
{
    return (unit == 0) ? max_target_port_count : 0;
}

/* -------------------------------------------------------------------- */
int
KvmDriverKx2::getAnyVscVideoLinkForDataLink(int data_link_id, int *video_link_id)
{
    int ret = PP_ERR;
    int i, cnt;
    LinkClient *client_i;
    bool found = false;
    int v_id = -1;
    int port;

    MUTEX_LOCK(&m_mtx);

    cnt = vector_size(m_link_clients);
    for (i = 0; i < cnt; i++) {
	client_i = (LinkClient*)vector_get(m_link_clients, i);
	if (client_i->GetDataLinkId() == data_link_id) {
	    v_id = client_i->GetVideoLinkId();
	    port = client_i->GetTargetPort();
	    if (v_id >= PP_KVM_VIDEO_LINK_REMOTE_ID_MIN
		    && v_id < PP_KVM_VIDEO_LINK_REMOTE_ID_MAX
		    && port >= 0) {
		found = true;
		break;
	    }
	}
    }
    if (!found) goto bail;

    *video_link_id = v_id;
    ret = PP_SUC;
bail:
    MUTEX_UNLOCK(&m_mtx);
    return ret;
}
