#ifndef _KVM_DRIVER_KX2_H
#define _KVM_DRIVER_KX2_H

/* firmware includes */
#include <pp/OS_Port.h>
#include <liberic_session.h>
#include "kvm_driver_base.h"

#include <pp/kvmdefs.h>


/*----------------------------------------
 *	Equates
 *--------------------------------------*/

/*----------------------------------------
 *	Data Types
 *--------------------------------------*/

/*----------------------------------------
 *	Function Prototypes
 *--------------------------------------*/

/*----------------------------------------
 *	Static Data
 *--------------------------------------*/

/*----------------------------------------
 *	Class  KVM
 *--------------------------------------*/

/*  --------------------------------------------------------------------*/
/** 
  *  @class KvmDriverKx2
  *
  *  @brief	This class manages the data link and the video links. It 
  *  also processes the connection request to the targets.
  *   
  *  The KvmDriverKx2 gets request from RFB connection handler or the 
  *  RFB protocol handler to open new connection/switch to the target.
  *  This class implements functions to handle above requests.It handles
  *  both non pc-share and non pc-share connection.
  *  connections.
  */
	//--------------------------------------------------------------
	// I am still using Link, LinkClient and KVM class. But, some
	// changes are made to suit the requirements by christian. 
	// Comments via e-mail and Telecon discussion were considred here.
	// --------------------------------------------------------------

namespace pp
{

class KvmDriverKx2 : public KvmDriverBase
{
    typedef KvmDriverBase super;

public:
    
    /*  --------------------------------------------------------------------*/
    /** 
     *  @brief Constructor for KvmDriverKx2 class
     *
     */

    KvmDriverKx2();

    virtual ~KvmDriverKx2();

    virtual LinkClient* NewLinkClient(
	    eric_session_int_id_t session_id, pp_kvm_session_type_t session_type);

    /*  --------------------------------------------------------------------*/
    /** 
     *  @brief	Handles Switch request to the target_port
     *
     *  The Switch function, finds the available data link and video link and uses 
     *  them to connect to the target. If the link is already connected  to the
     *  target_port  it will use  (Only when PCShare=TRUE) that otherwise new 
     *  data and video links are assigned
     *
     *  When the requested target is busy and PCShare is set TRUE at device level
     *  and at the user level then the connection is joined to the busy target.
     *  if PCShare is not set TRUE then connection to busy targe is denied.
     * 
     *  @param  LinkClient*    Client ID requesting the connection to the target 
     *  @param  unit           Currently not used for KX2 products
     *  @param  target_port    The port number  the user is trying to connect to
     *  @param  data_link      The data_link number assigned for this request
     *  @param  video_link     The video_link number assigned for this request 
     *
     *
     *  @return PP_SUC  if the connection is succesful. Otherwise
     *          PP_KVM_ERR_ASSIGN_DATALINK_ERROR if cannot assign data link
     *          PP_KVM_ERR_ASSIGN_VIDEOLINK_ERROR if cannot assign video link
     *          PP_KVM_ERR_PC_SHARE_NOT_ALLOWED_ERROR if pc share not allowed
     *          PP_KVM_ERR_SWITCH_DATA_LINK_ERROR if cannot switch data link
     *          PP_KVM_ERR_SWITCH_REMOTE_VIDEO_ERROR if cannot switch remote 
     *                                              video
     *
     */

    int Switch( LinkClient* client, u_char unit, u_short  target_port,
			u_char* data_link, u_char* video_link );

    /*  --------------------------------------------------------------------*/
    /** 
     *  @brief Removes <client> from its data_link and the video_link and
     *  releases it.
     * 
     *  The video link and the data link will be marked back as available
     *  if the target is not currently shared
     * 
     *  @param  LinkClient*    Client ID to be released from the Link
     * 
     *  @return PP_SUC  if the disconnect is succesful. Otherwise
     *          PP_KVM_ERR_RELEASE_VIDEO_LINK_ERROR, if cannot release video link
     *          PP_KVM_ERR_RELEASE_DATA_LINK_ERROR, if cannot release data link
     *          PP_KVM_ERR_DISCONNECT_DATA_LINK_ERROR, if there is an error  
     *                                                disconnecting data link
     *
     */
    int ReleaseLinkClient ( LinkClient * client );

    /* see superclass */
    virtual int getUnitAndPortForDataLink(int data_link_id,
	    int *unit_out, int *port_out);

    /* see superclass */
    virtual int getUnitAndPortForVideoLink(int video_link_id,
	    int *unit_out, int *port_out);

    /*  --------------------------------------------------------------------*/
    /** 
     *  @brief PortStatusChanged is called by the PortManager library	
     *  The PortStatusChanged function calls km_reconfigure 
     *  appropriate data link_num. 
     * 
     *  @param  target_port     The target number for which the status has changed 
     *  @param  target_state  Disconnected or connected.
     *
     *  @return PP_SUC  if the connection is succesful. Otherwise
     *          PP_KVM_ERR_RELEASE_DATA_LINK_ERROR;
     *          PP_KVM_ERR_SWITCH_LOCAL_VIDEO_ERROR;
     *          PP_KVM_ERR_SWITCH_DATA_LINK_ERROR if cannot switch data link
     *
     */

    int PortStatusChanged ( u_short  target_port , int target_state );

    /* Is USER permitted to switch to UNIT/PORT ? */
    int IsPortAllowedForUser(const char * user, u_char unit, u_short port);

    /* Has an active host been detected at UNIT/PORT ? */
    int IsHostAtPort(u_char unit, u_short port);

    /* Is switching to a specific port possible and allowed for this user? */
    bool isSwitchToPortAllowed(const char *user, u_char unit, u_short port);

    u_short GetUnitPortCount(unsigned char);

    int getAnyVscVideoLinkForDataLink(int data_link_id, int *video_link_id);

    virtual int getClientCntAtPort(int port);
private:
    void ReadLock();
    void ReadUnlock();
    void WriteLock();
    void WriteUnlock();

    int AssignDataLink(LinkClient *client, u_short target_port,
	    bool pc_share, int *data_link_id, int *err);
    int AssignVideoLink(LinkClient *client, u_short target_port,
	    bool pc_share, int *video_link_id, int *err);
    int ReleaseDataLink ( LinkClient * client );
    int ReleaseVideoLink ( LinkClient * client );

    void PortMgrPortIncBusy (int port_num );
    void PortMgrPortDecBusy (int port_num );

private:
    /* TODO: replace all uses by m_mtx */
    OS_RWLOCK   rwl;       // Critical Section

    /* TODO: check */
    u_short  max_target_port_count;    // maximum number of targets on this device
    u_short  max_data_link_count;    // maximum number of data links 
    u_short  max_video_link_count;   // maximum number of video links

    u_char m_unit;   // The unit number
};

} // namespace

#endif	//_KVM_DRIVER_KX2_H

