#include <pp/features.h>

#if PP_FEAT_MASTERCONSOLE_FW_UPDATE

/* system includes */
#include <sys/ioctl.h>
#include <unistd.h>
#include <stdio.h>
#include <ctype.h>
#include <termios.h> // for tcflush

/* firmware includes */
#include <pp/base.h>
#include <pp/kvm.h>
#include <liberic_misc.h>
#include <liberic_pthread.h>

/* local includes */
#include "mcupdatedoc.h"

/* debugging */
#define USE_FAKE_FILE	0

#define UPDATE_DEBUG	0
#if UPDATE_DEBUG
#define DBG(fmt, x...)	pp_log(fmt, ##x)
#else
#define DBG(fmt, x...)	((void)0)
#endif

#define FILE_DEBUG	0
#if FILE_DEBUG
FILE *f_dbg = NULL;
#define FDBG(fmt, x...)	{ if (f_dbg) fprintf(f_dbg, fmt, ##x); }
#else
#define FDBG(fmt, x...)	((void)0)
#endif

/* macros */
#define BOOL	int
#define TRUE	1
#define FALSE	0

#define PURGE_TXABORT	(1 << 0)
#define PURGE_RXABORT	(1 << 1)
#define PURGE_TXCLEAR	(1 << 2)
#define PURGE_RXCLEAR	(1 << 3)

/* local data */
#define MAX_DATA_LEN	(64 * 1024)
static char fw_data[MAX_DATA_LEN];

static BOOL m_UST;
static BOOL m_UXU2;
static BOOL m_XXX;
static BOOL m_64k;
static BOOL m_256KDownload;
static BOOL m_OKDownload;
static BOOL m_CancelFlag;
static int m_ComHandle;
static unsigned char SNAck;
static char SerialNum[17];
#define MaxAuxVals 20
static int  NumberAuxiliaryValues;
static char AuxiliaryValues[MaxAuxVals][4];
static char DevType[4];
static char FirmwareVersion[4];
static char HardwareVersion[3];
static char FileHardwareVersion[3];
static char FileAuxiliaryValues[MaxAuxVals][4];
static char FileDevType[4];
static char FileFirmwareVersion[4];

/* function prototypes */
static int Serialize(char* firmware, size_t length);
static int UploadHex(unsigned char mode);
static BOOL InitMC();
static BOOL ReadSerialNumber(unsigned char startcode);
static int UploadMC2Func();

#if USE_FAKE_FILE
static char *filebuffer = NULL;
static unsigned int firmware_length = 0;

using namespace pp;

int read_file() {
    int ret = -1;
    char* filename = "MPR8.HEX";
    FILE *f;
    unsigned int actually_read;

    // open the file
    f = fopen(filename, "rb");
    if (!f) {
	pp_log("Could not open file!\n");
	goto bail;
    }

    // find out filesize
    if (fseek(f, 0, SEEK_END)) {
	pp_log("Could not seek to end.\n");
	goto bail;
    }
    firmware_length = ftell(f);
    if (fseek(f, 0, SEEK_SET)) {
	pp_log("Could not seek to begin.\n");
	goto bail;
    }
    DBG("file is %d bytes big.\n", firmware_length);

    // read it
    filebuffer = (char *) malloc(firmware_length + 5000);
    if (!filebuffer) {
	 pp_log("Could not allocate space.\n");
	 goto bail;
    }
    memset(filebuffer, 0, firmware_length + 5000);
    if ((actually_read = fread(filebuffer, 1, firmware_length, f)) != firmware_length) {
	pp_log("Could not read all bytes from file, got %d bytes.\n", actually_read);
	goto bail;
    }

    ret = 0;

 bail:
    if (f) {
	fclose(f);
    }
    return ret;
}
#endif // USE_FAKE_FILE

/* main update function */
int update_master_console(int fd, char* firmware, size_t length)
{
    int ret = -1;
    int i;
    
    DBG("Updating MasterConsole.\n");
        
#if USE_FAKE_FILE
    if (read_file() != 0) {
    	pp_log("Cannot read firmware from file.\n");
    	goto bail;
    }
    
    firmware = filebuffer;
    length = firmware_length;
#endif // USE_FAKE_FILE

#if FILE_DEBUG
    f_dbg = fopen("debug.txt", "w");
#endif

    // initialize the variables
    memset(fw_data, 0, sizeof(fw_data));
    m_UST = FALSE;
    m_UXU2 = FALSE;
    m_XXX = FALSE;
    m_64k = FALSE;
    m_OKDownload = FALSE;
    m_256KDownload = FALSE;
    m_CancelFlag = FALSE;
    FirmwareVersion[0] = '?';
    HardwareVersion[0] = '?';
    DevType[0] = '?';
    FileFirmwareVersion[0] = '?';
    FileHardwareVersion[0] = '?';
    FileDevType[0] = '?';
    for (i = 0; i < MaxAuxVals; i++) {
    	AuxiliaryValues[i][0] = '?';
    	FileAuxiliaryValues[i][0] = '?';
    }
    
    // do the update
    m_ComHandle = fd;
    
    if (Serialize(firmware, length) < 0) {
    	pp_log("Serializing firmware failed.\n");
    	goto bail;
    }
    
    if (UploadHex(3) < 0) {
    	pp_log("Updating MasterConsole firmware failed!\n");
    	goto bail;
    }
    
    ret = 0;
    
 bail:
 
#if USE_FAKE_FILE
    if (filebuffer) {
    	free(filebuffer);
    	filebuffer = NULL;
    }
#endif // USE_FAKE_FILE

#if FILE_DEBUG
    fclose(f_dbg);
    f_dbg = NULL;
#endif
    return ret;
}

/******************************************************************************
* the following code is ported from the raritan MCUpdate software             *
******************************************************************************/

/******************************************************************************
* low level read/write functions                                              *
******************************************************************************/

#define SERIAL_TIMEOUT	1	// seconds

static int WriteFile(int fd, char* buffer, int size, int* written_size, void* /* overlapped */)
{
    struct timeval to;
    fd_set aliveset;

    FD_ZERO(&aliveset);
    FD_SET(fd, &aliveset);
    	
    to.tv_sec = SERIAL_TIMEOUT;
    to.tv_usec = 0;

    if (select(fd + 1, NULL, &aliveset, NULL, &to) < 0) {
	pp_log_err("WriteFile: select\n");
	*written_size = 0;
	return -1;
    }
    	
    if(!FD_ISSET(fd, &aliveset)) {
	// writing not possible
	DBG("Cannot write, select timed out.\n");
	*written_size = 0;
	return -1;
    }
    
    *written_size = write(fd, buffer, size);
    if (*written_size <= 0) {
    	*written_size  = 0;
    	return -1;
    }
    return 0;
}

static int ReadFile(int fd, char* buffer, int size, int* read_size, void* /* overlapped */)
{
    struct timeval to;
    fd_set aliveset;

    FD_ZERO(&aliveset);
    FD_SET(fd, &aliveset);
    	
    to.tv_sec = SERIAL_TIMEOUT;
    to.tv_usec = 0;

    if (select(fd + 1, &aliveset, NULL, NULL, &to) < 0) {
	pp_log_err("ReadFile: select\n");
	*read_size = 0;
	return -1;
    }
    	
    if(!FD_ISSET(fd, &aliveset)) {
	// writing not possible
	DBG("Cannot read, select timed out.\n");
	*read_size = 0;
	return -1;
    }
    
    *read_size = read(fd, buffer, size);
    if (*read_size <= 0) {
	pp_log_err("ReadFile: read\n");
    	*read_size  = 0;
    	return -1;
    }
    return 0;
}

static int PurgeComm(int fd, u_int32_t flags)
{
    if(flags & PURGE_TXABORT) tcflush(fd,TCOFLUSH);
    if(flags & PURGE_RXABORT) tcflush(fd,TCIFLUSH);
    if(flags & PURGE_TXCLEAR) tcflush(fd,TCOFLUSH);
    if(flags & PURGE_RXCLEAR) tcflush(fd,TCIFLUSH);
    
    return 0;
}

#if FILE_DEBUG

static int WriteFileDump(int fd, char* buffer, int size, int* written_size, void* overlapped UNUSED)
{
    int i;
    FDBG(":");
    for (i = 0; i < size; i++) {
    	FDBG("%02X", buffer[i]);
    }
    FDBG("\n");
    return WriteFile(fd, buffer, size, written_size, overlapped);
}

#else // FILE_DEBUG

#define WriteFileDump WriteFile

#endif // FILE_DEBUG

/******************************************************************************
* main firmware flashing and communicating functions                          *
******************************************************************************/

static const char* ShowProductName(const char* devcode)
{
    if (strncmp(devcode, "MXa", 3) == 0)
    	return("Raritan MX(Type A)");
    else if (strncmp(devcode, "MXb", 3) == 0)
    	return("Raritan MX(Type B)");
    else if (strncmp(devcode, "SMb", 3) == 0)
    	return("Sun MX(Type B)");
    else if (strncmp(devcode, "SMp", 3) == 0)
    	return("Sun MX(Type P)");
    else if (strncmp(devcode, "P1 ", 3) == 0)
    	return("Send To Paragon");
    else if (strncmp(devcode, "UMC", 3) == 0)
    	return("Send To Paragon");
    else if (strncmp(devcode, "UST", 3) == 0) {  
    	m_256KDownload=TRUE;
    	return("Send To Paragon");
    }
    else if (strncmp(devcode, "P2", 2) == 0) {
    	m_256KDownload=TRUE;
    	return("Send To Paragon");
    }
    else if (strncmp(devcode, "P1", 2) == 0)
    	return("Send To Paragon");
    else if (strncmp(devcode, "UXU", 3) == 0)
    	return("UMT2161");
    else if (strncmp(devcode, "XXX", 3) == 0) {  
    	m_256KDownload=TRUE;
    	return("Send To Paragon");
    }
    return("Raritan MasterConsole");
}


static int UploadHex(unsigned char mode) 
{
    SNAck = Ack;
    PurgeComm(m_ComHandle, PURGE_TXABORT | PURGE_RXABORT | PURGE_TXCLEAR | PURGE_RXCLEAR);
    
    switch(mode) {
    	case 3:	// //MC3 Mode(binary, accepts Intel extended Hex format)
    	    if (!InitMC()) {
    	    	pp_log("Connection error! Please reset power and try again!\n");
    	    	break;
    	    }
    	    
    	    if (m_256KDownload == FALSE && m_UST == FALSE && m_UXU2 == FALSE && m_XXX == FALSE && m_64k == TRUE) {
    	    	if(m_OKDownload==TRUE) {
    	    	    DBG("Firmware update possible. Starting.\n");
    	    	    return UploadMC2Func();
    	    	}
    	    }
    	    else {
    	    	pp_log("Wrong device or device does not match firmware file.\n");
    	    }
    	    pp_log("Firmware upload not possible.\n");
    	    return -1;
    	    
    	case 1:	// Raw data mode(text)
    	case 2:	// MC2 Mode(binary)
    	case 4:	// MC3 Mode(binary, accepts Intel extended Hex format), serial only
    	default:
    	    pp_log("%s: Mode 0x%x not supported!\n", __FUNCTION__, mode);
    }
    
    return -1;
}

static BOOL InitMC()
{
    char c = 0, d = ESC;			//char d is an escape code
    int bytes;
    int j;
    const char *prod;
    
    DBG("Attempting to communicate with Paragon Unit...\n");
    
    for (j = 0; ((j < 1) && (c != ESC)); j++)	//Send six escapes, wait 2 seconds each for an esc reply
    {
    	WriteFile(m_ComHandle, &d, 1, &bytes, NULL);
    	ReadFile(m_ComHandle, &c, 1, &bytes, NULL);
    	
    	switch (c) {
    	    case ESC:
    	    	DBG("Got a 64k device.\n");
    	    	m_64k = 1;
    	    	m_OKDownload = 1;
    	    	break;
    	    	
    	    case DeviceMC3:
    	    case DeviceMC3b:
    	    case DeviceParagon:
    	    case DeviceUXU2:
    	    case DeviceUSTParagon:
    	    	DBG("Communication Established.\n");
    	    	while (j != 1) {
    	    	    char x = NAk, temp;
    	    	    if (ReadSerialNumber(c)) {
    	    	    	j = 1;
    	    	    	temp = SerialNum[0];
    	    	    	
    	    	    	c = ESC;   m_OKDownload = 1;
    	    	    	if (((temp != 0xFF) && ((char)SNAck != Ack)) || (c != ESC)) {
    	    	    	    x = Reset;
    	    	    	}
    	    	    	else {
    	    	    	    x = SNAck;
    	    	    	}
    	    	    	SNAck = temp;
    	    	    }
    	    	    WriteFile(m_ComHandle, &x, 1, &bytes, NULL);
    	    	}
    	    	break;
    	    	
    	    default:
    	    	ReadFile(m_ComHandle, &c, 1, &bytes, NULL);
    	    	break;
    	}
    }
    prod = ShowProductName(DevType);	// this is needed
    DBG("Product name: %s\n", prod);
    if ((bytes != 0) && (c == ESC)) {		//If escape was received, inform user
    	pp_log("Communication established with Paragon Unit.\n");
    	if((DevType[0]=='U')&&(DevType[1]=='M')&&(DevType[2]=='C'))  m_UST=1;
    	else if((DevType[0]=='U')&&(DevType[1]=='X')&&(DevType[2]=='U')) m_UXU2=1;
    	else if((DevType[0]=='X')&&(DevType[1]=='X')&&(DevType[2]=='X'))  m_XXX=1;
    	
    	PurgeComm(m_ComHandle, PURGE_TXABORT | PURGE_RXABORT | PURGE_TXCLEAR | PURGE_RXCLEAR);
    	return TRUE;
    }
    else if (c==28) {
    	PurgeComm(m_ComHandle, PURGE_TXABORT | PURGE_RXABORT | PURGE_TXCLEAR | PURGE_RXCLEAR);
    	return TRUE;
    }
    else {
    	SerialNum[0] = 0xFF;
    	PurgeComm(m_ComHandle, PURGE_TXABORT | PURGE_RXABORT | PURGE_TXCLEAR | PURGE_RXCLEAR);
    	return FALSE;
    }
}

static BOOL ReadSerialNumber(unsigned char startcode)
{
    int bytes;
    char checksum;
    char buf[30];
    
    ReadFile(m_ComHandle, SerialNum, 16, &bytes, NULL);
    if (bytes != 16) {
    	pp_log("Read Error: Not enough bytes received.\n");
    	SerialNum[0] = 0xFF;
    	return FALSE;
    }
    
    if (startcode != 0xC3) {
	ReadFile(m_ComHandle, buf, 9, &bytes, NULL);

	NumberAuxiliaryValues = buf[0];

	ReadFile(m_ComHandle, &(buf[9]), NumberAuxiliaryValues * 3, &bytes, NULL);
	if (bytes != NumberAuxiliaryValues * 3) {
	    pp_log("Read Error: Missing Parameters.\n");
	    SerialNum[0] = 0xFF;
	    return FALSE;
	}
	

	if (NumberAuxiliaryValues > MaxAuxVals)
		NumberAuxiliaryValues = MaxAuxVals;
	strncpy(DevType, &buf[1], 3);
	DevType[3] = 0;
	strncpy(HardwareVersion, &buf[4], 2);
	HardwareVersion[2] = 0;
	strncpy(FirmwareVersion, &buf[6], 3);
	FirmwareVersion[3] = 0;
	for (bytes = 0; bytes < NumberAuxiliaryValues; bytes++) {
	    strncpy(AuxiliaryValues[bytes], &buf[9 + (bytes * 3)], 3);
	    AuxiliaryValues[bytes][3] = 0;
	}
	for (bytes = 0; bytes < 9 + (buf[0] * 3); bytes++)
	    startcode += buf[bytes];
    }
    else {
	DevType[0] = '?';
	HardwareVersion[0] = '?';
	FirmwareVersion[0] = '?';
    }
    
    for (bytes = 0; bytes < 16; bytes++)
	startcode += SerialNum[bytes];
	
    ReadFile(m_ComHandle, &checksum, 1, &bytes, NULL);
    
    if (startcode != checksum) {
	pp_log("Read Error: Checksum incorrect.\n");
	SerialNum[0] = 0xFF;
	return FALSE;
    }
    
    return TRUE;
}

/******************************************************************************
* uploading a firmware                                                        *
******************************************************************************/

#define MAX_STRING_LENGTH	256

static BOOL SendMCRecord(int sbytes, char * buf, char* s, BOOL hold)
{		//Sends a record and handles reponse, take bytes to be sent, data, message, and the progress
	int tries = 0, bytes = 0;
	char c = 0;
	tries=0;
	
	while ((c != ESC) && (c != Ack) && (c != Send) && (!m_CancelFlag)) {	//Loop until positive reponse
		tries++;
		if ((tries % 20) == 0) {					//Cancel if MC 2 stops responding
			pp_log("Warning: The Paragon Unit is not responding. Operation is cancelled.\n");
			m_CancelFlag = TRUE;
		}
		c = 0;
		WriteFileDump(m_ComHandle, buf, sbytes, &bytes, NULL);		//Write to port
		if (bytes != sbytes) {
			pp_log_err("Warning: Serial port is not accepting transmission. Bytes written: %d.\n", bytes);
		}
		DBG("%s%d\n", s, tries);						//Set number of tries
		ReadFile(m_ComHandle, &c, 1, &bytes, NULL);			//Read response, timeout 1 s.
	} 
	if (c == Hold) {
		while ((c != Send) && (!m_CancelFlag) && hold) {
			ReadFile(m_ComHandle, &c, 1, &bytes, NULL);		//Hold received, loop until Res
		}
	}

	if ((bytes != 1) || ((c != Ack) && (c != Send)) || (m_CancelFlag)) {	//If loop exited, see if error or okay
		return FALSE;							//Return error condition
	}
	return TRUE;								//Return okey
}

static void GetBinaryRecord(unsigned char Block, unsigned char Page, unsigned char Rec, char * buf, BOOL MC2)
{
	int i;
	
	buf[0] = 0x10;				//Set number of data bytes
	if (MC2)
		buf[1] = Page;						//Set upper byte of address on MC2(no offset)
	else
		buf[1] = (Block * 64 + Page) & 0xff;	//Set upper byte of address
	buf[2] = (Rec * 16) & 0xff;			//Set lower byte of address
	buf[3] = 0x00;					//Set record type
	for (i = 4; i < 20; i++)
		buf[i] = fw_data[(Block * 64 + Page)*256 + (Rec * 16) + i - 4];
	unsigned char checksum = 0x00;
	for (i = 0; i < 20; i ++)
		checksum -= buf[i];
	buf[20] = checksum;
}

static int UploadMC2Func()
{
	int Block, Page, Rec; 
	char s[MAX_STRING_LENGTH + 1];
	BOOL test;
	int bytes = 0;
	int j;
	int ret = 0;
	
	for (Block = 0; Block < 4; Block++) {		//Iterate through blocks(16KB)
		char c = 0x3F + (Block * 0x40);
		char buf[] = {2,(Block * 0x40),0,4,c,0xff,0x0 - 0xff - 2 - 4 - c - (Block * 0x40)};
		pp_log("Sending MCHex File: Block %d of 4: ", Block + 1);
		snprintf(s, MAX_STRING_LENGTH, "Sending MCHex File: Block %d ID,  Try:\n", Block + 1);	//Human Readable
		if (!SendMCRecord(7, buf, s, TRUE)) {
			break;
		}
		for (Page = 0; Page < 64; Page++) {		//Iterate through pages(256B)
			for (Rec = 0; Rec < 16; Rec++) {	//Iterate through records(16B)
				char buf2[21];   // = {0x10, Block * 64 + Page, Rec * 16, 00, 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, 0x00 - 0x10 - (Rec * 16) - (Block * 64 + Page)};
				GetBinaryRecord(Block, Page, Rec, buf2, TRUE);
				buf2[1] = Page;             //MC 2 specific override -- Each block starts at 0x0000
				if (buf2[3] == 0xff) {
					pp_log("Non-existant Segment Error\n");
					test = FALSE;
					break;
				}
				snprintf(s, MAX_STRING_LENGTH, "Sending MCHex File : Block %d/4,  Page %d/64,  Record %d/16,  Try:", Block + 1, Page + 1, Rec + 1);
				test = SendMCRecord(21, buf2, s, TRUE);
				if (!test) {
					break;
				}
			}
			printf(".");
			fflush(stdout);
			if (!test) {
				break;
			}
		}
		printf("\n");
		fflush(stdout);
		if (!test) {
			break;
		}
		buf[0] = 0; buf[1] = 0; buf[2] = 0; buf[3] = 1; buf[4] = 0xFF; //End of Block record
		snprintf(s, MAX_STRING_LENGTH, "Sending End of Block: Block %d, Try:", Block + 1);
		if (!SendMCRecord(5, buf, s, FALSE)) {
			break;
		}
		
		for (j = 60; j > 0; j--) {
			DBG("Waiting for MC 2 to program block.  Timeout: %d s\n", j);
			ReadFile(m_ComHandle, &c, 1, &bytes, NULL);
			if (((bytes != 0) && ((c == Send) || (c == EErr) || (c == PErr) || (c == CErr))) || (m_CancelFlag)) {
				break;
			}
		}
		if ((bytes == 1) && (c != Send)) {				//Display block status
			Block--;
			pp_log("Block Error\n");
			switch (c) {
				case EErr:
					pp_log("Error updating block: Error Code: F6(Erase Error)\n");
					break;
				case PErr:
					pp_log("Error updating block: Error Code: F7(Programming Error)\n");
					break;
				case CErr:
					pp_log("Error updating block: Error Code: F8(Block Checking Error)\n");
					break;
			}
		}
		if (m_CancelFlag) {
			break;
		}
	}
	if ((Block == 4) && (Page == 64) && (Rec == 16)) {
		char buf[] = {0,0,0,5,0xFB};
		WriteFileDump(m_ComHandle, buf, 5, &bytes, NULL);
		bytes = 1;
		pp_log("Transmission completed! The Paragon Unit should now restart automatically.\n");
	}
	else {
		pp_log("Transmission Terminated. Power off and on the Paragon Unit and try to upload again.\n");
		bytes = 0;
		ret = -1;
	}
	PurgeComm(m_ComHandle, PURGE_TXABORT | PURGE_RXABORT | PURGE_TXCLEAR | PURGE_RXCLEAR);
	
	return ret;
}

/******************************************************************************
* Serializing a firmware file                                                 *
******************************************************************************/

#if UPDATE_DEBUG
static int dump_it() {
    int ret = -1;
    char* filename = "dump.HEX";
    FILE *f;
    unsigned int i;

    // open the file
    f = fopen(filename, "w");
    if (!f) {
	pp_log("Could not open file!\n");
	goto bail;
    }

    for (i = 0; i < sizeof(fw_data); i++) {
	fprintf(f, "%02X", fw_data[i] & 0xff);
	if (!((i + 1) % 16)) {
		fprintf(f, "\n");
	}
    }

    ret = 0;
bail:
    if (f) {
    	fclose(f);
    }
    return ret;
}
#endif // UPDATE_DEBUG

static char* strupr(char *s) {
    unsigned int i;
    for (i = 0; i < strlen(s); i++) {
    	s[i] = toupper(s[i]);
    }
    return s;
}

static unsigned long string_to_hex(char *s) {
    unsigned char counter, end = strlen(s);
    unsigned long hex = 0;
    
    strupr(s);
    for (counter = 0; counter < end ; counter ++) {
	hex <<= 4;       //Another hex digit, so shift all current digits one place
	if (s[counter & 0xff] < 'A') {               // Determine if character is a digit or letter
	    hex += (s[counter & 0xff] - '0');        // Get ASCII digit, find value, add to total 
	}
    	else {
	    hex += (s[counter & 0xff] - 'A' + 0xA);  // Get ASCII letter, find value, add to total 
	}
		
    }
    return (hex);
}

#define MAX_BUF_LEN	16

static unsigned long load_bytes(char* ar, unsigned char bytes) {
    char c[2];    //Text Byte: A pair of ASCII characters representing a hex value
    char buf[MAX_BUF_LEN + 1] = "";
    int offset = 0;

    memset(c, 0, sizeof(c));
    bytes *= 2;									//2 characters for every byte
    if (bytes > MAX_BUF_LEN) {
	return 0;
    }
    for (;bytes > 0; bytes--) {
	c[0] = ar[offset++];
	strcat(buf, c);
    }
    return string_to_hex(buf);

}

#define MAX_RECORD_LEN	64

enum RecType {DataRec, EOFRec, ESAddrRec, SSAddrRec, ELAddrRec, SLAddrRec};

static int Serialize(char* firmware, size_t length) {
    int ret = -1;
    unsigned long record_type = 0xff;
    unsigned int offset = 0;
    
    DBG("Serializing firmware file.\n");
    
    if (firmware == NULL || length == 0) {
    	pp_log("No firmware!\n");
    	goto bail;
    }

    // read each record
    while (record_type != EOFRec) {
	char c[2], count;
	int record_offset = 0;
	int rec_len, address_offset;
	char record_data[MAX_RECORD_LEN + 1] = "";

	memset(c, 0, sizeof(c));

	// read the record
	while(c[0] != ':') {
	    c[0] = firmware[offset++];
	}
	rec_len = load_bytes(&firmware[offset], 1);		//Get length of record
	offset += 2;
	address_offset = load_bytes(&firmware[offset], 2);	//Get addr, 2 bytes long
	offset += 4;
	record_type = load_bytes(&firmware[offset], 1);	//Get type of record
	offset += 2;

	if ((2 * rec_len) > MAX_RECORD_LEN) {
	    pp_log("record too long.\n");
	    goto bail;
	}

	for (count = 2 * rec_len; count >0; count--) {
	    c[0] = firmware[offset++];
	    if (c[0] == ':') {
	    	pp_log("wrong record size!\n");
	    	goto bail;
	    }
	    strcat(record_data, c);
	}

	// ignore the checksum, it will be read before the next record

	// now copy the data
	if (record_type == DataRec) {
	    int i;
	    for (i = 0; i < rec_len; i++) {
	    	int address = address_offset + i;
	    	if (address > MAX_DATA_LEN) {
		    pp_log("data too big for buffer.\n");
		    goto bail;
	    	}
	    	fw_data[address] = (unsigned char)load_bytes(&record_data[record_offset], 1);
	    	record_offset += 2;
	    }
	}
	
	if (offset >= length) {
	    break;
	}
    }

    ret = 0;
    DBG("Firmware successfully serialized.\n");
    
#if UPDATE_DEBUG
    dump_it();
#endif

 bail:
    return ret;
}

#endif // PP_FEAT_MASTERCONSOLE_FW_UPDATE
