/* system includes */
#include <stdio.h>
#include <unistd.h>
#include <arpa/inet.h>
#include <sys/time.h>

/* firmware includes */
#include <pp/base.h>
#include <pp/cfg.h>
#include <pp/um.h>
#include <pp/rfb.h>
#include <pp/kvm.h>
#include <liberic_misc.h>
#include <pp/vsc.h>
#include <liberic_misc.h>
#include <liberic_pthread.h>
#include <pp_kernel_common.h>

/* local includes */
#include "kvm_defaults.h"
#include "kvm_driver.h"
#include "kbd_switch.h"
#include "0801ip.h"
#include "1601ip.h"
#include "0801iptt.h"
#include "1601iptt.h"
#include "ripc.h"
#include "icpmmd.h"
#include "virt_switch.h"
#include "dummy.h"

#include "kvm_internals.h"
#include "debug.h"
/* #include "distribute.h" */
#include "kvm_driver_kx2.h"

using namespace pp;

/* currently active driver */
class KvmDriverBase *kvm_driver;

pp_kvm_encoder_desc_t *encoder_hook = NULL;

int
pp_kvm_init(void)
{
    kvm_defaults_init();

#if defined (LARA_KACY)
    DP_NOTICE("%s(): initializing kx2 switch\n", ___F);
    kvm_driver = new KvmDriverKx2();
#elif defined(PRODUCT_LARA)
    DP_NOTICE("%s(): initializing virtual switch\n", ___F);
    kvm_driver = new KvmVirtSwitch();
#elif defined(PRODUCT_0801IP)
    DP_NOTICE("%s(): initializing 0801IP\n", ___F);
    kvm_driver = new Kvm0801ip();
#elif defined(PRODUCT_1601IP)
    DP_NOTICE("%s(): initializing 1601IP\n", ___F);
    kvm_driver = new Kvm1601ip();
    /* don't insert tests for pp_kvm_switch_port() here since they will fail */
#elif defined(PRODUCT_0801IPTT)
    DP_NOTICE("%s(): initializing 0801IPTT\n", ___F);
    kvm_driver = new Kvm0801iptt();
#elif defined(PRODUCT_1601IPTT)
    DP_NOTICE("%s(): initializing 1601IPTT\n", ___F);
    kvm_driver = new Kvm1601iptt();
#elif defined(PRODUCT_SMARTIPC)
    DP_NOTICE("%s(): initializing SmartIPC\n", ___F);
    kvm_driver = new KvmSmartipc();
#elif defined(PRODUCT_RIPCKIMXN)
    DP_NOTICE("%s(): initializing RIPC\n", ___F);
    kvm_driver = new KvmRipc();
#elif defined(PRODUCT_ICPMMD)
    DP_NOTICE("%s(): initializing ICPMMD\n", ___F);
    kvm_driver = new KvmIcpMmd();
#else
    DP_NOTICE("%s(): initializing dummy switch\n", ___F);
    kvm_driver = new KvmDummySwitch();
#endif

    assert(kvm_driver != NULL);
    return PP_SUC;
}

void
pp_kvm_cleanup(void)
{
    /* call destructor and free memory */
    if (kvm_driver)
		delete (kvm_driver);

	kvm_driver = NULL;
}

int pp_kvm_connect_encoder(pp_kvm_encoder_desc_t* encoder) {
    assert(encoder_hook == NULL);
    
    encoder_hook = encoder;
    return PP_SUC;
}

void
pp_kvm_port_changed(u_char channel, u_char /* old_unit */, u_char /* old_port */,
		    u_char new_unit, u_char new_port, u_char /* control */)
{
    DP_NOTICE("%s(): port changed on channel %hhu to unit/port %hhu/%hu\n", ___F, channel, new_unit, new_port);
    kvm_driver->PortChanged( channel, new_unit, new_port);
}
    
void
pp_kvm_set_local_video_state(u_char channel, int status)
{
    DP_NOTICE("%s(): channel: %hhu, status: %d\n", ___F, channel, status);

    kvm_driver->SetLocalVideoState (channel, status);
}

void
pp_kvm_console_beep(u_char channel UNUSED, u_short pitch UNUSED, u_char time UNUSED)
{
    DP_NOTICE("%s(): channel: %hhu, pitch: %hu, time: %hhu\n", ___F, channel, pitch, time);
    /* no-op at the moment */
}

void
pp_kvm_autoscan(u_char channel UNUSED, int status UNUSED)
{
    DP_NOTICE("%s(): channel: %hhu, status: %d\n", ___F, channel, status);
    /* no-op at the moment */
}

void
pp_kvm_ip_address(uint32_t ip, int ip_valid, uint32_t mask, int mask_valid, uint32_t gw, int gw_valid)
{
    kvm_driver->ChangeIpAddress (ip, ip_valid, mask, mask_valid, gw, gw_valid);
}

void
pp_kvm_led_change(u_char channel UNUSED, u_char scroll_lock UNUSED, u_char num_lock UNUSED, u_char caps_lock UNUSED)
{
    DP_NOTICE("%s(): channel: %hhu, scroll: %hhu, num: %hhu, caps: %hhu\n", ___F, channel, scroll_lock, num_lock, caps_lock);
    /* no-op at the moment */
}

int
pp_kvm_get_unit_port_for_data_link(u_char data_link_id, u_char * unit_p, u_short * port_p)
{
    int ret = PP_ERR;
    int unit, port;

    if (PP_SUCCED(ret = kvm_driver->getUnitAndPortForDataLink(data_link_id, &unit, &port))) {
	*unit_p = (u_char)unit;
	*port_p = (u_char)port;
    }

    return ret;
}

int
pp_kvm_get_unit_port_for_video_link(u_char video_link_id, u_char * unit_p, u_short * port_p)
{
    int ret = PP_ERR;
    int unit, port;

    if (PP_SUCCED(ret = kvm_driver->getUnitAndPortForVideoLink(video_link_id, &unit, &port))) {
	*unit_p = (u_char)unit;
	*port_p = (u_char)port;
	ret = PP_SUC;
    }

    return ret;	
}

int
pp_kvm_get_video_link_for_data_link(u_char data_link, u_char *video_link)
{
    int vl;

    if (PP_FAILED(kvm_driver->getAnyVscVideoLinkForDataLink((int)data_link, &vl))) {
	return PP_ERR;
    }

    *video_link = (u_char)vl;
    return PP_SUC;
}

int
pp_kvm_get_video_link(pp_kvm_link_client_t client, u_char *video_link)
{
    int vl;
    int ret = PP_ERR;

    assert(client);

    vl = ((LinkClient*)client)->GetVideoLinkId();
    if (vl < 0) goto bail;

    *video_link = vl;

    ret = PP_SUC;
bail:
    return ret;
}

int
pp_kvm_get_data_link(pp_kvm_link_client_t client, u_char *data_link)
{
    int vl;
    int ret = PP_ERR;

    assert(client);

    vl = ((LinkClient*)client)->GetDataLinkId();
    if (vl < 0) goto bail;

    *data_link = vl;

    ret = PP_SUC;
bail:
    return ret;
}

u_char
pp_kvm_get_channel_count(void)
{
    return PP_KVM_CHANNEL_COUNT;  
}

u_short
pp_kvm_get_unit_port_count(u_char unit)
{
    return kvm_driver->GetUnitPortCount( unit );
}

void
pp_kvm_set_unit_port_count(u_char unit, u_short count)
{
    kvm_driver->SetUnitPortCount( unit, count );
}

int
pp_kvm_is_unit_present(u_char unit)
{
    return kvm_driver->GetUnitPortCount(unit) ? 1 : 0;
}

/* port switch stuff */

int
pp_kvm_is_host_at_port(u_char unit, u_short port)
{
    return kvm_driver->IsHostAtPort( unit, port );
}

int
pp_kvm_is_port_allowed_for_user(const char * user, u_char unit, u_short port)
{
     return kvm_driver->IsPortAllowedForUser( user, unit, port );
}

int
pp_kvm_switch_allowed(const char * user, u_char unit, u_short port)
{
    return kvm_driver->isSwitchToPortAllowed(user, unit, port);
}

#if defined(PP_FEAT_MASTERCONSOLE_FW_UPDATE) 
int
pp_kvm_update_masterconsole_ip_firmware(char * firmware, size_t len)
{
// TODO: implement me    return kvm_driver->UpdateMasterConsoleFirmware (firmware, len);
return -1;
}
#endif


/********************* kx2 stuff **************************************/

pp_kvm_link_client_t
pp_kvm_new_link_client(eric_session_int_id_t session_id,
	pp_kvm_session_type_t session_type)
{
    return ((pp_kvm_link_client_t)
	kvm_driver->NewLinkClient(session_id, session_type));

}

int
pp_kvm_switch_port(pp_kvm_link_client_t client,
	u_char unit, int target_port, u_char* data_link, u_char* video_link)
{
    assert(client);

    return kvm_driver->Switch((LinkClient*)client,
		unit, target_port, data_link, video_link);
}

int
pp_kvm_set_link_client_links(pp_kvm_link_client_t client,
	u_char data_link, u_char video_link)
{
    assert(client);

    return kvm_driver->setLinkClientLinks((LinkClient*)client,
	    data_link, video_link); 
}

int
pp_kvm_release_link_client(pp_kvm_link_client_t client)
{
    assert(client);

    return kvm_driver->ReleaseLinkClient( (LinkClient*)client );
}

int
pp_kvm_get_unit_and_port(pp_kvm_link_client_t client,
	u_char *unit_out, u_short *port_out)
{
    int ret = PP_ERR;
    int unit, port;

    assert(client);

    if (PP_SUCCED(kvm_driver->getUnitAndPort((LinkClient*)client,
		    &unit, &port))) {
	ret = PP_SUC;
	*unit_out = (u_char)unit;
	*port_out = (u_short)port;
    }

    return ret;
}

int
pp_kvm_get_data_link_id_at_port(int unit, int port, int *data_link_id)
{
    int ret = PP_ERR;

    ret = kvm_driver->getDataLinkIdAtPort(unit, port, data_link_id);

    return ret;
}

int
pp_kvm_targetstatus_changed ( u_short target_port, int cim_state, int cim_type )
{
    return kvm_driver->PortStatusChanged ( target_port, cim_state, cim_type );
}

int pp_kvm_get_client_cnt_on_video_link(int video_link)
{
	return kvm_driver->getClientCntOnVideoLink(video_link);

}

