/* system includes */
#include <stdio.h>
#include <sys/time.h>
#include <pp/base.h>
#include <fcntl.h>
#include <unistd.h>

/* firmware includes */
#include <liberic_misc.h>
#include <liberic_pthread.h>
#include <pp/acl.h>
#include <pp/um.h>
extern "C" {
#include <pp/termios.h>
}

/* local includes */
#include "xx01ip.h"
#include "xx01ip_int.h"
#include "debug.h"
#include "master_console_update.h"

#define KVM_FW_UPDATE_TIMEOUT       5       // seconds

/* Implementation of KvmXx01ip. */

using namespace pp;


KvmXx01ip::KvmXx01ip()
{
    pthread_cond_t init_cond = PTHREAD_COND_INITIALIZER;
    pthread_mutex_t init_mtx = PTHREAD_ERRORCHECK_MUTEX_INITIALIZER_NP;

    DP_VERBOSE("%s called\n", ___F);


    /* assign initializer structs */
    _portChangeCond = init_cond;
    _readerSleepingCond = init_cond;
    _portChangeMtx = init_mtx;

/* FIXME: error: returning a value from a constructor
    if (InitReader() != 0) goto error;

    return this;

error:
    return NULL;
*/
InitReader();
}

KvmXx01ip::~KvmXx01ip()
{
    /* destroy non-inherited members */
    DestroyReader ();
}


int
KvmXx01ip::InitReader ()
{
    unsigned char cntl;
    int *reader_fd = &this->_readerFd;

    DP_VERBOSE("%s called\n", ___F);
    
    /* open serial connection to switch */
    if ((*reader_fd = open("/dev/ttyS1", O_RDWR | O_NDELAY)) < 0) {
	return -1;
    }

    cntl = fcntl(*reader_fd, F_GETFL, 0);
    fcntl(*reader_fd, F_SETFL, cntl & ~O_NDELAY);

    if (pp_base_set_tty_params(*reader_fd,
#if defined(PP_FEAT_MASTERCONSOLE_FW_UPDATE)
        9600,
#else // PP_FEAT_MASTERCONSOLE_FW_UPDATE
    	1200,
#endif // PP_FEAT_MASTERCONSOLE_FW_UPDATE
    	"", 8, 0, 0, 0) == -1) {
	
	close(*reader_fd);
	return -1;
    }
    
    pthread_mutex_init(&this->_readerSleepingMtx, NULL);    

    DP_VERBOSE("%s: creating xx01ip reader thread.\n", ___F);
    this->_readerThreadRun = 1;
    if (eric_pthread_create(&this->_readerThread, 0, 64 * 1024,
			    KvmXx01ip::ReaderThreadFunc, (void*) this)) {
	DP_ERROR("%s: error: Cannot create xx01ip reader thread.\n", ___F);
	return -1;
    }
    
    return 0;
}

void
KvmXx01ip::DestroyReader ()
{
    /* deinit reader thread*/
    this->_readerThreadRun = 0;
    pthread_join(this->_readerThread, NULL);
    close(this->_readerFd);
}

void *
KvmXx01ip::ReaderThreadFunc (void *arg)
{
    KvmXx01ip *_this = (KvmXx01ip*) arg;
    u_char raw, buffer[READER_BUFFER_SIZE];
    struct timeval to;
    fd_set set;
    int ret;
    int *reader_fd = &_this->_readerFd;
    int current_kvm_unit;
    int current_kvm_port, port;
    
    MUTEX_LOCK(&_this->_readerSleepingMtx);
    _this->_readerSleeping = 0;
    MUTEX_UNLOCK(&_this->_readerSleepingMtx);
    
    while (_this->_readerThreadRun) {
	int sleep_time = READER_SLEEP_TIME;
	FD_ZERO(&set);
	FD_SET(*reader_fd, &set);
	to.tv_sec = 1;
	to.tv_usec = 0;
	/* get port on channel 0 -- the only channel in xx01ip */
	_this->getUnitAndPort(0 /* channel */, &current_kvm_unit, &current_kvm_port);

	if (select((*reader_fd) + 1, &set, NULL, NULL, &to) < 0) {
	    pp_log("1601ip reader_thread_func: select failed\n");
	    goto next_run;
	}

	if (!FD_ISSET(*reader_fd, &set)) {
	    errno = ETIMEDOUT;
	    pp_log("1601ip reader_thread_func: select timeout\n");
	    goto next_run;
	}

	if ((ret = read(*reader_fd, buffer, READER_BUFFER_SIZE)) == -1) {
	    pp_log("1601ip reader_thread_func: read failed\n");
	    goto next_run;
	}

	if (ret < 2) {
	    sleep_time = READER_SLEEP_LONG_TIME;
	    goto next_run;
	}	
	raw = buffer[ret-1];
	if ((raw & 0xF0) != 0xC0 && (raw & 0xF0) != 0xD0) {
	    raw = buffer[ret-2];
	}

	if ((raw & 0xF0) == 0xC0 || (raw & 0xF0) == 0xD0) {
	    port = raw & 0x0F;
	    if (port < _this->GetUnitPortCount( 0 )) {
		if (current_kvm_port != port) {
		    DP_VERBOSE("%s: Got new Port %hu (raw %02x), old port %hu\n", ___F, port, raw, current_kvm_port);

		    /* tell driver that the port changed */
		    _this->PortChanged ( 0 /*chan*/, 0 /*unit*/, port );
		}
	    } else {
		DP_ERROR("%s: New Port %hu (raw %02x) is out of range\n", ___F, port, raw);
	    }
	}
 next_run:
	// wakeup the firmware updater if necessary
	MUTEX_LOCK(&_this->_readerSleepingMtx);
	_this->_readerSleeping = 1;
	pthread_cond_signal(&_this->_readerSleepingCond);
	MUTEX_UNLOCK(&_this->_readerSleepingMtx);

	do {
	    usleep(sleep_time);
	} while (_this->_FwUpdateInProgress);

	MUTEX_LOCK(&_this->_readerSleepingMtx);
	_this->_readerSleeping = 0;
	pthread_cond_signal(&_this->_readerSleepingCond);
	MUTEX_UNLOCK(&_this->_readerSleepingMtx);
    }

    return NULL;
}

int
KvmXx01ip::Switch(LinkClient* client, u_char unit, u_short target_port,
	u_char* data_link, u_char* video_link)
{
    const char* port_chars = "12345678ABCDEFGH";
    /* TODO: replace _switchPortMtx by global m_mtx */
    pthread_mutex_t *mtx = &_switchPortMtx;
    int error;
    vector_t *key_vec = NULL;
    char key_seq[KVM_MAX_KEY_SEQ_SIZE];
    
    if (unit != 0) {
	DP_ERROR("%s: cannot switch to unit %d, only unit 0 supported\n", ___F, unit);
	return PP_ERR;
    }

    DP_NOTICE("%s: switching to port %d\n", ___F, target_port);

    MUTEX_LOCK(mtx);

    /* _readerThread will set this to CHANGE_STATE_DONE when port changes */
    this->_portChangeState = CHANGE_STATE_NONE;

    /* generate symbolic keyseq */
    snprintf(key_seq, sizeof(key_seq), "CTRL-CTRL-*%c", port_chars[target_port]);

    /* send the keys out */
    error = SendKeySeq (0 /* channel */, key_seq);
    if (error) goto error;

    /* wait until we receive the new port nr from KVM */
    WaitForPortchange();
    	
    free(key_vec);
    MUTEX_UNLOCK(mtx);
    return PP_SUC;
    
error:
    free(key_vec);
    MUTEX_UNLOCK(mtx);
    return PP_ERR;
}

void
KvmXx01ip::WaitForPortchange()
{
    struct timeval now;
    struct timespec to;
    int ret = 0;
    pthread_mutex_t *mtx = &this->_portChangeMtx;
    
    MUTEX_LOCK(mtx);
    gettimeofday(&now, NULL);
    to.tv_sec  = now.tv_sec + 1;
    to.tv_nsec = now.tv_usec * 1000;
    while (this->_portChangeState != CHANGE_STATE_DONE
	   && ret != ETIMEDOUT) {
	ret = pthread_cond_timedwait(&_portChangeCond, mtx, &to);
    }
    MUTEX_UNLOCK(mtx);

#if DEBUGLEVEL >= D_ERROR
    if (ret == ETIMEDOUT) {
	DP_ERROR("%s: timeout waiting for switch to ack port change\n", ___F);
    } else {
	DP_NOTICE("%s: port change ack by switch\n", ___F);
    }
#endif
}

void
KvmXx01ip::PortChanged (int channel, int newunit, int newport)
{
    pthread_mutex_t *mtx = &this->_portChangeMtx;

    /* call superclass implementation */
    KvmDriver::PortChanged(channel, newunit, newport);

    /* wakeup kvm switcher if necessary */
    MUTEX_LOCK(mtx);
    this->_portChangeState = CHANGE_STATE_DONE;
    pthread_cond_signal(&_portChangeCond);
    MUTEX_UNLOCK(mtx);
}

/* TODO untested
 * - mostly copied from pp_kvm_update_masterconsole_ip_firmware (kvm_1601ip.c)
 *   kim-stable-libkvm-review-branch-point-20050112
 */
#if defined(PP_FEAT_MASTERCONSOLE_FW_UPDATE)
int
KvmXx01ip::UpdateMasterConsoleFirmware( char* firmware, size_t length)
{
    int ret = 0;
    struct timeval now;
    struct timespec timeout;
    pthread_mutex_t *sleeping_mtx = &this->_readerSleepingMtx;
    pthread_cond_t *sleeping_cond = &this->_readerSleepingCond;
    
    gettimeofday(&now, NULL);
    timeout.tv_sec = now.tv_sec + KVM_FW_UPDATE_TIMEOUT;
    timeout.tv_nsec = now.tv_usec * 1000;
    
    // wait for the reader thread to pause
    MUTEX_LOCK(sleeping_mtx);
    
    if (!this->_readerSleeping) {
    	int retcode;
    	
    	pp_log("Waiting for the reader to pause...\n");
    	retcode = pthread_cond_timedwait(sleeping_cond,
                                         sleeping_mtx,
                                         &timeout);
    	pp_log("...done\n");
    
    	if (retcode == ETIMEDOUT) {
    	    pp_log("timeout waiting for KVM reader thread.\n");
    	    MUTEX_UNLOCK(sleeping_mtx);
    	    ret = -1;
    	    goto bail;
    	}
    }
    
    this->_FwUpdateInProgress = 1;
    
    MUTEX_UNLOCK(sleeping_mtx);
    
    // now we can do the update
    ret = update_master_console(this->_readerFd, firmware, length);
    
    // we're finished
 bail:
    this->_FwUpdateInProgress = 0;
    
    return ret;
}
#endif

int KvmXx01ipIsPortAllowedForUser ( const char* user, int /* unit */, int kvm_port)
{
    char acl[32];

    /* TODO: add support for UNIT */
    snprintf(acl, 32, "access_port_%d", kvm_port);

    return PP_SUC == pp_um_user_has_permission(user, acl,
                                               pp_acl_raasip_control_str) ||
           PP_SUC == pp_um_user_has_permission(user, acl,
                                               pp_acl_raasip_view_str);
}
