/**
 * LDAP Caching: routines for caching values coming from an LDAP query
 *
 * Author: Thomas Breitfeld <thomas@peppercon.de>
 * Copyright 2002 Peppercon AG
 */
#include "ldap_cache.h"
#include <stdlib.h>

static int   hash_get(pp_ldap_cache_t* this, const char* key, void** value);
static void* hash_set(pp_ldap_cache_t* this, const char* key, void* value);
static void  hash_clear(pp_ldap_cache_t* this);
static void  hash_cleanup(pp_ldap_cache_t* this);
static int   single_get(pp_ldap_cache_t* this, const char* key, void** value);
static void* single_set(pp_ldap_cache_t* this, const char* key, void* value);
static void  single_clear(pp_ldap_cache_t* this);

void pp_ldap_cache_init(pp_ldap_cache_t* cache, pp_ldap_cache_mode_t cachenull,
			int cachenull_allowed_error,
	    int   (*fetch)(void** val, pp_ldap_t* ld, const char* key),
	    int   (*get)(pp_ldap_cache_t* this, const char* key, void** value),
	    void* (*set)(pp_ldap_cache_t* this, const char* key, void* value),
	    void  (*clear)(pp_ldap_cache_t* this),
	    void  (*cleanup)(pp_ldap_cache_t* this)) {
    pthread_mutexattr_t err_check_mtx_attr;
    int mutex_error;
    // create error checking mutex
    mutex_error = pthread_mutexattr_init(&err_check_mtx_attr);
    assert(mutex_error == 0);
    mutex_error = pthread_mutexattr_settype(&err_check_mtx_attr,
					    PTHREAD_MUTEX_ERRORCHECK_NP);
    assert(mutex_error == 0);
    mutex_error = pthread_mutex_init(&cache->mtx, &err_check_mtx_attr);
    // succeeds always and ignores attrs anyway
    pthread_cond_init(&cache->cond, NULL);
    cache->fetching = 0;
    cache->cachenull = cachenull;
    cache->cachenull_allowed_error = cachenull_allowed_error;
    cache->ldfetch = fetch;
    cache->get = get;
    cache->set = set;
    cache->clear = clear;
    cache->cleanup = cleanup;
}

void pp_ldap_cache_cleanup(pp_ldap_cache_t* cache) {
    cache->clear(cache);
    if (cache->cleanup) cache->cleanup(cache);
    pthread_cond_destroy(&cache->cond);
    pthread_mutex_destroy(&cache->mtx);
}

int pp_ldap_hash_cache_init(pp_ldap_hash_cache_t* cache,
			    pp_ldap_cache_mode_t cachenull,
			    int cachenull_allowed_error,
		            int (*fetch)(void** val, pp_ldap_t* ld, const char* key),
			    void* (*duplicate)(void* value),
			    void  (*vfree)(void *)) {
    pp_ldap_cache_init(&cache->base, cachenull, cachenull_allowed_error,
		       fetch, hash_get, hash_set, hash_clear, hash_cleanup);
    if (NULL == (cache->data = pp_hash_create(10)))
	return -1;
    cache->vfree = vfree;
    cache->duplicate = duplicate;
    return 0;
}

int pp_ldap_single_cache_init(pp_ldap_single_cache_t* cache,
			      pp_ldap_cache_mode_t cachenull,
			      int cachenull_allowed_error,
  		              int (*fetch)(void** val, pp_ldap_t* ld, const char* key),
			      void* (*duplicate)(void* value),
			      void  (*vfree)(void *)) {
    pp_ldap_cache_init(&cache->base, cachenull, cachenull_allowed_error,
		       fetch, single_get, single_set, single_clear, NULL);
    cache->data = NULL;
    cache->data_valid = 0;
    cache->vfree = vfree;
    cache->duplicate = duplicate;
    return 0;
}

/*
 * checks whether value is in cache and returns it, if yes
 * If no, we try to fetch it using our fetch function.
 * If this fails too, NULL is returned.
 * NOTE:
 *   - NULL can't be put in as cache value, obviously
 *   - fetching will be serialized among all threads,
 *     that ask for values, that are not in cache
 *   - a fetch does not block another lookup, as long
 *     as the value is in the cache, if that lookup requires
 *     a fetch too, that fetch will be queued
 *   - cache_clear invalidates a running fetch, so a fetch
 *     maybe carried out multiple times
 *   - repetitively looking up values that are neither in cache
 *     nor fetchable, is highly inefficient, as it will always
 *     lead to another fetch, except the cachenull flag will
 *     be set to PP_LDAP_CACHE_DO_NULL during cache creation
 * returns PP_SUC only, if value is valid
 */
int pp_ldap_cache_get(void** v, pp_ldap_cache_t* cache, pp_ldap_t* ld, 
                      const char* key) {
    int ret = PP_SUC;
    void *val;
    MUTEX_LOCK(&cache->mtx);
    
    // if value for key is not in cache
    if(!cache->get(cache, key, v)) {

	// wait until nobody else is fetching data, or found
	while(cache->fetching) {
	    pthread_cond_wait(&cache->cond, &cache->mtx);
	    // somebody else was fetching data,
	    // maybe he's got what we are looking for too...
	    if(cache->get(cache, key, v)) {
		MUTEX_UNLOCK(&cache->mtx);
                return PP_SUC;
	    }
	}
	
	// still no value, we need to fetch
	cache->fetching = 1;
	do {
            val = NULL;
	    cache->fetchvalid = 1;
	    // unlock, so others may succeed during long lasting fetch
	    MUTEX_UNLOCK(&cache->mtx);
	    ret = cache->ldfetch(&val, ld, key);
	    MUTEX_LOCK(&cache->mtx);
	    // if we got something, check whether its valid
	    if(ret != PP_ERR || 
               (cache->cachenull == PP_LDAP_CACHE_DO_NULL &&
		errno == cache->cachenull_allowed_error)) {
		// somebody might have invalidated our fetch, i.e. clear
		if(cache->fetchvalid) {
		    // ignore errors, cache will recover from it next time
		    val = cache->set(cache, key, val);
                    ret = PP_SUC;
		    break;
		}
	    } else { // if v == NULL && !cachenull just return
		break;
	    }
	} while(1);
	// wakeup others, who may wait to fetch data
	cache->fetching = 0;
	pthread_cond_signal(&cache->cond);
        *v = val;
    }
    MUTEX_UNLOCK(&cache->mtx);
    return ret;
}

void pp_ldap_cache_clear(pp_ldap_cache_t* cache) {
    MUTEX_LOCK(&cache->mtx);
    cache->clear(cache);
    cache->fetchvalid = 0;
    MUTEX_UNLOCK(&cache->mtx);
}

static void hash_cleanup(pp_ldap_cache_t* this) {
    pp_ldap_hash_cache_t* cache = (pp_ldap_hash_cache_t*)this;
    pp_hash_delete(cache->data);
}

static int hash_get(pp_ldap_cache_t* this, const char* key, void** value) {
    pp_ldap_hash_cache_t* cache = (pp_ldap_hash_cache_t*)this;
    void* v;
    int has_value = pp_hash_get_entry2(cache->data, key, &v);
    if (has_value) {
	*value = v == NULL ? v : cache->duplicate(v);
    }
    return has_value;
}

static void* hash_set(pp_ldap_cache_t* this, const char* key, void* value) {
    void* v;
    pp_ldap_hash_cache_t* cache = (pp_ldap_hash_cache_t*)this;
    v = value == NULL ? NULL : cache->duplicate(value);
    if(0 > pp_hash_set_entry(cache->data, key, value, cache->vfree)) {
	cache->vfree(value);
    }
    return v;
}

static void hash_clear(pp_ldap_cache_t* this) {
    pp_ldap_hash_cache_t* cache = (pp_ldap_hash_cache_t*)this;
    pp_hash_clear(cache->data);
}
    
static int single_get(pp_ldap_cache_t* this, UNUSED const char* key,
		      void** value) {
    pp_ldap_single_cache_t* cache = (pp_ldap_single_cache_t*)this;
    int has_value = cache->data_valid;
    if (has_value) {
	*value = cache->data ? cache->duplicate(cache->data) : NULL;
    }
    return has_value;
}

static void* single_set(pp_ldap_cache_t* this, UNUSED const char* key,
		       void* value) {
    pp_ldap_single_cache_t* cache = (pp_ldap_single_cache_t*)this;
    if(cache->data_valid && cache->data != NULL) {
	cache->vfree(cache->data);
    }
    cache->data = value;
    cache->data_valid = 1;
    return value ? cache->duplicate(value) : value;
}

static void single_clear(pp_ldap_cache_t* this) {
    pp_ldap_single_cache_t* cache = (pp_ldap_single_cache_t*)this;
    if(cache->data != NULL) {
	cache->vfree(cache->data);
    }
    cache->data = NULL;
    cache->data_valid = 0;
}
