/**
 * LDAP Caching: routines for caching values coming from an LDAP query
 *
 * Author: Thomas Breitfeld <thomas@peppercon.de>
 * Copyright 2002 Peppercon AG
 */
#ifndef __PP_LDAP_CACHE_H__
#define __PP_LDAP_CACHE_H__

#include <pthread.h>
#include <pp/base.h>

#include "ldap_intern.h"

typedef enum  {
    PP_LDAP_CACHE_DO_NULL,
    PP_LDAP_CACHE_DONT_NULL,
} pp_ldap_cache_mode_t;

typedef struct pp_ldap_cache_s {
    pthread_mutex_t      mtx;
    pthread_cond_t       cond;
    int                  fetching;
    int                  fetchvalid;
    pp_ldap_cache_mode_t cachenull;
    int                  cachenull_allowed_error;
    int   (*ldfetch)(void**, pp_ldap_t* ld, const char* key);
    int   (*get)(struct pp_ldap_cache_s* this, const char* key, void** value);
    void* (*set)(struct pp_ldap_cache_s* this, const char* key, void* value);
    void  (*clear)(struct pp_ldap_cache_s* this);
    void  (*cleanup)(struct pp_ldap_cache_s* this);
} pp_ldap_cache_t;

typedef struct {
    pp_ldap_cache_t base;
    pp_hash_t*    data;
    void            (*vfree)(void* value);
    void*           (*duplicate)(void* value);
} pp_ldap_hash_cache_t;

typedef struct {
    pp_ldap_cache_t base;
    void*           data;
    int             data_valid;
    void            (*vfree)(void* value);
    void*           (*duplicate)(void* value);
} pp_ldap_single_cache_t;

void pp_ldap_cache_init(pp_ldap_cache_t* cache,	pp_ldap_cache_mode_t cachenull,
			int cachenull_allowed_error,
	   int (*fetch)(void **val, pp_ldap_t* ld, const char* key),
           int (*get)(pp_ldap_cache_t* this, const char* key, void** value),
	   void* (*set)(pp_ldap_cache_t* this, const char* key, void* value),
	   void  (*clear)(pp_ldap_cache_t* this),
	   void  (*cleanup)(pp_ldap_cache_t* this));

int pp_ldap_hash_cache_init(pp_ldap_hash_cache_t* cache,
			    pp_ldap_cache_mode_t cachenull,
			    int cachenull_allowed_error,
	           int (*fetch)(void **val, pp_ldap_t* ld, const char* key),
			    void* (*duplicate)(void* value),
			    void  (*vfree)(void*));

int pp_ldap_single_cache_init(pp_ldap_single_cache_t* cache,
			      pp_ldap_cache_mode_t cachenull,
			      int cachenull_allowed_error,
		  int (*fetch)(void **val, pp_ldap_t* ld, const char* key),
			      void* (*duplicate)(void* value),
			      void  (*vfree)(void* value));
	
int pp_ldap_cache_get(void **val, pp_ldap_cache_t* cache, pp_ldap_t* ld, const char* key);

void pp_ldap_cache_clear(pp_ldap_cache_t* cache);

void pp_ldap_cache_cleanup(pp_ldap_cache_t* cache);

#endif
