#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <pp/base.h>
#include <pp/serial.h>
#include <termios.h>
#include <pp/powerswitch.h>

#include "debug.h"
#include "power_switch.h"

int
ps_get_fd_by_port_id(u_int port_id)
{
    if (port_id == PP_POWER_PORT_ID_SERIAL_1) {
	return pp_serial_port_get(0);
    } else if (port_id == PP_POWER_PORT_ID_SERIAL_2) {
	return pp_serial_port_get(1);
    }
    return -1;
}

/*
 * buf_size is the size of the whole buffer and *buf_len_p will be filled with
 * the byte count actually used in the buffer
 */

int
ps_read_response(int fd, char chr, const char* text, u_long sec,
		 u_long usec, char * buf, size_t buf_size,
		 size_t * buf_len_p, read_response_mode_t mode)
{
    int chr_to_read = 0;
    ssize_t n_read;
    struct timeval to;
    size_t buf_pos = 0; /* points to first free buf slot */
    fd_set set;
    int ret = PP_ERR;

    /* make sure strings are zero-terminated */
    memset(buf, '\0', buf_size);

    while (1) {
	/* check buf limits, keep one '\0' at the end */
	if (buf_pos >= buf_size) {
	    D(D_ERROR, "%s(): max buf size reached\n", ___F);
	    break;
	}
	FD_ZERO(&set);
	FD_SET(fd, &set);
	to.tv_sec = sec;
	to.tv_usec= usec;
	if (select(fd+1, &set, NULL, NULL, &to) == -1) {
	    if (errno == EINTR) continue;
	    pp_log_err("%s(): select() failed", ___F);
	    goto bail;
	}
	if (!FD_ISSET(fd, &set)) {		// timeout
	    D(D_ERROR, "%s(): select timeout\n", ___F);
	    errno = ETIMEDOUT;
	    /* timeout is error, unless TTY_READ_MAX_CHR - read until timeout */
	    if (mode == TTY_READ_MAX_CHR) break;
	    goto bail;
	}
	/* select succeeded -> try to read */
	if ((n_read = read(fd, &buf[buf_pos], 1)) == -1) {
	    if (EINTR == errno) continue;	 
	    pp_log_err("%s(): read", ___F);
	    return PP_ERR;
	} else if (n_read == 1) {
	    /* process the char just read */
	    buf_pos++;
	    if (mode == TTY_READ_STRING) {            	
		/* if TEXT is seen, return until there */
		if (strstr(buf, text) != NULL) break;
	    } else if (mode == TTY_READ_STOP_CHR) {
		/* if stop char, return until there */
		if (buf[buf_pos-1] == chr) break;
	    } else if (mode == TTY_READ_IPWR_MSG) {
		if ( buf_pos-1 == 2 ) {
		    /* length info at buf[2] */
		    chr_to_read = buf[buf_pos-1] + 2;
		} else if ( buf_pos-1 > 2 ) {
		    chr_to_read--;
		    if ( chr_to_read <= 0 ) break;
		}
	    }
	} else if (n_read == 0) {
	    continue;
	} else {
	    goto bail; /* should not happen */
	}
    }
    ret = PP_SUC;
bail:
    if (buf_len_p) *buf_len_p = buf_pos;
    return ret;
}

char*
ps_get_pswitch_name(const char* ps, int obj)
{
    char * name;
    
    assert(ps);

    pp_cfg_get(&name, "ps.type.%s.name_%s", ps, (obj == PWR_OBJ_SWITCH_SHORT_NAME) ? "short" : "long");

    return name;
}
