#include <stdlib.h>
#include <fcntl.h>
#include <stdio.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <liberic_misc.h>
#include <pp/um.h>
#include <pp/base.h>
#include <pp/powerswitch.h>
#include <pp/termios.h>

#include "debug.h"
#include "power_switch.h"

static int init(power_switch_t *this, ps_data_t * ps_data);
static int login(power_switch_t *this, ps_data_t * ps_data);
static int logout(power_switch_t *this, ps_data_t * ps_data);
static int get_count(power_switch_t *this, ps_data_t * ps_data, int device_id, int object);
static char* get_name(power_switch_t *this, ps_data_t * ps_data, int idx, int device_id, int object);
static int power_switch(power_switch_t *this, ps_data_t * ps_data, int port, int device, u_char on, int seq);
static int do_reset(power_switch_t *this, ps_data_t * ps_data, int port, int device);
static int get_state(power_switch_t *this, ps_data_t * ps_data, int port, int device);

/* -------------------- exported functions ---------------------------- */

int
power_switch_fjsms_init(power_switch_t *this)
{
    this->init		= init;
    this->login		= login;
    this->logout	= logout;
    this->get_count	= get_count;
    this->get_name	= get_name;
    this->power_switch	= power_switch;
    this->do_reset	= do_reset;
    this->get_state	= get_state;

    this->caps = POWER_CAP_SWITCH | POWER_CAP_RESET | POWER_CAP_GET_STATE;
    return PP_SUC;
}

void
power_switch_fjsms_cleanup(void* this)
{
    power_switch_t *entry = (power_switch_t*) this;
    free(entry);
}

static int
init(power_switch_t *this UNUSED, ps_data_t * ps_data)
{
    const char * lspeed_var_name = NULL;
    u_int tty_spd;
    int ret = PP_ERR;

    if (ps_data->port_id == PP_POWER_PORT_ID_SERIAL_1) {
	lspeed_var_name = "ps.serial[0].fjsms.speed";
	ps_data->fjsms.user_var_name = "ps.serial[0].fjsms.user";
	ps_data->fjsms.pass_var_name = "ps.serial[0].fjsms.pass";
    } else if (ps_data->port_id == PP_POWER_PORT_ID_SERIAL_2) {
	lspeed_var_name = "ps.serial[1].fjsms.speed";
	ps_data->fjsms.user_var_name = "ps.serial[1].fjsms.user";
	ps_data->fjsms.pass_var_name = "ps.serial[1].fjsms.pass";
    } else {
	ps_data->fjsms.user_var_name = NULL;
	ps_data->fjsms.pass_var_name = NULL;
    }

    if (lspeed_var_name == NULL
	|| PP_FAILED(pp_cfg_get_uint_nodflt(&tty_spd, lspeed_var_name))) {
	goto error;
    }

    if ((ps_data->fjsms.fd = ps_get_fd_by_port_id(ps_data->port_id)) < 0) {
	goto error;
    }

    if (PP_FAILED(pp_base_set_tty_params(ps_data->fjsms.fd, tty_spd, "", 8, 0, 0, 0))) {
	goto error;
    }

    ret = PP_SUC;

 error:
    return ret;
}

static int
login(power_switch_t *this UNUSED, ps_data_t * ps_data)
{
    ps_fjsms_t * ps_fjsms = &ps_data->fjsms;
    char rsp_buf[2048];
    char* user = NULL;
    char* pass = NULL;
    ssize_t cmd_len;
    int ret = PP_ERR;

    if (!ps_fjsms->user_var_name
	|| PP_FAILED(pp_cfg_get_nodflt(&user, ps_fjsms->user_var_name))) {
	goto error;
    }
    if (!ps_fjsms->pass_var_name
	|| PP_FAILED(pp_cfg_get_nodflt(&pass, ps_fjsms->pass_var_name))) {
	goto error;
    }

    if (write(ps_fjsms->fd, "\r", 1) != 1) return PP_ERR;
    if (PP_FAILED(ps_read_response(ps_fjsms->fd, 0, "Please enter user name : ", 2, 0,
				   rsp_buf, sizeof(rsp_buf), NULL, TTY_READ_STRING))) {
	D(D_ERROR, "FJSMS: %s(): waiting for password prompt failed\n", ___F);
	goto error;
    }

    cmd_len = strlen(user);
    if (write(ps_fjsms->fd, user, cmd_len) != cmd_len
	|| write(ps_fjsms->fd, "\r", 1) != 1) {
	D(D_ERROR, "FJSMS: %s(): sending username failed\n", ___F);
	goto error;
    }
    if (PP_FAILED(ps_read_response(ps_fjsms->fd, 0, "Please enter pass phrase : ", 2, 0,
				   rsp_buf, sizeof(rsp_buf), NULL, TTY_READ_STRING ))) {
	D(D_ERROR, "FJSMS: %s(): waiting for password prompt failed\n", ___F);
	goto error;
    }
    cmd_len = strlen(pass);
    if (write(ps_fjsms->fd, pass, cmd_len) != cmd_len
	|| write(ps_fjsms->fd, "\r", 1) != 1) {
	D(D_ERROR, "FJSMS: %s(): sending password failed\n", ___F);
	goto error;
    }
 again:
    if (PP_FAILED(ps_read_response(ps_fjsms->fd, 0, "Enter selection or (0) to quit: ", 2, 0,
				   rsp_buf, sizeof(rsp_buf), NULL, TTY_READ_STRING))) {
	if (strstr(rsp_buf, "Enter new pass phrase : ")) {
	    /* switch askes to change default password - set it to the same */
	    cmd_len = strlen(pass);
	    if (write(ps_fjsms->fd, pass, cmd_len) != cmd_len
		|| write(ps_fjsms->fd, "\r", 1) != 1) {
		D(D_ERROR, "FJSMS: %s(): sending password failed\n", ___F);
		goto error;
	    }
	    if (PP_FAILED(ps_read_response(ps_fjsms->fd, 0, "Repeat new pass phrase: ", 2, 0,
					   rsp_buf, sizeof(rsp_buf), NULL, TTY_READ_STRING))) {
		D(D_ERROR, "FJSMS: %s(): waiting for repeated password prompt failed\n", ___F);
		goto error;
	    }
	    cmd_len = strlen(pass);
	    if (write(ps_fjsms->fd, pass, cmd_len) != cmd_len
		|| write(ps_fjsms->fd, "\r", 1) != 1) {
		D(D_ERROR, "FJSMS: %s(): sending repeated password failed\n", ___F);
		goto error;
	    }
	    goto again;
	}
	D(D_ERROR, "FJSMS: %s(): waiting for prompt failed\n", ___F);
	goto error;
    }
    if (strstr(rsp_buf, "(3) Power Management...")) {
	if (write(ps_fjsms->fd, "3", 1) != 1) {
	    D(D_ERROR, "FJSMS: %s(): sending command to change submenu failed\n", ___F);
	    goto error;
	}
	if (PP_FAILED(ps_read_response(ps_fjsms->fd, 0, "Enter selection or (0) to quit: ", 2, 0,
				       rsp_buf, sizeof(rsp_buf), NULL, TTY_READ_STRING))) {
	    D(D_ERROR, "FJSMS: %s(): waiting for prompt failed\n", ___F);
	    goto error;
	}
	if (!strstr(rsp_buf, "Power Cycle")) {
	    D(D_ERROR, "FJSMS: %s(): could change into power management sub menu\n", ___F);
	    goto error;
	}
    } else {
	D(D_ERROR, "FJSMS: %s(): could change into power management sub menu\n", ___F);
	goto error;
    }

    ret = PP_SUC;

 error:
    free(user);
    free(pass);
    return ret;
}

static int
logout(power_switch_t *this UNUSED, ps_data_t * ps_data)
{
    ps_fjsms_t * ps_fjsms = &ps_data->fjsms;
    const char * cmd_buf;
    char rsp_buf[2048];
    int ret = PP_SUC;

    cmd_buf = "\r";
    while (1) {
	ssize_t cmd_len = strlen(cmd_buf);
	if (write(ps_fjsms->fd, cmd_buf, cmd_len) != cmd_len) {
	    D(D_ERROR, "FJSMS: %s(): could not logout.\n", ___F);
	    return PP_ERR;
	}
	if (PP_SUCCED(ret = ps_read_response(ps_fjsms->fd, 0, "Enter selection or (0) to quit: ", 2, 0,
					     rsp_buf, sizeof(rsp_buf), NULL, TTY_READ_STRING))) {
	    cmd_buf = "0";
	} else if (strstr(rsp_buf, "Please enter either 'yes' or 'no': ")) {
	    cmd_buf = "no";
	} else if (strstr(rsp_buf, "Press any key to continue ")) {
	    cmd_buf = "\r";
	} else {
	    break;
	}
    }

    return ret;
}

static int
get_count(power_switch_t *this UNUSED, ps_data_t * ps_data UNUSED,
	  int device_id UNUSED, int object)
{
    switch (object) {
      case PWR_OBJ_PORT: /* fall through */
      case PWR_OBJ_DEVICE: return 1;
      default: return 0;  
    }
}

static char *
get_name(power_switch_t *this UNUSED, ps_data_t * ps_data UNUSED,
	 int idx UNUSED, int device_id UNUSED, int object)
{
    switch (object) {
      case PWR_OBJ_PORT:
      case PWR_OBJ_SWITCH_SHORT_NAME:
      case PWR_OBJ_SWITCH_LONG_NAME:
	  return ps_get_pswitch_name(POWER_SWITCH_FJSMS_NAME, object);
      case PWR_OBJ_SWITCH_ID:
	  return strdup(POWER_SWITCH_FJSMS_NAME);
      default: return NULL;
    }
}

static int
power_switch(power_switch_t *this UNUSED, ps_data_t * ps_data,
	     int port, int device UNUSED, u_char on, int seq UNUSED)
{
    int state = get_state(this, ps_data, port, device);
    char rsp_buf[2048];
    int ret = PP_ERR;

    if (state == -1) {
	goto error;
    } else if (state == 0 && on) {
	if (write(ps_data->fjsms.fd, "4", 1) != 1) {
	    D(D_ERROR, "FJSMS: %s(): sending 'power on' command failed\n", ___F);
	    goto error;
	}
	if (PP_FAILED(ps_read_response(ps_data->fjsms.fd, 0, "Do you really want to power on (yes/no)? ", 2, 0,
				       rsp_buf, sizeof(rsp_buf), NULL, TTY_READ_STRING))) {
	    D(D_ERROR, "FJSMS: %s(): waiting for prompt failed\n", ___F);
	    goto error;
	}
	if (write(ps_data->fjsms.fd, "yes\r", 4) != 4) {
	    D(D_ERROR, "FJSMS: %s(): sending confirmation failed\n", ___F);
	    goto error;
	}
	if (PP_FAILED(ps_read_response(ps_data->fjsms.fd, 0, "Press any key to continue ", 5, 0,
				       rsp_buf, sizeof(rsp_buf), NULL, TTY_READ_STRING))) {
	    D(D_ERROR, "FJSMS: %s(): waiting for prompt failed\n", ___F);
	    goto error;
	}
	pp_propchange_enqueue(PP_PROP_PWR_ON, 0);
    } else if (state == 1 && !on) {
	if (write(ps_data->fjsms.fd, "1", 1) != 1) {
	    D(D_ERROR, "FJSMS: %s(): sending 'power off' command failed\n", ___F);
	    goto error;
	}
	if (PP_FAILED(ps_read_response(ps_data->fjsms.fd, 0, "Do you really want to power off (yes/no)? ", 2, 0,
				       rsp_buf, sizeof(rsp_buf), NULL, TTY_READ_STRING))) {
	    D(D_ERROR, "FJSMS: %s(): waiting for prompt failed\n", ___F);
	    goto error;
	}
	if (write(ps_data->fjsms.fd, "yes\r", 4) != 4) {
	    D(D_ERROR, "FJSMS: %s(): sending confirmation failed\n", ___F);
	    goto error;
	}
	if (PP_FAILED(ps_read_response(ps_data->fjsms.fd, 0, "Press any key to continue ", 5, 0,
				       rsp_buf, sizeof(rsp_buf), NULL, TTY_READ_STRING))) {
	    D(D_ERROR, "FJSMS: %s(): waiting for prompt failed\n", ___F);
	    goto error;
	}
	pp_propchange_enqueue(PP_PROP_PWR_OFF, 0);
    }

    ret = PP_SUC;

 error:
    return ret;
}

static int
do_reset(power_switch_t *this UNUSED, ps_data_t * ps_data,
	 int port UNUSED, int device UNUSED)
{
    int state = get_state(this, ps_data, port, device);
    char rsp_buf[2048];
    int ret = PP_ERR;

    if (state == -1) {
	goto error;
    } else if (state == 1) {
	if (write(ps_data->fjsms.fd, "2", 1) != 1) {
	    D(D_ERROR, "FJSMS: %s(): sending 'reset' command failed\n", ___F);
	    goto error;
	}
	if (PP_FAILED(ps_read_response(ps_data->fjsms.fd, 0, "Do you really want to reset (yes/no)? ", 2, 0,
				       rsp_buf, sizeof(rsp_buf), NULL, TTY_READ_STRING))) {
	    D(D_ERROR, "FJSMS: %s(): waiting for prompt failed\n", ___F);
	    goto error;
	}
	if (write(ps_data->fjsms.fd, "yes\r", 4) != 4) {
	    D(D_ERROR, "FJSMS: %s(): sending confirmation failed\n", ___F);
	    goto error;
	}
	if (PP_FAILED(ps_read_response(ps_data->fjsms.fd, 0, "Press any key to continue ", 5, 0,
				       rsp_buf, sizeof(rsp_buf), NULL, TTY_READ_STRING))) {
	    D(D_ERROR, "FJSMS: %s(): waiting for prompt failed\n", ___F);
	    goto error;
	}
	pp_propchange_enqueue(PP_PROP_RST, 0);
    }

    ret = PP_SUC;

 error:
    return ret;
}

static int
get_state(power_switch_t *this UNUSED, ps_data_t * ps_data,
	  int port UNUSED, int device UNUSED)
{
    char rsp_buf[2048];

    if (write(ps_data->fjsms.fd, "\r", 1) != 1) goto error;

    if (PP_FAILED(ps_read_response(ps_data->fjsms.fd, 0, "Enter selection or (0) to quit: ", 2, 0,
				   rsp_buf, sizeof(rsp_buf), NULL, TTY_READ_STRING))) {
	goto error;
    }

    if (strstr(rsp_buf, "Power Status: Off")) {
	return 0;
    } else if (strstr(rsp_buf, "Power Status: On")) {
	return 1;
    }

 error:
    return -1;
}
