#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <liberic_pthread.h>
#include <pp/powerswitch.h>
#include <pp/cfg.h>
#include <lara.h>
#include "debug.h"
#include "power_switch.h"

#if defined(PP_FEAT_PCI_ADC) && defined(POWER_DEBUG)
#include <pp/sense.h>
#endif


static int power_switch(power_switch_t *this, ps_data_t * ps_data, int port, int device, u_char on, int seq);
static int do_reset(power_switch_t *this, ps_data_t * ps_data, int port, int device);
static int get_count(power_switch_t *this, ps_data_t * ps_data, int device_id, int object);
static char* get_name(power_switch_t *this, ps_data_t * ps_data, int idx, int device_id, int object);

/* -------------------- exported functions ---------------------------- */

int
power_switch_intern_init(power_switch_t *this)
{
    this->power_switch	= power_switch;
    this->do_reset	= do_reset;
    this->get_count	= get_count;
    this->get_name	= get_name;

    this->caps = POWER_CAP_SWITCH | POWER_CAP_RESET;

     // pins are gpios 
     pp_gpio_set_alternate(PP_GPIO_HOST_DEV, 0, PP_GPIO_MASK(PP_GPIO_HOST_ATX)); 
     pp_gpio_set_alternate(PP_GPIO_HOST_DEV, 0, PP_GPIO_MASK(PP_GPIO_HOST_RST)); 

     // configure output and input pins 
     pp_gpio_bit_set_enable(PP_GPIO_HOST_DEV, PP_GPIO_HOST_ATX, 1); 
     pp_gpio_bit_set_enable(PP_GPIO_HOST_DEV, PP_GPIO_HOST_RST, 1); 

     pp_gpio_bit_set(PP_GPIO_HOST_DEV, PP_GPIO_HOST_ATX, 0);
     pp_gpio_bit_set(PP_GPIO_HOST_DEV, PP_GPIO_HOST_RST, 0);

     return PP_SUC;
}

void
power_switch_intern_cleanup(void* this)
{
    power_switch_t *entry = (power_switch_t*) this;
    free(entry);
}

static int
get_count(power_switch_t *this UNUSED, ps_data_t * ps_data UNUSED,
	  int device_id UNUSED, int object)
{
    switch (object) {
      case PWR_OBJ_PORT: return 1;
      case PWR_OBJ_DEVICE: return 1;
      default: return 0;  
    }
}

static char*
get_name(power_switch_t *this UNUSED, ps_data_t * ps_data UNUSED,
	 int idx UNUSED, int device_id UNUSED, int object)
{
    switch (object) {
      case PWR_OBJ_PORT:
      case PWR_OBJ_SWITCH_SHORT_NAME:
      case PWR_OBJ_SWITCH_LONG_NAME:
	  return ps_get_pswitch_name(POWER_SWITCH_INTERN_NAME, object);
      case PWR_OBJ_SWITCH_ID:
	  return strdup(POWER_SWITCH_INTERN_NAME);
      default:
	  return NULL;
    }
}

static int
power_switch(power_switch_t *this UNUSED, ps_data_t * ps_data UNUSED,
	     int port UNUSED, int device UNUSED, u_char short_press, int seq UNUSED)
{
    u_int def_duration, duration;
    
    if (short_press) {
	def_duration = DEF_SHORT_POWER_SIGNAL_DURATION;
	pp_cfg_get_int(&duration, "ps.internal.power_short_duration");
    } else {
	def_duration = DEF_LONG_POWER_SIGNAL_DURATION;
	pp_cfg_get_int(&duration, "ps.internal.power_long_duration");
    }
    if (duration < MIN_POWER_SIGNAL_DURATION || duration > MAX_POWER_SIGNAL_DURATION) {
	duration = def_duration;
    }

    /* FIXME: waiting via thread? */
    //    pp_gpio_bit_set_enable(PP_GPIO_HOST_DEV, PP_GPIO_HOST_ATX, 1);
    pp_gpio_bit_set(PP_GPIO_HOST_DEV, PP_GPIO_HOST_ATX, 1);
    usleep(duration * 1000);
    pp_gpio_bit_set(PP_GPIO_HOST_DEV, PP_GPIO_HOST_ATX, 0);
    pp_propchange_enqueue(PP_PROP_PWR_ON, 0); /* FIXME: power on? */
    usleep(1000);
    
#if defined POWER_DEBUG
    pp_log("libpp_powerswitch: intern power switch %s press (duration: %u ms)\n", short_press ? "short":"long", duration);
#if defined(PP_FEAT_PCI_ADC) && defined(POWER_DEBUG)
    pp_sense_read_all();
#endif    
#endif
    return PP_SUC;
}

static int
do_reset(power_switch_t *this UNUSED, ps_data_t * ps_data UNUSED,
	 int port UNUSED, int device UNUSED)
{
    u_int duration;

    pp_cfg_get_int(&duration, "ps.internal.reset_duration");
    
    if (duration < MIN_RESET_SIGNAL_DURATION || duration > MAX_RESET_SIGNAL_DURATION) {
	duration = DEF_RESET_SIGNAL_DURATION;
    }
    //    pp_gpio_bit_set_enable(PP_GPIO_HOST_DEV, PP_GPIO_HOST_RST, 1);
    pp_gpio_bit_set(PP_GPIO_HOST_DEV, PP_GPIO_HOST_RST, 1);
    usleep(duration * 1000);
    pp_gpio_bit_set(PP_GPIO_HOST_DEV, PP_GPIO_HOST_RST, 0);
    pp_propchange_enqueue(PP_PROP_RST, 0);
    usleep(1000);
    
#if defined POWER_DEBUG
    pp_log("libpp_powerswitch: intern reset \n");
#endif    
    return PP_SUC;
}
