#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <pp/cfg.h>
#include <liberic_pthread.h>
#include <liberic_misc.h>
#include <pp/powerswitch.h>
#include <pp/termios.h>

#include "debug.h"
#include "power_switch.h"

static int init(power_switch_t *this, ps_data_t * ps_data);
static void uninit(power_switch_t *this, ps_data_t * ps_data);
static int get_count(power_switch_t *this, ps_data_t * ps_data, int device_id, int object);
static char* get_name(power_switch_t *this, ps_data_t * ps_data, int idx, int device_id, int object);
static int power_switch(power_switch_t *this, ps_data_t * ps_data, int port, int device, u_char on, int seq);
static int get_state(power_switch_t *this, ps_data_t * ps_data, int port, int device);

static void* ipmtty_thread_func(void* arg);

/* -------------------- exported functions ---------------------------- */

int
power_switch_ipmtty_init(power_switch_t *this)
{
    this->init          = init;
    this->uninit        = uninit;
    this->power_switch  = power_switch;
    this->get_state     = get_state;
    this->get_count     = get_count;
    this->get_name      = get_name;

    this->caps = POWER_CAP_SWITCH | POWER_CAP_GET_STATE;
    return 0;
}

void
power_switch_ipmtty_cleanup(void* this)
{
    power_switch_t *entry = (power_switch_t*) this;

    free(entry);
}

static int
init(power_switch_t *this UNUSED, ps_data_t * ps_data)
{
    ps_data->ipmtty.power = 0;
    ps_data->ipmtty.ipmtty_thread_running = 0;
    if ((ps_data->ipmtty.fd = ps_get_fd_by_port_id(ps_data->port_id)) < 0) {
	return PP_ERR;
    }
    if (PP_FAILED(pp_base_set_tty_params(ps_data->ipmtty.fd, 115200, "", 8, 0, 0, 0))) {
        return PP_ERR;
    }
#if !defined(PRODUCT_ERIC2) && !defined(PRODUCT_ERICXP) && !defined(PRODUCT_ERICG4)
    if (ps_data->port_id == PP_POWER_PORT_ID_SERIAL_1) {
        ps_data->ipmtty.state_var_name = "ps.serial[0].ipmtty.state";
    } else if (ps_data->port_id == PP_POWER_PORT_ID_SERIAL_2) {
        ps_data->ipmtty.state_var_name = "ps.serial[1].ipmtty.state";
    } else {
        ps_data->ipmtty.state_var_name = NULL;
    }
    if (ps_data->ipmtty.state_var_name) {
        int state = 0;
        pp_cfg_is_enabled_nodflt(&state, ps_data->ipmtty.state_var_name);
        power_switch(this, ps_data, 0, 0, state ? 1 : 0, 0);
    }
#endif /* !PRODUCT_ERIC2 && !PRODUCT_ERICXP */
    return PP_SUC;
}

static void
uninit(power_switch_t *this, ps_data_t * ps_data)
{
    power_switch(this, ps_data, 0, 0, 1, 0);
}

static int
get_count(power_switch_t *this UNUSED, ps_data_t * ps_data UNUSED,
	  int device_id UNUSED, int object)
{
    switch (object) {
      case PWR_OBJ_PORT: return 1;
      case PWR_OBJ_DEVICE: return 1;
      default: return 0;  
    }
}

static char *
get_name(power_switch_t *this UNUSED, ps_data_t * ps_data UNUSED,
	 int idx UNUSED, int device_id UNUSED, int object)
{
    switch (object) {
      case PWR_OBJ_PORT:
      case PWR_OBJ_SWITCH_SHORT_NAME:
      case PWR_OBJ_SWITCH_LONG_NAME:
	  return ps_get_pswitch_name(POWER_SWITCH_IPMTTY_NAME, object);
      case PWR_OBJ_SWITCH_ID:
	  return strdup(POWER_SWITCH_IPMTTY_NAME);
      default:
	  return NULL;
    }
}

static int
power_switch(power_switch_t *this UNUSED, ps_data_t * ps_data,
	     int port UNUSED, int device UNUSED, u_char on, int seq UNUSED)
{
    int ret = PP_ERR;

    if (!ps_data->ipmtty.power && !on) {
        /* turn off power */
        ps_data->ipmtty.ipmtty_thread_running = 1;
        if (eric_pthread_create(&ps_data->ipmtty.ipmtty_thread, 0, 16 * 1024,
                                ipmtty_thread_func, (void*)&ps_data->ipmtty)) {
            ps_data->ipmtty.ipmtty_thread_running = 0;
            goto error;
        }
#if !defined(PRODUCT_ERIC2) && !defined(PRODUCT_ERICXP) && !defined(PRODUCT_ERICG4)
        if (ps_data->ipmtty.state_var_name) {
            pp_cfg_set_enabled(0, ps_data->ipmtty.state_var_name);
            pp_cfg_save(DONT_FLUSH);
        }
#endif
        ps_data->ipmtty.power = 1;
        pp_propchange_enqueue(PP_PROP_PWR_OFF, 0);
    } else if (ps_data->ipmtty.power && on) {
        /* turn on power */
        ps_data->ipmtty.ipmtty_thread_running = 0;
        pthread_join(ps_data->ipmtty.ipmtty_thread, NULL);
#if !defined(PRODUCT_ERIC2) && !defined(PRODUCT_ERICXP) && !defined(PRODUCT_ERICG4)
        if (ps_data->ipmtty.state_var_name) {
            pp_cfg_set_enabled(1, ps_data->ipmtty.state_var_name);
            pp_cfg_save(DONT_FLUSH);
        }
#endif
        ps_data->ipmtty.power = 0;
        pp_propchange_enqueue(PP_PROP_PWR_ON, 0);
    }

    ret = PP_SUC;

 error:
    return ret;
}

static int
get_state(power_switch_t *this UNUSED, ps_data_t * ps_data,
	  int port UNUSED, int device UNUSED)
{
    return !ps_data->ipmtty.power;
}

/* -------------------- internal functions ---------------------------- */

static void*
ipmtty_thread_func(void * arg)
{
    ps_ipmtty_t * ps_ipmtty = (ps_ipmtty_t *)arg;
    /* To switch the IPM (tty), LARA sends continous ones to tx */
    while (ps_ipmtty->fd != -1 && ps_ipmtty->ipmtty_thread_running) {
	write(ps_ipmtty->fd, "\0", 1);
    }
    ps_ipmtty->ipmtty_thread_running = 0;
    return NULL;
}
