#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <liberic_pthread.h>
#include <gpio_hilevel_functions.h>
#include <pp/powerswitch.h>
#include <pp/termios.h>

#include "debug.h"
#include "power_switch.h"
#include "crc16.h"

//#define IPWR_SIMULATION 1

#define IPWR_MAX_DEV_CNT 16

static int init(power_switch_t * this, ps_data_t * ps_data);
static void uninit(power_switch_t * this, ps_data_t * ps_data);
static int lookup_devices(power_switch_t *this, ps_data_t * ps_data);
static int get_count(power_switch_t * this, ps_data_t * ps_data, int device_id, int object);
static int get_sw_version(power_switch_t *this, ps_data_t * ps_data, int device);
static int get_delay_enable(power_switch_t *this, ps_data_t * ps_data, int device);
static int set_delay_enable(power_switch_t *this, ps_data_t * ps_data, int device, u_char enable);
static char * get_current(power_switch_t *this UNUSED, ps_data_t * ps_data, int device_id);
static char* get_name(power_switch_t * this, ps_data_t * ps_data, int idx, int device_id, int object);
static int set_name(power_switch_t * this, ps_data_t * ps_data, int idx, int device_id, int object, const char* name);
static u_short* get_port_delays(power_switch_t * this, ps_data_t * ps_data, int idx, int device_id);
static int set_port_delays(power_switch_t * this, ps_data_t * ps_data, int idx, int device_id, u_short* delay);
static int power_switch_simultan(power_switch_t * this, ps_data_t * ps_data, int device, u_short port_values);
static int power_switch_all(power_switch_t * this, ps_data_t * ps_data, int device, u_char on);
static int power_switch(power_switch_t *this, ps_data_t * ps_data, int port, int device, u_char on, int seq);
static int get_state(power_switch_t * this, ps_data_t * ps_data, int port, int device);

static int  last_port_id;
static int device_cnt;
static u_char device_list[IPWR_MAX_DEV_CNT];

#if !defined(IPWR_SIMULATION)
static int ask_port_names(ps_data_t * ps_data, char *pnames, int port, int device);
static int ask_device_name(ps_data_t * ps_data, char *device_name, int device); 
static int set_device_name(ps_data_t * ps_data, const char *device_name, int device);
static int set_port_names(ps_data_t * ps_data, const char *pnames, int device, int port_index);
static int do_request_response(ps_data_t * ps_data, int device, char cmd, int data_len, const char* data, char *rsp_buf, int rsp_len, int timeout, int flags);
static int process_request(ps_data_t * ps_data, int device, char cmd, int data_len, const char* data, char *rsp_buf, int rsp_len, int timeout);
static void print_hex(const char *prefix, const char *buf, int len);
static int setrts(int fd, int on);

static char port_names[64 + 1];
#else
u_short internal_port_stat[16];
u_char delay_enable = 0;
#endif

/* -------------------- exported functions ---------------------------- */
int
power_switch_ipwr_init(power_switch_t *this)
{
    this->init		  = init;
    this->uninit	  = uninit;
    this->lookup_devices  = lookup_devices;
    this->get_state	  = get_state;
    this->power_switch    = power_switch;
    this->power_switch_simultan	= power_switch_simultan;
    this->power_switch_all	= power_switch_all;
    this->get_name        = get_name;
    this->set_name        = set_name;
    this->get_port_delays  = get_port_delays;
    this->set_port_delays  = set_port_delays;
    this->get_count       = get_count;
    this->get_sw_version  = get_sw_version;
    this->get_current     = get_current;
    this->get_delay_enable = get_delay_enable;
    this->set_delay_enable = set_delay_enable;
    
    this->caps = POWER_CAP_SWITCH | POWER_CAP_SWITCH_SIMULTAN |
		 POWER_CAP_SWITCH_ALL | POWER_CAP_PORT_DELAYS;

    return PP_SUC;
}

void
power_switch_ipwr_cleanup(void* this)
{
    power_switch_t *entry = (power_switch_t*) this;
    free(entry);
}

static int
init(power_switch_t *this UNUSED, ps_data_t * ps_data)
{
    ps_data->smart.smart_thread_running = 0;
    if (ps_data->ipwr.port_id == PP_POWER_PORT_ID_SERIAL_1) {
	ps_data->ipwr.devaddrcnt_var_name = "ps.serial[0].ipwr._s_";
	ps_data->ipwr.devaddr_var_name = "ps.serial[0].ipwr[%u].dev_addr";
    } else if (ps_data->ipwr.port_id == PP_POWER_PORT_ID_SERIAL_2) {
	ps_data->ipwr.devaddrcnt_var_name = "ps.serial[1].ipwr._s_";
	ps_data->ipwr.devaddr_var_name = "ps.serial[1].ipwr[%u].dev_addr";
    } else {
	ps_data->smart.devaddrcnt_var_name = NULL;
	ps_data->smart.devaddr_var_name = NULL;
    }

    /* this enabled the RS-485 mode */
    usleep(50000);
    pp_gpio_bit_set_enable(PP_GPIO_SERIAL_MODE_DEV, PP_GPIO_SERIAL_MODE, 1);
    pp_gpio_bit_set(PP_GPIO_SERIAL_MODE_DEV, PP_GPIO_SERIAL_MODE, 1);
    usleep(50000);
    
    if ((ps_data->ipwr.fd = ps_get_fd_by_port_id(ps_data->port_id)) < 0) {
	return PP_ERR;
    }

    if (PP_FAILED(pp_base_set_tty_params(ps_data->ipwr.fd, 9600, "", 8, 0, 0, 0))) {
	return PP_ERR;
    }
    
    D(D_VERBOSE, "IPWR: %s(): Init Power Switch...\n", ___F);
    
    lookup_devices(this, ps_data);

    return PP_SUC;
}

static void
uninit(power_switch_t *this UNUSED, ps_data_t * ps_data UNUSED)
{
    D(D_VERBOSE, "IPWR: %s(): Uninit Power Switch...\n", ___F);

    /* this disables the RS-485 mode */
    usleep(1000);
    pp_gpio_bit_set(PP_GPIO_SERIAL_MODE_DEV, PP_GPIO_SERIAL_MODE, 0);
    pp_gpio_bit_set_enable(PP_GPIO_SERIAL_MODE_DEV, PP_GPIO_SERIAL_MODE, 0);
    usleep(1000);
}

static int
lookup_devices(power_switch_t *this UNUSED, ps_data_t * ps_data)
{
    device_cnt = 0;
    memset(&device_list, 0, sizeof(device_list));

    D(D_VERBOSE, "IPWR: %s(): Scan Devices...\n", ___F);

#if !defined(IPWR_SIMULATION)
    u_int dev_addr;
    char rsp_buf[128];
    for ( dev_addr = 1; dev_addr <= 16; dev_addr++ ) {
	if ( PP_FAILED(do_request_response(ps_data, dev_addr, 0x31, 0, NULL,
					   rsp_buf, sizeof(rsp_buf), SCAN_DEV_TIMOUT, FLAG_NO_RETRY))) continue;

	D(D_VERBOSE, "IPWR: %s(): Device Found, Address: %u\n", ___F, dev_addr);
	device_list[device_cnt] = dev_addr;
	device_cnt++;
    }
#else /* IPWR_SIMULATION */
    (void)ps_data;
    device_list[0] = 0x03;
    device_list[1] = 0x05;
    device_cnt = 2;
#endif /* IPWR_SIMULATION */
    return PP_SUC;
}

static int
get_count(power_switch_t *this UNUSED, ps_data_t * ps_data UNUSED, int device, int object)
{
    if ( object == PWR_OBJ_PORT ) {
#if !defined(IPWR_SIMULATION)
	int n = 0;
	char rsp_buf[128];
	D(D_VERBOSE, "IPWR: %s(): Get Outlet Count\n", ___F);
	if ( PP_FAILED(do_request_response(ps_data, device_list[device], 0x31, 0, NULL,
		  		  rsp_buf, sizeof(rsp_buf), DEFAULT_TIMOUT, 0)) ) return 0;
	if ( rsp_buf[3] & 1 ) n += 4;
	if ( rsp_buf[3] & 2 ) n += 4;
	if ( rsp_buf[3] & 4 ) n += 4;
	if ( rsp_buf[3] & 8 ) n += 4;
	D(D_VERBOSE, "IPWR: %s(): Power switch %d has %d ports.\n", ___F, device, n);
	return n;
#else /* IPWR_SIMULATION */
	(void)device;
	return 16;
#endif /* IPWR_SIMULATION */
    } else if (object == PWR_OBJ_DEVICE) {
	D(D_VERBOSE, "IPWR: %s(): Get Device Count\n", ___F);
	return device_cnt;
    } else {
	return 0;
    }
}

static int
get_sw_version(power_switch_t *this UNUSED, ps_data_t * ps_data UNUSED, int device)
{
    D(D_VERBOSE, "IPWR: %s(): Get Software Version\n", ___F);
#if !defined(IPWR_SIMULATION)
    char rsp_buf[128];
    if ( PP_FAILED(do_request_response(ps_data, device_list[device], 0x38, 0, NULL,
				    rsp_buf, sizeof(rsp_buf), DEFAULT_TIMOUT, 0)) ) return PP_ERR;
    
    return rsp_buf[3];
#else /* IPWR_SIMULATION */
    (void)device;
    return 1;
#endif /* IPWR_SIMULATION */
}

static int
get_delay_enable(power_switch_t *this UNUSED, ps_data_t * ps_data, int device)
{
    D(D_VERBOSE, "IPWR: %s(): Get Delay Enable\n", ___F);

#if !defined(IPWR_SIMULATION)
    char rsp_buf[128];
    if ( PP_FAILED(do_request_response(ps_data, device_list[device], 0x2D, 0, NULL,
				    rsp_buf, sizeof(rsp_buf), DEFAULT_TIMOUT, 0)) ) return PP_ERR;

    return rsp_buf[3];
#else /* IPWR_SIMULATION */
    (void)ps_data;
    (void)device;
    return delay_enable;
#endif    
}

static int
set_delay_enable(power_switch_t *this UNUSED, ps_data_t * ps_data, int device, u_char enable)
{
    D(D_VERBOSE, "IPWR: %s(): Set Delay Enable: %x\n", ___F, enable);

#if !defined(IPWR_SIMULATION)
    char rsp_buf[128];
    if ( PP_FAILED(do_request_response(ps_data, device_list[device], 0x2C, 1, (u_char *) &enable,
				    rsp_buf, sizeof(rsp_buf), DEFAULT_TIMOUT, FLAG_CHECK_RESPONSE)) ) return PP_ERR;

    return rsp_buf[3];
#else /* IPWR_SIMULATION */
    (void)ps_data;
    (void)device;
    delay_enable = enable;
    return 1;
#endif /* IPWR_SIMULATION */
}

static char *
get_current(power_switch_t *this UNUSED, ps_data_t * ps_data, int device)
{
    D(D_VERBOSE, "IPWR: %s(): Get Current Value\n", ___F);
    
#if !defined(IPWR_SIMULATION)
    char rsp_buf[128];
    float current_value;
    u_char current_buf[32];
    if ( PP_FAILED(do_request_response(ps_data, device_list[device], 0x31, 0, NULL,
				       rsp_buf, sizeof(rsp_buf), DEFAULT_TIMOUT, 0))) return NULL;

    if ( rsp_buf[8] == 0xFF &&
	 rsp_buf[9] == 0xFF &&
	 rsp_buf[10] == 0xFF ) {
	return NULL;
    }

    current_value = rsp_buf[10] * 0.1 + rsp_buf[9] + rsp_buf[8] * 10;

    snprintf(current_buf, sizeof(current_buf), "%2.1f Amp", current_value);
    
    return strdup(current_buf);
#else /* IPWR_SIMULATION */
    (void)ps_data;
    (void)device;
    return strdup("1.1 Amp");
#endif /* IPWR_SIMULATION */
}

static char *
get_name(power_switch_t *this UNUSED, ps_data_t * ps_data,
	 int idx, int device_id, int object)
{
#if !defined(IPWR_SIMULATION)
    char device_name[64 + 1];
#else /* IPWR_SIMULATION */
    (void)ps_data;
#endif /* IPWR_SIMULATION */
    char * val = NULL;

    if (!idx) last_port_id = -1;

    switch (object) {
      case PWR_OBJ_PORT:
#if !defined(IPWR_SIMULATION)
	  if ((idx <= 7 && last_port_id > 7)
	      || (idx > 7 && last_port_id <= 7)
	      || (last_port_id < 0 )) {
	      if (PP_SUCCED(ask_port_names(ps_data, port_names, idx, device_id))) {
		  last_port_id = idx;
	      }
	  }
	  if (last_port_id < 0) {
	      asprintf(&val, "Outlet%d", idx + 1);
	      return val;
	  } else {
	      last_port_id = idx;
	      return strndup(port_names + (idx % 8) * 8, 8);
	  }
#else /* IPWR_SIMULATION */
	  asprintf(&val, "Outlet%d", idx + 1);
	  return val;
#endif /* IPWR_SIMULATION */
      case PWR_OBJ_DEVICE:
#if !defined(IPWR_SIMULATION)
	  memset(device_name, 0, sizeof(device_name));
	  if (PP_SUCCED(ask_device_name(ps_data, device_name, device_id))) {
	      asprintf(&val, "%s\n", device_name);
	  } else {
	      asprintf(&val, "Chain ID: %d\n", device_list[device_id]);
	  }
	  return val;
#else /* IPWR_SIMULATION */
	  asprintf(&val, "Chain ID: %d\n", device_list[device_id]);
	  return val;
#endif /* IPWR_SIMULATION */
      case PWR_OBJ_DEVICE_ADDR:
	  asprintf(&val, "%d", device_list[device_id]);
	  return val;
      case PWR_OBJ_SWITCH_SHORT_NAME:
      case PWR_OBJ_SWITCH_LONG_NAME:
	  return ps_get_pswitch_name(POWER_SWITCH_IPWR_NAME, object);
      case PWR_OBJ_SWITCH_ID:
	  return strdup(POWER_SWITCH_IPWR_NAME);
      default:
	  return NULL;
    }
}

static int
set_name(power_switch_t *this UNUSED, ps_data_t * ps_data,
	 int idx, int device_id, int object, const char* name)
{
#if !defined(IPWR_SIMULATION)

    int ret = PP_ERR;

    switch (object) {
      case PWR_OBJ_PORT:
	  D(D_VERBOSE, "IPWR: %s(): Set Port Name (%d-%d): %s\n", ___F, device_id, idx, name);
	  if (PP_FAILED(set_port_names(ps_data, name, device_id, idx))) goto error;
	  break;
      case PWR_OBJ_DEVICE:
	  D(D_VERBOSE, "IPWR: %s(): Set Device Name (%d): %s\n", ___F, device_id, name);
	  if (PP_FAILED(set_device_name(ps_data, name, device_id))) goto error;
	  usleep(500000);
	  break;
      default:
	  break;
    }
    ret = PP_SUC;

 error:
    return ret;

#else /* IPWR_SIMULATION */

    (void)ps_data;
    (void)idx;
    (void)device_id;
    (void)object;
    (void)name;
    return PP_SUC;

#endif /* IPWR_SIMULATION */
}

static u_short* get_port_delays(power_switch_t * this UNUSED, ps_data_t * ps_data, int idx, int device_id) {
    int i;
    u_short* port_delays = calloc(2, 8);

    D(D_VERBOSE, "IPWR: %s(): Get Port Delays\n", ___F);

#if !defined(IPWR_SIMULATION)
    char rsp_buf[128];
    if ( !idx ) {
	if (PP_FAILED(do_request_response(ps_data, device_list[device_id], 0x2E, 0, NULL,
					  rsp_buf, sizeof(rsp_buf), DEFAULT_TIMOUT, 0))) {
	    return NULL;
	}
    } else {
	if (PP_FAILED(do_request_response(ps_data, device_list[device_id], 0x2F, 0, NULL,
					  rsp_buf, sizeof(rsp_buf), DEFAULT_TIMOUT, 0))) {
	    return NULL;
	}
    }
    
    // switch LS and MS Bytes
    for ( i = 0; i < 8; i++ ) {
	port_delays[i] = rsp_buf[3 + i * 2] | (rsp_buf[4 + i * 2] << 8);
    }
    
    return port_delays;
#else /* IPWR_SIMULATION */
    (void)ps_data;
    (void)idx;
    for ( i = 0; i < 8; i++ ) {
	port_delays[i] = (u_short) (i * ( device_id + 1 )) + 1;
    }
    
    return port_delays;
#endif /* IPWR_SIMULATION */
}

static int
power_switch_simultan(power_switch_t *this UNUSED, ps_data_t * ps_data,
	              int device, u_short port_values)
{
    D(D_VERBOSE, "IPWR: %s(): Switch Power Device %d\n", ___F, device);

#if !defined(IPWR_SIMULATION)
    u_short port_values_le = cpu_to_be16(port_values);
    char rsp_buf[128];
    if ( PP_FAILED(do_request_response(ps_data, device_list[device], 0x30, 2, (u_char *) &port_values_le,
				    rsp_buf, 0, DEFAULT_TIMOUT, 0)) ) return PP_ERR;
#else /* IPWR_SIMULATION */
    (void)ps_data;
    internal_port_stat[device] = port_values;
#endif /* IPWR_SIMULATION */
    return 0;
}

static int
power_switch_all(power_switch_t *this UNUSED, ps_data_t * ps_data,
	         int device, u_char on)
{
    D(D_VERBOSE, "IPWR: %s(): Switch Power Device %d\n", ___F, device);
#if !defined(IPWR_SIMULATION)
    char rsp_buf[128];
    if ( on ) {
	if (PP_FAILED(do_request_response(ps_data, device_list[device], 0x36, 0, NULL,
					rsp_buf, 0, DEFAULT_TIMOUT, 0)) ) return PP_ERR;
    } else {
	if (PP_FAILED(do_request_response(ps_data, device_list[device], 0x37, 0, NULL,
					rsp_buf, sizeof(rsp_buf), DEFAULT_TIMOUT, 0)) ) return PP_ERR;
    }
#else /* IPWR_SIMULATION */
    (void)ps_data;
    if ( on ) {
	internal_port_stat[device] = 0xFFFF;
    } else {
	internal_port_stat[device] = 0;
    }
#endif /* IPWR_SIMULATION */
    return 0;
}

static int
power_switch(power_switch_t *this UNUSED, ps_data_t * ps_data,
	     int port, int device, u_char on, int seq UNUSED)
{
    D(D_VERBOSE, "IPWR: %s(): IP-Power Single Outlet Switch\n", ___F);

#if !defined(IPWR_SIMULATION)
    u_short port_states = 0, port_states_le = 0;
    char rsp_buf[128];
    if (PP_FAILED(do_request_response(ps_data, device_list[device], 0x31, 0, NULL,
				    rsp_buf, sizeof(rsp_buf), DEFAULT_TIMOUT, 0)) ) return PP_ERR;

    if ( rsp_buf[2] != 0x08 ) {
	D(D_ERROR, "IPWR: %s(): IP-Power: 8 bytes expected, got %x\n", ___F, rsp_buf[2]);
	return -1;
    }

    port_states = rsp_buf[4] << 8 | rsp_buf[5];

    port_states ^= ( 1 << port );

    port_states_le = cpu_to_be16(port_states);

    if (PP_FAILED(do_request_response(ps_data, device_list[device], 0x30, 2, (u_char *) &port_states_le,
				    rsp_buf, 0, DEFAULT_TIMOUT, 0)) ) return PP_ERR;
#else /* IPWR_SIMULATION */
    (void)ps_data;
    internal_port_stat[device] ^= ( 1 << port );
#endif /* IPWR_SIMULATION */
    D(D_VERBOSE, "IPWR: %s(): Power Switch: Port %d, Dev %d, On %d\n", ___F, port, device, on);
    
    return 0;
}

static int
get_state(power_switch_t *this UNUSED, ps_data_t * ps_data, int port UNUSED, int device)
{
    D(D_VERBOSE, "IPWR: %s(): Getting State\n", ___F);

#if !defined(IPWR_SIMULATION)
    int port_states = 0;
    char rsp_buf[128];
    if (PP_FAILED(do_request_response(ps_data, device_list[device], 0x31, 0, NULL,
				    rsp_buf, sizeof(rsp_buf), DEFAULT_TIMOUT, 0)) ) return PP_ERR;

    if (rsp_buf[2] != 8) {
	D(D_ERROR, "IPWR: %s(): 8 bytes expected, got %u\n", ___F, rsp_buf[2]);
	goto error;
    }

    port_states = rsp_buf[4] << 8 | rsp_buf[5];

 error:    
    return port_states;
#else /* IPWR_SIMULATION */
    (void)ps_data;
    return internal_port_stat[device];
#endif /* IPWR_SIMULATION */
}

static int set_port_delays(power_switch_t * this UNUSED, ps_data_t * ps_data, int idx, int device_id, u_short* delay) {

    D(D_VERBOSE, "IPWR: %s(): Set Port Delays\n", ___F);

#if !defined(IPWR_SIMULATION)
    char rsp_buf[128];
    if ( !idx ) {
	if (PP_FAILED(do_request_response(ps_data, device_list[device_id], 0x3D, 16, (u_char *) delay,
					rsp_buf, sizeof(rsp_buf), DEFAULT_TIMOUT, FLAG_CHECK_RESPONSE)) ) return PP_ERR;
    } else {
	if (PP_FAILED(do_request_response(ps_data, device_list[device_id], 0x3E, 16, (u_char *) delay,
					rsp_buf, sizeof(rsp_buf), DEFAULT_TIMOUT, FLAG_CHECK_RESPONSE)) ) return PP_ERR;
    }
#else /* IPWR_SIMULATION */
    (void)ps_data;
    (void)idx;
    (void)device_id;
    (void)delay;
#endif /* IPWR_SIMULATION */
    return 0;
}

#if !defined(IPWR_SIMULATION)
static int set_device_name(ps_data_t * ps_data, const char *device_name, int device) {
    D(D_VERBOSE, "IPWR: %s(): Set Device Name\n", ___F);
    
    char name_buf[8];
    char rsp_buf[128];
    memset(name_buf, 0, sizeof(name_buf));

    memcpy(name_buf, device_name, strlen(device_name) > 8 ? 8 : strlen(device_name));

    if (PP_FAILED(do_request_response(ps_data, device_list[device], 0x3B, 8, name_buf,
				     rsp_buf, sizeof(rsp_buf), DEFAULT_TIMOUT, FLAG_CHECK_RESPONSE)) ) return PP_ERR;
    return 0;
}

static int set_port_names(ps_data_t * ps_data, const char *pnames, int device, int port_index) {

    D(D_VERBOSE, "IPWR: %s(): Set Port Names\n", ___F);
    
    char rsp_buf[128];
    if ( port_index == 0 ) {
	if (PP_FAILED(do_request_response(ps_data, device_list[device], 0x34, 64, pnames,
					rsp_buf, sizeof(rsp_buf), DEFAULT_TIMOUT, FLAG_CHECK_RESPONSE)) ) return PP_ERR;
    } else if ( port_index == 1 ) {
	if (PP_FAILED(do_request_response(ps_data, device_list[device], 0x35, 64, pnames,
					rsp_buf, sizeof(rsp_buf), DEFAULT_TIMOUT, FLAG_CHECK_RESPONSE)) ) return PP_ERR;
    }
    return 0;
}

static int ask_device_name(ps_data_t * ps_data, char *device_name, int device) {
    char rsp_buf[128];

    D(D_VERBOSE, "IPWR: %s(): Get Device Name\n", ___F);

    if (PP_FAILED(do_request_response(ps_data, device_list[device], 0x3C, 0, NULL,
				    rsp_buf, sizeof(rsp_buf), DEFAULT_TIMOUT, 0)) ) return PP_ERR;

    if (rsp_buf[1] != 0x45) {
	D(D_ERROR, "IPWR: %s(): Command failure. Result: 0x%02x\n", ___F, rsp_buf[1]);
	return PP_ERR;
    }

    memcpy(device_name, rsp_buf + 3, 8);

    return PP_SUC;
}

static int ask_port_names(ps_data_t * ps_data, char *pnames, int port, int device) {
    char rsp_buf[128];
    int ret = PP_ERR;

    D(D_VERBOSE, "IPWR: %s(): Get Port Names\n", ___F);

    if ( port <= 7 ) {
	if (PP_FAILED(do_request_response(ps_data, device_list[device], 0x32, 0, NULL,
					rsp_buf, sizeof(rsp_buf), DEFAULT_TIMOUT, 0)) ) return PP_ERR;
    } else {
	if (PP_FAILED(do_request_response(ps_data, device_list[device], 0x33, 0, NULL,
					rsp_buf, sizeof(rsp_buf), DEFAULT_TIMOUT, 0)) ) return PP_ERR;
    }

    if ((port <= 7 && rsp_buf[1] != 0x42)
	|| (port >  7 && rsp_buf[1] != 0x43)) {
	D(D_ERROR, "IPWR: %s(): Command failure. Result: 0x%02x\n", ___F, rsp_buf[1]);
	goto error;
    }

    if (rsp_buf[2] != 0x40) {
	D(D_ERROR, "IPWR: %s(): Wrong data length. Expected: 0x40 - got: 0x%02x\n", ___F, rsp_buf[2]);
	goto error;
    }

    memcpy(pnames, rsp_buf + 3, 64);

    ret = PP_SUC;

 error:
    return ret;
}

static int do_request_response(ps_data_t * ps_data, int device, char cmd, int data_len, const char* data, char *rsp_buf, int rsp_len, int timeout, int flags)
{
    int i, ret;
    
    for ( i = 0; i < MAX_RETRY_CNT; i++ ) {
	ret = process_request(ps_data, device, cmd, data_len, data, rsp_buf, rsp_len, timeout);
	switch ( ret ) {	 
	  case ERR_NO_RESPONSE:
	      if ( flags & FLAG_NO_RETRY ) {
		  return ret;
	      } else {
		  continue;
	      }
	  case ERR_NO_ERROR:
	      if ( flags & FLAG_CHECK_RESPONSE ) {
		  if ( rsp_buf[1] != 0x48 ) {		      
		      return ERR_RSP_FAILED;
		  }
	      }
	  default: return ret;
	}
    }
    return ret;
}

static int process_request(ps_data_t * ps_data, int device, char cmd, int data_len, const char* data, char *rsp_buf, int rsp_len, int timeout)
{
    int  reclen, error = ERR_NO_ERROR;
    char cmd_buf[data_len+5];

    memset(cmd_buf, 0, sizeof(cmd_buf));

    cmd_buf[0] = device;
    cmd_buf[1] = cmd;
    cmd_buf[2] = data_len;
    
    if ( (data_len > 0) && (data != NULL) ) {
	memcpy( &cmd_buf[3], data, data_len);
    }

    CRC16(cmd_buf, 3 + data_len, &cmd_buf[3 + data_len], &cmd_buf[4 + data_len]);

    print_hex("Param Request: ", cmd_buf, 5 + data_len);

    if (write(ps_data->ipwr.fd, cmd_buf, 5 + data_len) != 5 + data_len) {
	D(D_ERROR, "IPWR: %s(): Writing command failed\n", ___F);
	return ERR_WRITE_FAILED;
    }

    usleep(100+(data_len*5000));

    setrts(ps_data->ipwr.fd, 0);
    
    memset( rsp_buf, 0, rsp_len );

    if (PP_FAILED(ps_read_response(ps_data->ipwr.fd, 0, rsp_buf, 0, timeout,
				   rsp_buf, rsp_len, &reclen, TTY_READ_IPWR_MSG))) {
	D(D_ERROR, "IPWR: %s(): no response\n", ___F);
	if ( rsp_len > 0 ) {
	    error = ERR_NO_RESPONSE;
	}
    }

    usleep(20000);

    setrts(ps_data->ipwr.fd, 1);

    usleep(1);
  
    if ( !error ) {
	print_hex("Param Response: ", rsp_buf, reclen);
    }

    return error;
}

static void print_hex(const char *prefix, const char *buf, int len) {
    int i;

    printf(prefix);
    for ( i = 0; i < len; i++ ) {
	printf("%x ", buf[i]);
    }
    printf("\n");
}

/*
 *   Set or Clear RTS modem control line
 *
 *   Note:  TIOCMBIS:   CoMmand BIt Set
 *          TIOCMBIC:   CoMmand BIt Clear
 *
 */
static int
setrts(int fd, int on)
{
    int controlbits = TIOCM_RTS;
    return ioctl(fd, (on ? TIOCMBIS : TIOCMBIC), &controlbits) == 0 ? PP_SUC : PP_ERR;
}
#endif /* !IPWR_SIMULATION */
