#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <liberic_pthread.h>
#include <liberic_misc.h>
#include <liberic_config.h>
#include <pp/powerswitch.h>
#include <pp/termios.h>

#include "debug.h"
#include "power_switch.h"

#define STOP_CHARACTER 0x5b

static int init(power_switch_t *this, ps_data_t * ps_data);
static char* get_name(power_switch_t *this, ps_data_t * ps_data, int idx, int device_id, int object);
static int get_count(power_switch_t *this, ps_data_t * ps_data, int device_id, int object);
static int get_state(power_switch_t *this, ps_data_t * ps_data, int port, int device);
static int power_switch(power_switch_t *this, ps_data_t * ps_data, int port, int device, u_char on, int seq);

/* -------------------- exported functions ---------------------------- */

int
power_switch_les_init(power_switch_t *this)
{
    this->init		= init;
    this->power_switch	= power_switch;
    this->get_state	= get_state;
    this->get_count	= get_count;
    this->get_name	= get_name;
    
    this->caps = POWER_CAP_SWITCH | POWER_CAP_GET_STATE;
    return PP_SUC;
}

void
power_switch_les_cleanup(void* this)
{
    power_switch_t *entry = (power_switch_t*) this;
    free(entry);
}

static int
init(power_switch_t *this UNUSED, ps_data_t * ps_data)
{
    if (PP_FAILED(pp_base_set_tty_params(ps_data->les.fd, 57600, "", 8, 1, 0, 0))) {
	return PP_ERR;
    }
    return PP_SUC;
}

static int
get_count(power_switch_t *this UNUSED, ps_data_t * ps_data UNUSED,
	  int device_id UNUSED, int object)
{
    switch (object) {
      case PWR_OBJ_PORT:
	  return 8;      
      default:
	  return 0;
    }
}

static char *
get_name(power_switch_t *this UNUSED, ps_data_t * ps_data UNUSED,
	 int idx, int device_id UNUSED, int object)
{    
    char * val = NULL;

    switch (object) {
      case PWR_OBJ_PORT:	  
	  asprintf(&val, "Socket %d", idx + 1);
	  return val;
      case PWR_OBJ_SWITCH_SHORT_NAME:
      case PWR_OBJ_SWITCH_LONG_NAME:
	  return ps_get_pswitch_name(POWER_SWITCH_LES_NAME, object);
      case PWR_OBJ_SWITCH_ID:
	  return strdup(POWER_SWITCH_LES_NAME);
      default:
	  return NULL;
    }
}

static int
power_switch(power_switch_t *this UNUSED, ps_data_t * ps_data,
	     int port, int device UNUSED, u_char on, int seq UNUSED)
{
    char state;
    char cmd_buf[32] = { 0x52, 0x7f, 0xc0 };
    char rsp_buf[35];
    int ret = PP_ERR;

    assert(port >= 0 && port <= 3);
    
    // read current state
    if (write(ps_data->les.fd, cmd_buf, 3) != 3) {
	D(D_ERROR, "LES: %s(): write() to read stats failed\n", ___F);
	goto error;
    }

    memset(cmd_buf, 0, sizeof(cmd_buf));
    memset(rsp_buf, 0, sizeof(rsp_buf));

    if (PP_FAILED(ps_read_response(ps_data->les.fd, STOP_CHARACTER, NULL, 0, 500000,
				   rsp_buf, sizeof(rsp_buf), NULL, TTY_READ_STOP_CHR))) {
	D(D_ERROR, "LES: %s(): reading stats failed\n", ___F);
	goto error;
    }
    state = *(rsp_buf + 1);
    
    // set command
    cmd_buf[0] = 0x57;
    if (write(ps_data->les.fd, cmd_buf, 3) != 3) {
	D(D_ERROR, "LES: %s(): set command failed\n", ___F);
	goto error;
    }

    if (PP_FAILED(ps_read_response(ps_data->les.fd, STOP_CHARACTER, NULL, 0, 500000,
				   rsp_buf, 2, NULL, TTY_READ_STOP_CHR))) {
	D(D_ERROR, "LES: %s(): set command - no response\n", ___F);
	goto error;
    }

    /* set cmd depending on the on/off action */
    cmd_buf[0] = (on ? (state | (1 << port)) : (state & ~(1 << port)));
    cmd_buf[1] = 0;
    cmd_buf[2] = 0;
    
    // send data
    if (write(ps_data->les.fd, cmd_buf, 32) != 32) {
	D(D_ERROR, "LES: %s(): sending data failed\n", ___F);
	goto error;
    }

    if (PP_FAILED(ps_read_response(ps_data->les.fd, STOP_CHARACTER, NULL, 0, 500000,
				   rsp_buf, 2, NULL, TTY_READ_STOP_CHR))) {
	D(D_ERROR, "LES: %s(): no response for sent data\n", ___F);
	goto error;
    }
        
    // confirm
    cmd_buf[0] = 0x57;
    cmd_buf[1] = 0xff;
    cmd_buf[2] = 0xff;
    if (write(ps_data->les.fd, cmd_buf, 3) != 3) {
	D(D_ERROR, "LES: %s(): sending confirm failed\n", ___F);
	goto error;
    }

    if (PP_FAILED(ps_read_response(ps_data->les.fd, STOP_CHARACTER, NULL, 0, 500000,
				   rsp_buf, 2, NULL, TTY_READ_STOP_CHR))) {
	D(D_ERROR, "LES: %s(): no response to confirmation\n", ___F);
	goto error;
    }

    pp_propchange_enqueue(on ? PP_PROP_PWR_ON : PP_PROP_PWR_OFF, 0);

    ret = PP_SUC;

 error:
    return ret;
}

static int
get_state(power_switch_t *this UNUSED, ps_data_t * ps_data,
	  int port, int device UNUSED)
{
    int state;
    char cmd_buf[32] = { 0x52, 0x7f, 0xc0 };    
    char rsp_buf[35];
    
    assert(port >= 0 && port <= 3);

    if (write(ps_data->les.fd, cmd_buf, 3) != 3) {
	D(D_ERROR, "LES: %s(): write() to read stats failed\n", ___F);
	return -1;
    }

    memset(rsp_buf, 0, sizeof(rsp_buf));

    if (PP_FAILED(ps_read_response(ps_data->les.fd, STOP_CHARACTER, NULL, 0, 500000,
				   rsp_buf, sizeof(rsp_buf), NULL, TTY_READ_STOP_CHR))) {
	D(D_ERROR, "LES: %s(): reading stats failed\n", ___F);
        return -1;
    }
    state = *(rsp_buf + 1);

    return (state & (1 << port)) ? 1 : 0;
}
