#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <liberic_pthread.h>
#include <liberic_misc.h>
#include <liberic_config.h>
#include <pp/powerswitch.h>
#include <pp/termios.h>

#include "debug.h"
#include "power_switch.h"

#define STOP_CHARACTER 0x0D

static int init(power_switch_t *this, ps_data_t * ps_data);
static char* get_name(power_switch_t *this, ps_data_t * ps_data, int idx, int device_id, int object);
static int get_count(power_switch_t *this, ps_data_t * ps_data, int device_id, int object);
static int get_state(power_switch_t *this, ps_data_t * ps_data, int port, int device);
static int power_switch(power_switch_t *this, ps_data_t * ps_data, int port, int device, u_char on, int seq);

/* -------------------- exported functions ---------------------------- */

int
power_switch_lesms_init(power_switch_t *this)
{
    this->init		= init;
    this->power_switch	= power_switch;
    this->get_state	= get_state;
    this->get_count	= get_count;
    this->get_name	= get_name;

    this->caps = POWER_CAP_SWITCH | POWER_CAP_GET_STATE;
    return PP_SUC;
}

void
power_switch_lesms_cleanup(void* this)
{
    power_switch_t *entry = (power_switch_t*) this;
    free(entry);
}

static int
init(power_switch_t *this UNUSED, ps_data_t * ps_data)
{
    if (PP_FAILED(pp_base_set_tty_params(ps_data->lesms.fd, 9600, "", 8, 0, 0, 0))) {
	return PP_ERR;
    }
    return PP_SUC;
}

static int
get_count(power_switch_t *this UNUSED, ps_data_t * ps_data UNUSED,
	  int device_id UNUSED, int object)
{
    switch (object) {
      case PWR_OBJ_PORT:
	  return 8;
      case PWR_OBJ_DEVICE:
	  return 4;
      default:
	  return 0;
    }
}

static char*
get_name(power_switch_t *this UNUSED, ps_data_t * ps_data UNUSED,
	 int idx, int device_id, int object)
{
    char * val = NULL;

    switch (object) {
      case PWR_OBJ_PORT:	  
	  asprintf(&val, "Socket %d", idx + 1);
	  return val;
      case PWR_OBJ_DEVICE:	  
	  asprintf(&val, "Switch %d", device_id + 1);
	  return val;
      case PWR_OBJ_SWITCH_SHORT_NAME:
      case PWR_OBJ_SWITCH_LONG_NAME:
	  return ps_get_pswitch_name(POWER_SWITCH_LESMS_NAME, object);
      case PWR_OBJ_SWITCH_ID:
	  return strdup(POWER_SWITCH_LESMS_NAME);
      default:
	  return NULL;
    }
}

static int
power_switch(power_switch_t *this UNUSED, ps_data_t * ps_data,
	     int port, int device, u_char on, int seq UNUSED)
{         
    char cmd_buf[16];
    ssize_t cmd_len;
    int ret = PP_ERR;

    snprintf(cmd_buf, sizeof(cmd_buf), "P%d%d=%d\r", device + 1, port + 1, on);

    cmd_len = strlen(cmd_buf);
    if (write(ps_data->lesms.fd, cmd_buf, cmd_len) != cmd_len) {
	D(D_ERROR, "LESMS: %s(): sending command failed\n", ___F);
	goto error;
    }

    usleep(300000);

    pp_propchange_enqueue(on ? PP_PROP_PWR_ON : PP_PROP_PWR_OFF, 0);

    ret = PP_SUC;

 error:
    return ret;
}

static int
get_state(power_switch_t *this UNUSED, ps_data_t * ps_data,
	  int port, int device)
{
    int  state = -1;
    char cmd_buf[16];
    char rsp_buf[16];
    ssize_t cmd_len;
    
    snprintf(cmd_buf, sizeof(cmd_buf), "R%1d%1d\r", device + 1, port + 1);

    cmd_len = strlen(cmd_buf);
    if (write(ps_data->lesms.fd, cmd_buf, cmd_len) != cmd_len) {
	D(D_ERROR, "LESMS: %s(): sending command failed\n", ___F);
	return -1;
    }

    usleep(500000);

    if (PP_FAILED(ps_read_response(ps_data->lesms.fd, STOP_CHARACTER, NULL, 0, 500000,
				   rsp_buf, sizeof(rsp_buf), NULL, TTY_READ_STOP_CHR))) {
	D(D_ERROR, "LESMS: %s(): no response to command\n", ___F);
	return -1;
    }

    snprintf(cmd_buf, sizeof(cmd_buf), "\rP%1d%1d=%%d", device + 1, port + 1);
    if (sscanf(rsp_buf, cmd_buf, &state) != 1)  {
	snprintf(cmd_buf, sizeof(cmd_buf), "\rR%1d%1d=%%d", device + 1, port + 1);
	if (sscanf(rsp_buf, cmd_buf, &state) != 1)  {
	    return -1;
	}
    }

    usleep(300000);

    return state;
}
