#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <liberic_pthread.h>
#include <pp/powerswitch.h>
#ifdef PP_FEAT_PCI_ADC
#include <pp/sense.h>
#endif
#include <pp/cfg.h>
#include <lara.h>
#include "debug.h"
#include "power_switch.h"

static int power_switch(power_switch_t *this, ps_data_t * ps_data, int port, int device, u_char on, int seq);
static int power_stdby_switch(power_switch_t *this, ps_data_t * ps_data, int port, int device, u_char on, int seq);
static int power_button(power_switch_t *this, ps_data_t * ps_data, int port, int device, u_char pb_enabled, int seq);
static int get_count(power_switch_t *this, ps_data_t * ps_data, int device_id, int object);
static char* get_name(power_switch_t *this, ps_data_t * ps_data, int idx, int device_id, int object);

/* -------------------- exported functions ---------------------------- */

int
power_switch_psu_adapt_init(power_switch_t *this)
{
    int pb_enabled;

    this->power_switch	= power_switch;
    this->stdby_switch	= power_stdby_switch;
    this->power_button	= power_button;
    this->get_count	= get_count;
    this->get_name	= get_name;
    this->caps = POWER_CAP_SWITCH;

#if defined POWER_DEBUG
    pp_log("libpp_powerswitch: U-BOOT -> PP_GPIO_PSU_STDBY   = %u \n", pp_gpio_bit_get(PP_GPIO_PSU_DEV, PP_GPIO_PSU_STDBY));
    pp_log("libpp_powerswitch: U-BOOT -> PP_GPIO_PSU_ONOFF   = %u \n", pp_gpio_bit_get(PP_GPIO_PSU_DEV, PP_GPIO_PSU_ONOFF));
    pp_log("libpp_powerswitch: U-BOOT -> PP_GPIO_PSU_ISOLATE = %u \n", pp_gpio_bit_get(PP_GPIO_PSU_DEV, PP_GPIO_PSU_ISOLATE));    
#endif    

    // pins are gpios
    pp_gpio_set_alternate(PP_GPIO_PSU_DEV, 0, PP_GPIO_MASK(PP_GPIO_PSU_STDBY));
    pp_gpio_set_alternate(PP_GPIO_PSU_DEV, 0, PP_GPIO_MASK(PP_GPIO_PSU_ONOFF));
    pp_gpio_set_alternate(PP_GPIO_PSU_DEV, 0, PP_GPIO_MASK(PP_GPIO_PSU_ISOLATE));

    // configure output and input pins

    pp_gpio_bit_set_enable(PP_GPIO_PSU_DEV, PP_GPIO_PSU_STDBY, 1);
    pp_gpio_bit_set_enable(PP_GPIO_PSU_DEV, PP_GPIO_PSU_ONOFF, 1);
    pp_gpio_bit_set_enable(PP_GPIO_PSU_DEV, PP_GPIO_PSU_ISOLATE, 1);

    pp_gpio_bit_set(PP_GPIO_PSU_DEV, PP_GPIO_PSU_ONOFF, 0);	
    pp_gpio_bit_set(PP_GPIO_PSU_DEV, PP_GPIO_PSU_STDBY, 1);
    
    pp_cfg_is_enabled(&pb_enabled, "ps.power_button_enabled");
    power_button(this, 0, 0, 0, pb_enabled, 0);

    return PP_SUC;
}

void
power_switch_psu_adapt_cleanup(void* this)
{
    power_switch_t *entry = (power_switch_t*) this;
    free(entry);
}

static int
get_count(power_switch_t *this UNUSED, ps_data_t * ps_data UNUSED,
	  int device_id UNUSED, int object)
{
    switch (object) {
      case PWR_OBJ_PORT: return 1;
      case PWR_OBJ_DEVICE: return 1;
      default: return 0;  
    }
}

static char*
get_name(power_switch_t *this UNUSED, ps_data_t * ps_data UNUSED,
	 int idx UNUSED, int device_id UNUSED, int object)
{
    switch (object) {
      case PWR_OBJ_PORT:
      case PWR_OBJ_SWITCH_SHORT_NAME:
      case PWR_OBJ_SWITCH_LONG_NAME:
	  return ps_get_pswitch_name(POWER_SWITCH_PSU_ADAPT_NAME, object);
      case PWR_OBJ_SWITCH_ID:
	  return strdup(POWER_SWITCH_PSU_ADAPT_NAME);
      default:
	  return NULL;
    }
}

static int
power_switch(power_switch_t *this UNUSED, ps_data_t * ps_data UNUSED,
	     int port UNUSED, int device UNUSED, u_char power_on, int seq UNUSED)
{

    float voltage;
    int currently_on;

    if (PP_FAILED(pp_sense_read_adc_value_raw(&voltage, PCI_VOLTAGE_12V)) ) {
#if defined POWER_DEBUG
	pp_log("libpp_powerswitch: psu power switch ERROR reading ADC !\n");
#endif
	return PP_ERR;
    } else {
	currently_on = (voltage > 10);
    }

    if ((power_on != currently_on)) {
	// send impulse -> toggle power state driven towards PSU
	pp_gpio_bit_set(PP_GPIO_PSU_DEV, PP_GPIO_PSU_ONOFF, 1);
	usleep(5000);
        pp_gpio_bit_set(PP_GPIO_PSU_DEV, PP_GPIO_PSU_ONOFF, 0);
	usleep(1000);
#if defined POWER_DEBUG
	pp_log("libpp_powerswitch: psu power switch \n");
	pp_sense_read_all();
    } else {
	pp_log("libpp_powerswitch: NO psu power switch (voltage: %f ; power_on: %u)\n", voltage, power_on);
	pp_sense_read_all();
#endif
    }
    pp_propchange_enqueue(PP_PROP_PWR_ON, 0);

    return PP_SUC;
}

static int
power_stdby_switch(power_switch_t *this UNUSED, ps_data_t * ps_data UNUSED,
	     int port UNUSED, int device UNUSED, u_char stdby_on, int seq UNUSED)
{
    pp_gpio_bit_set(PP_GPIO_PSU_DEV, PP_GPIO_PSU_STDBY, stdby_on);
#if defined POWER_DEBUG
    pp_log("libpp_powerswitch: standby voltage %s\n", stdby_on ? "ON": "OFF");
    pp_sense_read_all();
#endif
    usleep(1000);
    return PP_SUC;
}


static int
power_button(power_switch_t *this UNUSED, ps_data_t * ps_data UNUSED,
	     int port UNUSED, int device UNUSED, u_char pb_enabled, int seq UNUSED)
{

    pp_gpio_bit_set(PP_GPIO_PSU_DEV, PP_GPIO_PSU_ISOLATE, pb_enabled);
#if defined POWER_DEBUG
    pp_log("libpp_powerswitch: psu power button %s\n", pb_enabled ? "enabled": "disabled");
#endif
    usleep(1000);    
    return PP_SUC;
}
