#include <stdlib.h>
#include <fcntl.h>
#include <stdio.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <liberic_misc.h>
#include <liberic_config.h>
#include <pp/base.h>
#include <pp/powerswitch.h>
#include <pp/termios.h>
#include <pp/cfg.h>

#include "debug.h"
#include "power_switch.h"

static int init(power_switch_t *this, ps_data_t * ps_data);
static int login(power_switch_t *this, ps_data_t * ps_data);
static int logout(power_switch_t *this, ps_data_t * ps_data);
static int get_count(power_switch_t *this, ps_data_t * ps_data, int device_id, int object);
static char* get_name(power_switch_t *this, ps_data_t * ps_data, int idx, int device_id, int object);
static int power_switch(power_switch_t *this, ps_data_t * ps_data, int port, int device, u_char on, int seq);
static int get_state(power_switch_t *this, ps_data_t * ps_data, int port, int device);

static int sentry_get_login(ps_sentry_t * ps_sentry);
static int sentry_get_prompt(ps_sentry_t * ps_sentry);

/* -------------------- exported functions ---------------------------- */

int
power_switch_sentry_init(power_switch_t *this)
{
    this->init		= init;
    this->login		= login;
    this->logout	= logout;
    this->get_count	= get_count;
    this->get_name	= get_name;
    this->power_switch	= power_switch;
    this->get_state	= get_state;

    this->caps = POWER_CAP_SWITCH | POWER_CAP_GET_STATE;
    return 0;
}

void
power_switch_sentry_cleanup(void* this)
{
    power_switch_t *entry = (power_switch_t*) this;
    free(entry);
}

static int
init(power_switch_t *this UNUSED, ps_data_t * ps_data)
{
    const char * lspeed_var_name = NULL;
    u_int tty_spd;
    int ret = PP_ERR;

    if (ps_data->port_id == PP_POWER_PORT_ID_SERIAL_1) {
	lspeed_var_name = "ps.serial[0].sentry.speed";
	ps_data->sentry.user_var_name = "ps.serial[0].sentry.user";
	ps_data->sentry.pass_var_name = "ps.serial[0].sentry.pass";
    } else if (ps_data->port_id == PP_POWER_PORT_ID_SERIAL_2) {
	lspeed_var_name = "ps.serial[1].sentry.speed";
	ps_data->sentry.user_var_name = "ps.serial[1].sentry.user";
	ps_data->sentry.pass_var_name = "ps.serial[1].sentry.pass";
    } else {
	ps_data->sentry.user_var_name = NULL;
	ps_data->sentry.pass_var_name = NULL;
    }

    if (lspeed_var_name == NULL
	|| PP_FAILED(pp_cfg_get_uint_nodflt(&tty_spd, lspeed_var_name))) {
	goto error;
    }

    if (PP_FAILED(pp_base_set_tty_params(ps_data->sentry.fd, tty_spd, "", 8, 1, 0, 0))) {
	goto error;
    }

    ret = PP_SUC;

 error:
    return ret;
}

static int
login(power_switch_t *this UNUSED, ps_data_t * ps_data)
{
    ps_sentry_t * ps_sentry = &ps_data->sentry;
    char rsp_buf[2048];
    char* user = NULL;
    char* pass = NULL;
    ssize_t cmd_len;
    int ret = PP_ERR;

    /* first try to get prompt without login */
    if (PP_FAILED(sentry_get_prompt(ps_sentry))) {
	/* try to login two times*/
	if (PP_FAILED(sentry_get_login(ps_sentry))) {
	    sleep(2);
	    if (PP_FAILED(sentry_get_login(ps_sentry))) {
		D(D_ERROR, "SENTRY: %s(): login failed\n", ___F);
		goto error;
	    }
	}
	
	if (!ps_sentry->user_var_name
	    || PP_FAILED(pp_cfg_get_nodflt(&user, ps_sentry->user_var_name))) {
	    goto error;
	}
	if (!ps_sentry->pass_var_name
	    || PP_FAILED(pp_cfg_get_nodflt(&pass, ps_sentry->pass_var_name))) {
	    goto error;
	}

	sleep(1);

	ps_read_response(ps_sentry->fd, 0, NULL, 0, 100000, rsp_buf, sizeof(rsp_buf), NULL, TTY_READ_MAX_CHR);

	cmd_len = strlen(user);
	if (write(ps_sentry->fd, user, cmd_len) != cmd_len
	    || write(ps_sentry->fd, "\r", 1) != 1) {
	    D(D_ERROR, "SENTRY: %s(): sending username failed\n", ___F);
	    goto error;
	}
	if (PP_FAILED(ps_read_response(ps_sentry->fd, 0, "Password:", 2, 0,
				       rsp_buf, sizeof(rsp_buf), NULL, TTY_READ_STRING))) {
	    D(D_ERROR, "SENTRY: %s(): waiting for password prompt failed\n", ___F);
	    goto error;
	}

	cmd_len = strlen(pass);
	if (write(ps_sentry->fd, pass, cmd_len) != cmd_len
	    || write(ps_sentry->fd, "\r", 1) != 1) {
	    D(D_ERROR, "SENTRY: %s(): sending password failed\n", ___F);
	    goto error;
	}
	if (PP_FAILED(ps_read_response(ps_sentry->fd, 0, "Sentry:", 2, 0,
				       rsp_buf, sizeof(rsp_buf), NULL, TTY_READ_STRING))) {
	    D(D_ERROR, "SENTRY: %s(): waiting for prompt failed\n", ___F);
	    goto error;
	}
    }

    ret = PP_SUC;

 error:
    free(user);
    free(pass);
    return ret;
}

static int
logout(power_switch_t *this UNUSED, ps_data_t * ps_data)
{
    ps_sentry_t * ps_sentry = &ps_data->sentry;
    const char * cmd_buf = "quit\r";
    char rsp_buf[2048];
    ssize_t cmd_len = strlen(cmd_buf);
    int ret = PP_ERR;

    if (write(ps_sentry->fd, cmd_buf, cmd_len) != cmd_len) {
	D(D_ERROR, "SENTRY: %s(): sending 'quit' command failed\n", ___F);
	goto error;
    } else if (PP_FAILED(ps_read_response(ps_sentry->fd, 0, NULL, 0, 500000,
					  rsp_buf, sizeof(rsp_buf), NULL, TTY_READ_MAX_CHR))) {
	D(D_ERROR, "SENTRY: %s(): logout failed\n", ___F);
	goto error;
    }

    ret = PP_SUC;

 error:
    return ret;
}

static int
get_count(power_switch_t *this UNUSED, ps_data_t * ps_data UNUSED,
	  int device_id UNUSED, int object)
{
    switch (object) {
      case PWR_OBJ_PORT: return 16;
      case PWR_OBJ_DEVICE: return 1;
      default: return 0;  
    }
}

static char *
get_name(power_switch_t *this UNUSED, ps_data_t * ps_data UNUSED,
	 int idx, int device_id UNUSED, int object)
{
    char * val = NULL;

    switch (object) {
      case PWR_OBJ_PORT:
	  asprintf(&val, "A%d", idx + 1);
	  return val;
      case PWR_OBJ_SWITCH_SHORT_NAME:
      case PWR_OBJ_SWITCH_LONG_NAME:
	  return ps_get_pswitch_name(POWER_SWITCH_SENTRY_NAME, object);
      case PWR_OBJ_SWITCH_ID:
	  return strdup(POWER_SWITCH_SENTRY_NAME);
      default:
	  return NULL;
    }
}

static int
power_switch(power_switch_t *this UNUSED, ps_data_t * ps_data,
	     int port, int device UNUSED, u_char on, int seq UNUSED)
{
    char cmd_buf[16];
    char rsp_buf[2048];
    ssize_t cmd_len;
    int ret = PP_ERR;
    
    snprintf(cmd_buf, sizeof(cmd_buf), "%s .A%d\r", on ? "on" : "off", port + 1);

    cmd_len = strlen(cmd_buf);
    if (write(ps_data->sentry.fd, cmd_buf, cmd_len) != cmd_len) {
	D(D_ERROR, "SENTRY: %s(): sending command failed\n", ___F);
	goto error;
    }
    
    if (PP_FAILED(ps_read_response(ps_data->sentry.fd, 0, "Command successful", 2, 0,
				   rsp_buf, sizeof(rsp_buf), NULL, TTY_READ_STRING))) {
	D(D_ERROR, "SENTRY: %s(): no response to command\n", ___F);
	goto error;
    }

    pp_propchange_enqueue(on ? PP_PROP_PWR_ON : PP_PROP_PWR_OFF, 0);

    ret = PP_SUC;

 error:
    return ret;
}

static int
get_state(power_switch_t *this UNUSED, ps_data_t * ps_data,
	  int port, int device UNUSED)
{
    char cmd_buf[16];
    char expect_buf[16];
    char rsp_buf[2048];
    ssize_t cmd_len;
    int ret = -1;

    snprintf(cmd_buf, sizeof(cmd_buf), "status .A%d\r", port + 1);

    cmd_len = strlen(cmd_buf);
    if (write(ps_data->sentry.fd, cmd_buf, cmd_len) != cmd_len) {
	D(D_ERROR, "SENTRY: %s(): sending command failed\n", ___F);
	goto error;
    }

    /* we wait for the outlet name two times:
       -1. our own command echo, up to CR
       -2. the outlet name on the actual status output, up to <space> */
    snprintf(expect_buf, 10, ".A%d\r", port + 1);
    if (PP_FAILED(ps_read_response(ps_data->sentry.fd, 0, expect_buf, 2, 0,
				   rsp_buf, sizeof(rsp_buf), NULL, TTY_READ_STRING))) {
	D(D_ERROR, "SENTRY: %s(): no command response\n", ___F);
	goto error;
    }

    snprintf(expect_buf, sizeof(expect_buf), ".A%d ", port + 1);
    if (PP_FAILED(ps_read_response(ps_data->sentry.fd, 0, expect_buf, 2, 0,
				   rsp_buf, sizeof(rsp_buf), NULL, TTY_READ_STRING))) {
	D(D_ERROR, "SENTRY: %s(): no status response\n", ___F);
	goto error;
    }

    if (PP_FAILED(ps_read_response(ps_data->sentry.fd, 0, "Command successful", 2, 0,
				   rsp_buf, sizeof(rsp_buf), NULL, TTY_READ_STRING))) {
	D(D_ERROR, "SENTRY: %s(): no command success response\n", ___F);
	goto error;
    }

    if (strstr(rsp_buf, "Off")) {
        ret = 0;
    } else if (strstr(rsp_buf, "On")) {
        ret = 1;
    }

 error:
    return ret;
}

/* -------------------- internal functions ---------------------------- */

static int
sentry_get_prompt(ps_sentry_t * ps_sentry)
{
    char cmd_buf[1] = { 0x0d };
    char rsp_buf[2048];
    
    if (write(ps_sentry->fd, &cmd_buf, 1) != 1) return PP_ERR;
    
    return ps_read_response(ps_sentry->fd, 0, "Sentry:", 2, 0,
			    rsp_buf, sizeof(rsp_buf), NULL, TTY_READ_STRING);
}

static int
sentry_get_login(ps_sentry_t * ps_sentry)
{
    char cmd_buf[3] = { 0x1b, 0x0d, 0x0d };
    char rsp_buf[2048];
    
    if (write(ps_sentry->fd, &cmd_buf, 3) != 3) return PP_ERR;

    return ps_read_response(ps_sentry->fd, 0, "Username:", 2, 0,
			    rsp_buf, sizeof(rsp_buf), NULL, TTY_READ_STRING);
}
