/**
 * PP_DDA_object.h
 * 
 * Peppercon DDA objects, derived from CDDA_Device, CDDA_Port and CDDA_Path.
 * Implement WriteData functions with code that stores the corresponding values
 * in the config system.
 *
 * (c) 2006 Peppercon AG, Georg Hoesch <geo@peppercon.de>
 */

#ifndef __PP_DDA_OBJECTS_H__
#define __PP_DDA_OBJECTS_H__

#include "pp/RDM_DDA_Port.h"
#include "pp/RDM_DDA_Device.h"

/**
 * 
 */
class PP_DDA_Device: public CDDA_Device {
      
protected:
    /**
     * RDM device default constructor. Use only in subclasses that call
     * initialize() directly.
     */
    PP_DDA_Device(CRDM* rdm);

public:

    /**
     * Public Constructor. Wraps initialize().
     * @param rdm        RDM database that this device belongs to
     * @param pID        device id
     * @param pType      device type
     * @param pModel     device model
     * @param baseDevice set to 0 if this is a subdevice
     */
    PP_DDA_Device(CRDM* rdm, const char* pID, const char* pType, const char* pModel, int baseDevice = 1);

protected:
    /**
     * Real initialization code. Must be called in constructor.
     * @param rdm        RDM database that this device belongs to
     * @param pID        device id
     * @param pType      device type
     * @param pModel     device model
     * @param baseDevice set to 0 if this is a subdevice
     */
    virtual
    void initialize(CRDM* rdm, const char* pID, const char* pType, const char* pModel, int baseDevice);
    
public:
    /**
     * Device elements have been updated. Supported keys: Name
     */
    virtual
    int  WriteData(const char *pID, const char *pKey, const char *pData);  // supported elements: Name
    
    virtual
    void CommitData();

};


/**
 * 
 */
class PP_DDA_Port : public CDDA_Port {
protected:
    /**
     * RDM port default constructor. Use only in subclasses that call
     * initialize() directly.
     */
    PP_DDA_Port(CRDM* rdm);

public:
    /**
     * Public Constructor. Wraps initialize().
     * @param rdm          RDM database that this device belongs to
     * @param parentDevice device that this port belongs to
     * @param pID          port id
     * @param pClass       port class ("KVM", "Serial", "Outlet")
     * @param pType        port type
     * @param internal     set to 1 if this is an internal (invisible) port
     */
    PP_DDA_Port (CRDM * rdm, PP_DDA_Device *parentDevice,
                 const char *pID, const char *pClass, const char *pType,
                 int internal = 0);

protected:
    /**
     * Real initialization code. Must be called in constructor.
     * @param rdm          RDM database that this device belongs to
     * @param parentDevice device that this port belongs to
     * @param pID          port id
     * @param pClass       port class ("KVM", "Serial", "Outlet")
     * @param pType        port type
     * @param internal     set to 1 if this is an internal (invisible) port
     */
    virtual
    void initialize (CRDM * rdm, PP_DDA_Device *parentDevice,
                     const char *pID, const char *pClass, const char *pType,
                     int internal);
    
public:
    /**
     * Device elements have been updated. Supported keys: Name, AssociatedPort, 
     * AssociatedOutlet[x], DataRate, Parity, StopBits, DataBits, FlowControl
     */
    virtual
    int  WriteData(const char *pID, const char *pKey, const char *pData);
    
    virtual
    void CommitData();
    
};


/**
 * 
 */
class PP_DDA_PowerPort : public PP_DDA_Port {

protected:
    /**
     * RDM port default constructor. Use only in subclasses that call
     * initialize() directly.
     */
    PP_DDA_PowerPort(CRDM* rdm);

public:
    /**
     * Public Constructor. Wraps initialize().
     * @param rdm          RDM database that this device belongs to
     * @param parentDevice device that this port belongs to
     * @param pID          port id
     * @param pType        port type
     * @param internal     set to 1 if this is an internal (invisible) port
     */
    PP_DDA_PowerPort (CRDM * rdm, PP_DDA_Device *parentDevice,
                 const char *pID, const char *pType, int internal = 0);

protected:
    /**
     * Real initialization code. Must be called in constructor.
     * @param rdm          RDM database that this device belongs to
     * @param parentDevice device that this port belongs to
     * @param pID          port id
     * @param pType        port type
     * @param internal     set to 1 if this is an internal (invisible) port
     */
    virtual
    void initialize (CRDM * rdm, PP_DDA_Device *parentDevice,
                     const char *pID, const char *pType, int internal);

public:
    /**
     * Set the state of this outlet, use DEFINES from RDM_Power.h
     * Override in derived object.
     */
    virtual
    int setPowerStateOfOutlet(int state);
    
    /**
     * Set the state of the power strip, return NULL if not applicable.
     * Override in derived object.
     */
    virtual
    char* getPowerStripStatus();
    
    PP_DDA_PowerPort* next;
};

/**
 * 
 */
class PP_DDA_KVMport : public PP_DDA_Port {
public:
    /**
     * Public Constructor. Wraps initialize().
     * @param rdm          RDM database that this device belongs to
     * @param parentDevice device that this port belongs to
     * @param pID          port id
     * @param pClass       port class ("KVM", "Serial", "Outlet")
     * @param pType        port type
     * @param internal     set to 1 if this is an internal (invisible) port
     */
    PP_DDA_KVMport (CRDM * rdm, PP_DDA_Device *parentDevice,
                 const char *pID, const char *pClass, const char *pType,
                 int internal = 0);
    
    /**
     * Add a static powerport association to this KVM port
     */
    virtual void
    associatePowerPort(PP_DDA_PowerPort* powerPort);
    
    /**
     * Remove a static powerport association
     */
    virtual void
    unassociatePowerPort(PP_DDA_PowerPort* powerPort);
    
    // FIXME: maybe we have to overwrite WriteData here to disallow
    // that someone overwrites this static association...
};


/* Note: PP_DDA_SerialPort not done yet, maybe never needed */

/* Note: PP_DDA_Path not done yet, we do not have paths */


#endif
