/**
 * A branchmanager for peppercon objects. Base object for peppercon
 * RDM object hierarchy. Each object that represents a part of the
 * RDM database can inherit from this object to create its own
 * unique branchmanager. Objects can then overwrite Append(), 
 * Update() and Delete() to handle their very unique data, process
 * it and write it back.
 * 
 * (c) 2006 Raritan     georg.hoesch@raritan.com
 */

#ifndef __PP_RDM_BRANCHMANAGER_H__
#define	__PP_RDM_BRANCHMANAGER_H__

#include "pp/RDM_BranchManager.h"
#include "pp/Session.h"
#include "pp/SXDB.h"
#include "pp/SXML.h"
#include "pp/SXDB_Parse_Table.h"
#include "pp/RDM.h"


class PP_RDM_BranchManager: public CRDM_BranchManager
{
public:
    /**
     * Create a new branchmanager. The branchmanager won't be usable
     * until initialization.
     */
    PP_RDM_BranchManager(CRDM* rdm);
    virtual ~PP_RDM_BranchManager();
    
    /**
     * Initialize a branchmanager. This registers the branchmanager at the
     * database service and does some other initializations. This method
     * must be called before usage.
     * 
     * @param pDBService:    a reference to the database service
     */
    virtual
    void
    initialize(CRDM_Database* pDBService);
    
    /**
     * Append an element with the specified name to the node specified
     * by the xPath. This is mainly for initial bootstrapping.
     */
    CSXDB_Element*                    // new node if okay, NULL if error
    appendElement(
        CSXDB_Node* parentNode,       // parent node to which the new node will be appended
        			      // if parentNode is NULL result is NULL
        const char* childName         // name of the new element
    );
    
    /**
     * Append an element with the specified name and ID to the node specified
     * by the xPath. This is mainly for initial bootstrapping.
     */
    CSXDB_Element*                    // new node if okay, NULL if error
    appendElementWithID(
        CSXDB_Node* parentNode,       // parent node to which the new node will be appended
        			      // if parentNode is NULL result is NULL
        const char* childName,        // name of the new element
        const char* childId           // id of the new element, will be added as attribute "id=childId"
    );

    /** returns the unique BranchManager name, must be overwritten by subclasses */
    virtual
    const char*
    GetName() { return NULL; }

    /** return the hashcode for this branchmanager */
    virtual
    int
    GetHashCode();
    
    /**
     * Normalize a user request. This will calculate the required partial
     * parse table and normalize the target data and target root node.
     * The resulting variables can be used to parse the user request,
     * validate it and write it to the database without further hassle:
     * - use partialParseTable and parseDataPartial to read  normalizedTargetData
     * - use partialParseTable and parseDataPartial to write normalizedTargetNode (in DB)
     * Note: The normalized versions of node and data are both on the same level
     * while the original nodes still related as parent(node) to child(data).
     * 
     * A practical example for normalization (from DeviceSettings.cpp):
     *   rootNode   = /Device/DeviceSettings
     *   targetNode = /Device/DeviceSettings/Management/RFB/IPPort
     *   targetData = <data>443</data>
     * Best matching parsetable is DS_RFB_Table, thus (normalized) data will be:
     *   targetData = <RFB><IPPort>443</IPPort></RFB>
     *   targetRoot = /Device/DeviceSettings/Management/RFB
     * 
     * This is just a convenience function that is not required for a 
     * branchmanager. Nevertheless all subclasses will want to use this
     * to implement their Update()/Append() methods.
     * 
     * Note (geo): I don't know if this kind of normalization is the best
     * way to handle updates, I just took it as legacy stuff.
     * 
     * Note: The original parsetable must not contain PT_MOVE_UP or PT_MOVE_DOWN.
     * Use recursive tables (PT_TABLE) instead.
     */
    int                                          // returns 0 on success, RDM error (less than 0)  else
    PP_RDM_BranchManager::NormalizeRequest(
	// input params from request / local config
	CSXDB_Node*  rootNode,                   // node corresponding to rootParseTable
	SXDB_PT*     rootParseTable,             // rootParseTable that will be reduced
	void*        rootParseTableData,         // data structure for root parse table
	CSXDB_Node*  targetNode,                 // target node in DB as passed to branchmanager, has parent relationship to data
	CSXDB_Node*  targetData,                 // data to be written as passed to BM (includes <data/>)
	// output params - normalized/partial versions ready for use
	SXDB_PT**    partialParseTable,          // reduced parse table, part of rootParseTable
	void**       partialParseTableData,      // reference (within rootParseTableData) that matches partialParseTable
	CSXDB_Node** normalizedTargetNode,       // normalized targetNode (without <data/>)
	CSXDB_Node** normalizedTargetData        // normalized targetData
    );

    /** implements branchmanager Append(), must be overwritten by concrete object */
    virtual
    int                             // return 0 if ok, RDM errorcode else
    Append (
	CSession*    pUserSession,  // User session or NULL for super user
	CSXDB_Node*  pNode,         // The Node to be appended too
	CSXDB_Node*  pData          // Ptr to data to append
    );
    
    /** implements branchmanager Update(), must be overwritten by concrete object */
    virtual
    int                             // return 0 if ok, RDM errorcode else
    Update (
	CSession*    pUserSession,  // User session or NULL for super user
	CSXDB_Node*  pNode,         // The Node to update
	CSXDB_Node*  pData          // Ptr to data to update
    );

    /** implements branchmanager Delete(), must be overwritten by concrete object */
    virtual
    int                             // return 0 if ok, RDM errorcode else
    Delete (
	CSession*    pUserSession,  // User session or NULL for super user
	CSXDB_Node*  pNode          // The Node to be deleted
    );

    /** implements branchmanager PreProcessDelete(), returns 0 by default */
    virtual
    int                             // 0 = not processed (okay), 1 = processed, set error yourself
    PreProcessDelete (
	CSession*    pUserSession,  // User session or NULL for super user
	const char*  pXPath,        // Ptr to the xpath
	CSIO*        pResponse      // SIO Response
    );

    /** implements branchmanager AssociateOutlet(), returns RDM_ERROR_NOT_SUPPORTED */
    // FIXME: is this applicable for peppercon devices at all ?
    virtual
    int                             // 0 or error code
    AssociateOutlet (
	const char*  port_id,       // The target that will be controlled
	const char*  outlet_id      // The outlet that controls it
    );

    /** implements branchmanager SetPowerState(), can be overwritten by concrete object */
    // FIXME: not sure if this is called by power service yet
    virtual
    int                             // 0 or error code
    SetPowerState (
	const char*  outlet_id,     // The outlet to set the power state for
	int          state          // 0 = cycle power, 1 = On, 2 = Off
    );

    /** implements branchmanager SetPowerState(), returns NULL */
    virtual
    char*                           // Ptr to the allocated power status string
    GetPowerStripStatus (
	const char*  strip_id       // The outlet strip to get the status for
    );

protected:
	int    hashCode;            // hash of service name for fast compare
	CRDM*  rdm;                 // reference to rdm object

};

#endif // __PP_RDM_BRANCHMANAGER_H__

