/**	
 *	@file	RDM_Protocol.cpp
 *	@brief	Implementation of CRDM_Protocol_Factory & CRDM_Protocol
 *
 */

#include	<assert.h>
#include	<pp/syms.h>
#include	<pp/OS_Port.h>
#include	<pp/SIO.h>
#include	<pp/RDM.h>
#include	<pp/NetConn.h>

#include	"RDM_Protocol.h"

/*----------------------------------------
 *	Equates
 *--------------------------------------*/

#define	MAX_BUF_SIZE	512
#define	DBLog(...)    printf(__VA_ARGS__)
#define DBLog2(...)   printf(__VA_ARGS__)
#define DBLog3(...)   printf(__VA_ARGS__)

/*----------------------------------------
 *	Data Types
 *--------------------------------------*/

/*----------------------------------------
 *	Function Prototypes
 *--------------------------------------*/

DWORD RDM_Protocol_Thread(LPVOID param);

/*----------------------------------------
 *	static data
 *--------------------------------------*/

extern CRDM *rdm;

/*----------------------------------------
 *	CRDM_Protocol_Factory Class
 *--------------------------------------*/

/*  --------------------------------------------------------------------*/
void CRDM_Protocol_Factory::AcceptClientSocket(CSession *pSession, CNetConn *pNetConn, int NOTUSED(ipAddress), int NOTUSED(tcpPort))
{
    printf("CRDM_Protocol_Factory::AcceptClientSocket()\n");
	CRDM_Protocol * p = new CRDM_Protocol( pNetConn, pSession );
	assert( p != NULL );
	if (p == NULL)
	{
		pSession->Release();
		pNetConn->Shutdown();
	}
}

/*----------------------------------------
 *	CRDM_Protocol Class
 *--------------------------------------*/

/*  --------------------------------------------------------------------*/
CRDM_Protocol::CRDM_Protocol( CNetConn *pNewNetConn, CSession * pNewSession )
{
	assert(pNewNetConn != NULL);

	pNetConn = pNewNetConn;

	if (pNewSession != NULL) {
		pSession = pNewSession;
		pSession->AddSessionObject( this );
		pSession->Event( pSession, SESSION_EVENT_CONNECT, "RDM" );
	}
	else {
		pSession = NULL;
	}

	// Create a thread to process this RDM connection

	hThread = OS_CreateThread( (void *) RDM_Protocol_Thread, 0, this );
	assert( hThread != NULL );
	if (hThread == NULL)
	{
		if (pSession != NULL)
			pSession->Release();
		pNetConn->Shutdown();
	}
}

/*  --------------------------------------------------------------------*/
CRDM_Protocol::~CRDM_Protocol()
{
	pNetConn->Shutdown(); // Causes Process() to exit
	if (hThread != NULL)
		OS_DeleteThread(hThread);
}

/*  --------------------------------------------------------------------*/
void CRDM_Protocol::Event(CSession * NOTUSED(pUserSession), int event, const void * NOTUSED(pData))
{
	if (event == SESSION_EVENT_CLOSE)
		pNetConn->Shutdown(); // Causes Process() to exit
}

/*  --------------------------------------------------------------------*/
void CRDM_Protocol::Process()
{
	BOOL		bOk;
	int			result;
	int			length,x;
	CSIO_DMEM	in;
	CSIO_DMEM	out;
	char		buffer[MAX_BUF_SIZE+1];

	// Loop and process RDM requests

	do
	{
		// ---------------------------------------
		// Read request

		// Get the length

		bOk = pNetConn->Read( (BYTE *) &length, 4 );
		if (!bOk)
			goto EXIT;

		length = ntohl(length) - 4;

		if (length < 5)
		{
			DBLog(("RDM Request length < 5\n"));
			goto EXIT;
		}

		// ---------------------------------------
		// Read in the request data

		in.Seek(SEEK_SET,0);

		DBLog3(("RDM -------------------------------------------\n"));

		while (length)
		{
			if (length > MAX_BUF_SIZE)
				x = MAX_BUF_SIZE;
			else
				x = length;

			bOk = pNetConn->Read( (BYTE *) buffer, x );
			if (!bOk)
				goto TERMINATED;

			buffer[x] = 0;

			DBLog3((buffer));

			length -= x;

			result = in.Write( buffer, x );

			if (result < 0)
				goto SIOERROR;
		}

		DBLog3(("\n"));

		// ---------------------------------------
		// Process the request

		in.Seek(0,SEEK_SET);
		out.Seek(0,SEEK_SET);

		result = rdm->rdmp.Command( pSession, &in, &out );

		if (result != 0)
			DBLog2("RDM Error = %d\n",result);

		// ---------------------------------------
		// Write the response

		out.Putc(0);	// NULL Terminator

		length = out.Tell();

		out.Seek(0,SEEK_SET);

		x = htonl(length + 4);

		result = pNetConn->Write( (BYTE *) &x, 4 );

        DBLog3(("RDM response ----------------------------------\n"));
		while (length)
		{
			if (length > MAX_BUF_SIZE)
				x = MAX_BUF_SIZE;
			else
				x = length;

			result = out.Read( buffer, x );
			if (result < 0)
				goto SIOERROR;

            buffer[x] = 0;  // terminate with NULL just to make sure
            DBLog3((buffer));

			bOk = pNetConn->Write( (BYTE *) &buffer, x );
			if (!bOk)
				goto TERMINATED;

			length -= x;
		}
        DBLog3(("\n\n"));

	} while (1);

TERMINATED:
	DBLog2(("RDM Connection Terminated\n"));
	goto EXIT;

SIOERROR:
	DBLog2(("RDM SIO error\n"));

EXIT:
	if (pSession != NULL)
	{
		pSession->Event( pSession, SESSION_EVENT_DISCONNECT, "RDM" );
		pSession->Release();
	}
}

/*  --------------------------------------------------------------------*/
/** 
 *	@brief	Glue code to get into the listen thread method.
 * 
 */

DWORD RDM_Protocol_Thread(LPVOID param)
{
	CRDM_Protocol * p = (CRDM_Protocol *) param;

	p->Process();

	return 0;
}


