#include <stdio.h>
#include <time.h>
#include <pp/base.h>
#include "predictor.h"

/* start factors for data size and time per compression
   indices : VSC1: hextile = 0, tight = 1-10
             VSC2: LRLE uncompressed = 0, ZLIBbed LRLE = 1-10
 */

#define COMPRESSION_COUNT	 11
// FIXME: predictor correction factors for VSC2 hw encoding only a rough guess for now
float fv_start_hw[] = { 0.42, 0.165, 0.16, 0.155, 0.15, 0.145, 0.142, 0.14, 0.135, 0.13, 0.125 };
float ft_start_hw[] = { 0.42, 1.20,  1.33, 1.57,  1.81, 2.04,  2.28,  2.52, 4.51,  6.51, 8.5   };
float fv_start_sw[] = { 0.42, 0.165, 0.16, 0.155, 0.15, 0.145, 0.142, 0.14, 0.135, 0.13, 0.125 };
float ft_start_sw[] = { 0.42, 1.20,  1.33, 1.57,  1.81, 2.04,  2.28,  2.52, 4.51,  6.51, 8.5   };

static unsigned long pred_count_pixels(RegionRec *reg);
static inline float calc_damp(float old, float new);

/******************************
 * Exported predictor functions
 *******************************/

int
pred_init(rfb_cl_t * clp)
{
    clp->pred_fv_korr = 1.0;
    clp->pred_ft_korr = 1.0;
    clp->pred_bwidth_state = PRED_BWIDTH_STATE_NONE;
    clp->pred_time    = time(NULL);
    
    return 0;
}

void
pred_cleanup(rfb_cl_t * clp UNUSED)
{
}

/**
 * pred_measure_start
 * start single measurement
 */
void
pred_measure_start(rfb_cl_t * clp, RegionRec *reg)
{
    if (clp->pred_type == PRED_TYPE_NONE) return;
	
    clp->pred_pixels += pred_count_pixels(reg);
    clp->pred_t_ns = 0;
    pp_hrtime_start(&clp->pred_watch_real);
}

/**
 * pred_measure_start
 * stop single measurement, accumulate results
 */
void
pred_measure_stop(rfb_cl_t * clp)
{
    if (clp->pred_type == PRED_TYPE_NONE) return;
    
    if (clp->pred_bwidth_state == PRED_BWIDTH_STATE_NONE) {
	clp->pred_bwidth_state = PRED_BWIDTH_STATE_REQ;
    } else {
	clp->pred_t_real += clp->pred_t_ns / 1000;
    }
}

/**
 * pred_predict_coding
 * process last bunch of measures and predict compression
 */
void
pred_predict_coding(rfb_cl_t * clp, RegionRec *reg)
{    
    int i;
    time_t cur_time;
    float fv_real, ft_real;
    float *fv_start, *ft_start;
    unsigned long v_code, t_code, t_all=0, t_min=0, pixels;

    if ((clp->pred_bwidth_state != PRED_BWIDTH_STATE_DONE) ||
	(clp->pred_type == PRED_TYPE_NONE)) {
	
	return;
    }

    if (clp->pred_type == PRED_TYPE_HW) {
	fv_start = fv_start_hw;
	ft_start = ft_start_hw;
    } else {
	fv_start = fv_start_sw;
	ft_start = ft_start_sw;
    }
    
    cur_time = time(NULL);    
    if (cur_time - clp->pred_time < PRED_INTERVAL) {
	return;
    }
    clp->pred_time = cur_time;
    
    PD("Pixels=%lu, RealT=%lu us, RealV=%lu\n", clp->pred_pixels,
       clp->pred_t_real, clp->pred_v_real);

    fv_real = clp->pred_v_real / (fv_start[clp->pred_code] * clp->pred_pixels);
    ft_real = clp->pred_t_real / (ft_start[clp->pred_code] * clp->pred_pixels);

    PD("FVreal=%f, FTreal=%f\n", fv_real, ft_real);

    clp->pred_fv_korr = calc_damp(clp->pred_fv_korr, fv_real);
    clp->pred_ft_korr = calc_damp(clp->pred_ft_korr, ft_real);

    PD("FV=%f, FT=%f\n", clp->pred_fv_korr, clp->pred_ft_korr);
    
    pixels = pred_count_pixels(reg); // (A)

    for (i=0; i < COMPRESSION_COUNT; ++i) {
	v_code = clp->pred_fv_korr * fv_start[i] * clp->pred_pixels;
	t_code = clp->pred_ft_korr * ft_start[i] * clp->pred_pixels;
	t_all  = (float)v_code/(float)clp->pred_bwidth*1000000.0 + t_code;
	
	PD("%02d: v_code=%lu, t_code=%lu -> t_all=%lu\n",
	   i, v_code, t_code, t_all);

	if (i==0 || t_all < t_min) {
	    t_min = t_all;
	    clp->pred_code  = i;
	}
    }

    PD("Compression %d\n", clp->pred_code);

    if (clp->pred_type == PRED_TYPE_HW) {
	clp->enc.pref_enc = rfbEncodingLRLEHard;
	clp->hwenc_subenc = rfbLRLESubenc15bitDirectLossless;
	clp->hwenc_zlib_level  = clp->pred_code;
    } else {
	if (clp->pred_code == 0) {
	    clp->enc.pref_enc = rfbEncodingHextile;
	    clp->enc.send_rect_softenc_fn = rfb_send_rect_hextile;
	} else {
	    clp->enc.pref_enc = rfbEncodingTight;
	    clp->enc.send_rect_softenc_fn = rfb_send_rect_tight;
	    clp->enc.tightCompressLevel = clp->pred_code - 1;
	}
    }
       
    // clear measured real time/data    
    clp->pred_v_real = 0;
    clp->pred_t_real = 0;
    clp->pred_pixels = 0;    
}

/******************************
 * Internal predictor functions
 *******************************/

static unsigned long
pred_count_pixels(RegionRec *reg)
{
    unsigned long pixels = 0;
    int i;

    for (i = 0; i < REGION_NUM_RECTS(reg); i++) {
	int x = REGION_RECTS(reg)[i].x1;
	int y = REGION_RECTS(reg)[i].y1;
	int w = REGION_RECTS(reg)[i].x2 - x;
	int h = REGION_RECTS(reg)[i].y2 - y;

	pixels += w*h;
    }

    return pixels;
}

static inline
float calc_damp(float old, float new)
{
    return (new - old) * 0.5 + old;
}
