#include <stdio.h>
#include <sys/poll.h>

#include <liberic_net.h>
#include <liberic_misc.h>

#include "rfb.h"
#include "debug.h"

/******************************************************************************
* shared functions                                                            *
******************************************************************************/

/*-----------------------------------------------------------------------------
- do we really need a proxy connection?                                       -
-----------------------------------------------------------------------------*/

int rfb_needs_proxy_connection(rfb_cl_t * clp) {
#if defined PRODUCT_RIPCKIMXN
    return clp->kvm_node != 0;
#else /* PRODUCT_RIPCKIMXN */
    (void)clp;  // shut compiler up
    return 0;
#endif /* PRODUCT_RIPCKIMXN */
}

int rfb_can_be_connected_by_proxy(rfb_cl_t * clp) {
    (void)clp;  // shut compiler up
#if defined PRODUCT_RIPCKIMXN
    return pp_misc_get_kvm_node() != 0;
#else /* PRODUCT_RIPCKIMXN */
    return 0;
#endif /* PRODUCT_RIPCKIMXN */
}

void rfb_proxy_disconnect(rfb_cl_t * clp) {
    if (clp->proxy_bio) {
        eric_net_close(clp->proxy_bio, WAIT);
        clp->proxy_bio = NULL;
        D(D_VERBOSE, "Closed connection to external kvm_node %u\n", clp->kvm_node);
    }
}

/******************************************************************************
* dummy functions for devices which don't support the proxying                *
******************************************************************************/

#if !defined PRODUCT_RIPCKIMXN

int rfb_proxy_connect_to_channel(rfb_cl_t * clp UNUSED) {
    assert(0);
    return -1;
}

void rfb_proxy_run(rfb_cl_t * clp UNUSED) {
    assert(0);
}

#else /* PRODUCT_RIPCKIMXN */

/******************************************************************************
* functions for devices which support the proxying                            *
******************************************************************************/

/*-----------------------------------------------------------------------------
- constants and local data, function prototypes                               -
-----------------------------------------------------------------------------*/

#define PROXY_CONNECT_TIMEOUT   10  // seconds

static int rfb_proxy_negotiate_protocol_version(rfb_cl_t * clp);
static int rfb_proxy_authenticate_client(rfb_cl_t * clp);
static int rfb_proxy_send_client_init(rfb_cl_t * clp);
static int rfb_proxy_handle_osd_message(rfb_cl_t * clp);
static int rfb_proxy_handle_utf8_message(rfb_cl_t * clp, int do_send);
static int rfb_proxy_handle_keyboard_layout(rfb_cl_t * clp);
static int rfb_proxy_handle_connection_parameters(rfb_cl_t * clp);
static int rfb_proxy_handle_fb_format(rfb_cl_t * clp);

static struct {
    const char *ip;
    const char *port;
} proxy_channels[] = {
    { NULL, NULL },         // channel 0 not supported!
    {
        ip:     "127.1.0.2",
        port:   "443"
    }
};

static size_t no_proxy_channels = sizeof(proxy_channels) / sizeof(proxy_channels[0]);

/*-----------------------------------------------------------------------------
- proxy connect function                                                      -
-----------------------------------------------------------------------------*/

#define CLIENT  clp->current_bio = clp->conn_data->bio
#define PROXY   clp->current_bio = clp->proxy_bio
  
int rfb_proxy_connect_to_channel(rfb_cl_t * clp) {
    int ret = -1;
    u_int8_t type = 0;
    
    D(D_NOTICE, "Connecting to channel %u\n", clp->kvm_node);
    
    /* check for wrong parameters */
    if (clp->kvm_node <= 0 || clp->kvm_node >= no_proxy_channels) {
        D(D_ERROR, "channel ID (%i) out of range\n", clp->kvm_node);
        rfb_send_quit(clp, rfbQuitKVMPortDenied);
        return -1;
    }
    
    /* connect proxy to the external port */
    if ((clp->proxy_bio = eric_net_open_connection(proxy_channels[clp->kvm_node].ip, proxy_channels[clp->kvm_node].port, PROXY_CONNECT_TIMEOUT)) == NULL) {
        D(D_ERROR, "Could not connect proxy to external channel %u\n", clp->kvm_node);
        goto bail;
    }
    D(D_VERBOSE, "Successfully connected to external channel %u\n", clp->kvm_node);
    
    /* establish RFB connection */
    PROXY;
    if (rfb_send_connection_init_string(clp) == -1 ||
        rfb_proxy_negotiate_protocol_version(clp) == -1 ||
        rfb_proxy_authenticate_client(clp) == -1 ||
        rfb_proxy_send_client_init(clp) == -1) {
        
        CLIENT;
        rfb_send_quit(clp, rfbQuitInternalError);
        goto bail;
    }

    while ((type = rfb_read_message_type(clp)) == rfbOSDState) {
        if (rfb_proxy_handle_osd_message(clp) == -1) {
            CLIENT;
            rfb_send_quit(clp, rfbQuitInternalError);
            goto bail;
        }
    }
    
    if (rfb_proxy_handle_utf8_message(clp, 0) == -1||
        rfb_read_message_type(clp) != rfbOSDState ||
        rfb_proxy_handle_osd_message(clp) == -1 ||
        rfb_read_message_type(clp) != rfbKeyboardLayoutEvent ||
        rfb_proxy_handle_keyboard_layout(clp) == -1 ||
        rfb_proxy_handle_connection_parameters(clp) == -1 ||
        rfb_read_message_type(clp) != rfbServerFBFormat ||
        rfb_proxy_handle_fb_format(clp) == -1) {
        
        CLIENT;
        rfb_send_quit(clp, rfbQuitInternalError);
        goto bail;
    }
    
    D(D_VERBOSE, "Successfully started RFB connection to external channel %u\n", clp->kvm_node);
    ret = 0;
    
 bail:
    return ret;
}

/*-----------------------------------------------------------------------------
- proxy run function                                                          -
-----------------------------------------------------------------------------*/

#define PROXY_BUF_SIZE 1024

void rfb_proxy_run(rfb_cl_t * clp) {
    char buf[PROXY_BUF_SIZE];
    int n, buf_len = PROXY_BUF_SIZE;
    int poll_val;
    
    BIO* clt_bio = clp->conn_data->bio;
    int clt_fd = BIO_get_fd(clt_bio, NULL);
    BIO* proxy_bio = clp->proxy_bio;
    int proxy_fd = BIO_get_fd(proxy_bio, NULL);

    struct pollfd fds[2] = {
	{ clt_fd, POLLIN, 0 },
	{ proxy_fd, POLLIN, 0 }
    };
        
    while(1) {
    	poll_val = poll(fds, 2, -1);

    	if(poll_val < 0) {
    	    D(D_ERROR, "poll() failed: %s\n", strerror(errno));
    	    break;
    	}
    	
    	// data from the client to the proxied connection
    	if(fds[0].revents & POLLIN) {
            if ((n = eric_net_read(clt_bio, buf, buf_len, 1)) < 0 || eric_net_write(proxy_bio, buf, n, 1) < 0) {
		break;
	    }
    	}
    	
    	// data from the proxied connection to the client
    	if(fds[1].revents & POLLIN) {
            if((n = eric_net_read(proxy_bio, buf, buf_len, 1)) < 0 || rfb_writer_enqueue_pdu(clp, buf, n) < 0) {
		break;
	    }
	}
    }
}

/*-----------------------------------------------------------------------------
- internal functions for initial connection establishing                      -
-----------------------------------------------------------------------------*/

static int rfb_proxy_negotiate_protocol_version(rfb_cl_t * clp) {
    int major, minor;
    
    if (rfb_read_protocol_version(clp, &major, &minor) == -1)
        return -1;

    D(D_NOTICE, "Client protocol version %d.%d\n", major, minor);

    if (major != rfbProtocolMajorVersion || minor != rfbProtocolMinorVersion) {
	D(D_ERROR, "Protocol version mismatch\n");
	return -1;
    }

    if (rfb_send_protocol_version(clp, rfbProtocolMajorVersion, rfbProtocolMinorVersion) == -1)
        return -1;

    return 0;
}

static int rfb_proxy_authenticate_client(rfb_cl_t * clp) {
    unsigned caps;
    u_int8_t type;
    u_int32_t flags;
    
    if (rfb_read_auth_caps(clp, &caps) == -1) {
        return -1;
    }
    
    if ((caps & rfbAuthNoAuth) == 0) {
        D(D_ERROR, "Could not connect to secondary channel without authentication.\n");
        return -1;
    }

    if (rfb_send_login_pdu(clp, "", rfbAuthNoAuth) == -1 ||
        rfb_read_auth_challenge(clp, NULL, 0) == -1 ||
        rfb_send_auth_response_pdu(clp, NULL, 0) == -1) {
        
        return -1;
    }
    
    type = rfb_read_message_type(clp);
    if (type == rfbQuit) {
        u_int8_t reason = rfbQuitInternalError;
        rfb_read_quit(clp, &reason);
        return -1;
    } else if (type != rfbAuthSuccessful) {
        return -1;
    }
    
    rfb_read_auth_successful(clp, &flags);
    
    return 0;
}

static int rfb_proxy_send_client_init(rfb_cl_t * clp) {
    return rfb_send_init_pdu(clp, 0 /* kvm_node */, 0 /* unit */, 0 /* port */,
	    rfbinitMsgFlagsNone );
}

static int rfb_proxy_handle_osd_message(rfb_cl_t * clp) {
    char *msg = NULL;
    u_int8_t blank;
    u_int16_t timeout;
    struct timeval tv;
    
    int ret = (rfb_read_osd_state(clp, &msg, &blank, &timeout, &tv) == -1 ||
        rfb_enqueue_osd_state_time(clp, msg, blank, timeout, &tv) == -1) ? -1 : 0;

    if (msg) free(msg);

    return ret;
}

static int rfb_proxy_handle_utf8_message(rfb_cl_t * clp, int do_send) {
    char *msg = NULL;
    int len = 0;
    int ret = 0;
    
    if (rfb_read_utf8string_pdu(clp, &msg, &len) == -1) {
        if (msg) free(msg);
        return -1;
    }
    
    if (do_send) {
        ret = rfb_enqueue_utf8string(clp, msg, len);
    }
    if (msg) free(msg);
    return ret;
}

static int rfb_proxy_handle_keyboard_layout(rfb_cl_t * clp) {
    char *kbd = NULL;
    
    int ret = (rfb_read_kbd_layout(clp, &kbd) == -1 ||
        rfb_enqueue_kbd_layout(clp, kbd) == -1) ? -1 : 0;

    if (kbd) free(kbd);
    return ret;
}

static int rfb_proxy_handle_connection_parameters(rfb_cl_t * clp) {
    rfb_connection_parameter_list_t *list = rfb_read_connection_parameters(clp);
    if (list) {
        rfb_free_connection_parameter_list(list);
        return 0;
    } else {
        return -1;
    }
}

static int rfb_proxy_handle_fb_format(rfb_cl_t * clp) {
    int fb_width, fb_height;
    u_int8_t unsupported;
    rfbPixelFormat fmt;
    struct timeval tv;
    
    return (rfb_read_fb_format_update(clp, &fb_width, &fb_height, &unsupported, &fmt, &tv) == -1 ||
            rfb_enqueue_fb_format_update_params(clp, fb_width, fb_height, unsupported, &fmt, &tv) == -1) ? -1 : 0;
}

#endif /* PRODUCT_RIPCKIMXN */
