/*
 * raw.c
 *
 * Routines to implement Raw / RawVSC Encoding
 */

/*
 *  Copyright (C) 1999 AT&T Laboratories Cambridge.  All Rights Reserved.
 *
 *  This is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This software is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,
 *  USA.
 */

#include <stdio.h>
#include <netinet/in.h>
#include <rfb.h>

/*
 * Send a given rectangle in raw encoding (rfbEncodingRaw).
 */

static int
encode_rect_raw(rfb_cl_t * clp, int x, int y, int w, int h)
{
    int n_lines;
    int bytes_per_line = w * (clp->pix_fmt.bitsPerPixel_8 / 8);
    char * fb_ptr = GET_PTR_INTO_FB(clp, x, y);

    if (clp->ub_len + 16 * bytes_per_line > UPDATE_BUF_SIZE) {
	if (rfb_send_update_buf(clp) == -1) {
	    return -1;
	}
    }

    n_lines = (UPDATE_BUF_SIZE - clp->ub_len) / bytes_per_line;

    while (1) {
	if (n_lines > h) {
	    n_lines = h;
	}
	
    	/* the pointer into the framebuffer must be 16 aligned */
	if(n_lines > 16) {
	    n_lines = (n_lines / 16) * 16;
	}

	clp->translate_fn(clp->translate_lookup_table,
			  &rfb_pix_fmt, &clp->pix_fmt,
			  fb_ptr, &clp->update_buf[clp->ub_len],
			  clp->fb_tile_pitch, w, n_lines);

	clp->ub_len += n_lines * bytes_per_line;
	y += n_lines;
	h -= n_lines;

	if (h == 0) {	/* rect fitted in buffer, do next one */
	    return 0;
	}

	/* buffer full - flush partial rect and do another nlines */

	if (rfb_send_update_buf(clp) == -1) {
	    return -1;
	}

	fb_ptr = GET_PTR_INTO_FB(clp, x, y);

	n_lines = (UPDATE_BUF_SIZE - clp->ub_len) / bytes_per_line;
	if (n_lines == 0) {
	    pp_log("rfb_send_rect_raw(): Send buffer too small "
		     "for %d bytes per line\n", bytes_per_line);
	    return -1;
	}
    }
}

int
rfb_send_rect_raw(rfb_cl_t * clp, int x, int y, int w, int h)
{
    char *rect;
    size_t size;

#ifdef TIME_MEASURE
    clp->time_pixel += w*h;
#endif

    rect = rfb_create_fb_update_rect_header(x, y, w, h, rfbEncodingRaw, &size);
    if (!rect || size == 0) {
    	return -1;
    }

    if (clp->ub_len + size > UPDATE_BUF_SIZE) {
	if (rfb_send_update_buf(clp) == -1) {
	    free(rect);
	    return -1;
	}
    }

    memcpy(&clp->update_buf[clp->ub_len], rect, size);
    clp->ub_len += size;
    
    free(rect);
    return encode_rect_raw(clp, x, y, w, h);
}


/*
 * Send a given rectangle in Raw VSC encoding (rfbEncodingRawVSC).
 */

void
rfb_translate_none_rawvsc(char * table UNUSED, rfbPixelFormat * in UNUSED,
		                rfbPixelFormat * out, char * iptr, char * optr,
		                u_char tile_pitch, int width, int height)
{
    int bytesPerOutputLine = width * (out->bitsPerPixel_8 / 8);

    while (height > 0) {
    	int linesToCopy = min(height, PP_FB_TILE_HEIGHT);
    	int bytesToCopy = linesToCopy * bytesPerOutputLine;
	memcpy(optr, iptr, bytesToCopy);
	iptr += tile_pitch * PP_FB_TILE_SIZE * (out->bitsPerPixel_8 / 8);
	optr += bytesToCopy;
	height -= PP_FB_TILE_HEIGHT;
    }
}

/* this function sends a raw screen update without
   linearising the framebuffer; this should be done by the client */
static int
encode_rect_rawvsc(rfb_cl_t * clp, int x, int y, int w, int h)
{
    int n_lines;
    int bytes_per_line = w * (clp->pix_fmt.bitsPerPixel_8 / 8);
    char * fb_ptr = GET_PTR_INTO_FB(clp, x, y);

    if (clp->ub_len + 16 * bytes_per_line > UPDATE_BUF_SIZE) {
	if (rfb_send_update_buf(clp) == -1) {
	    return -1;
	}
    }

    n_lines = (UPDATE_BUF_SIZE - clp->ub_len) / bytes_per_line;
    
    while (1) {
	if (n_lines > h) {
	    n_lines = h;
	}
	
    	/* the pointer into the framebuffer must be 16 aligned */
	if(n_lines > 16) {
	    n_lines = (n_lines / 16) * 16;
	}
	clp->rawvsc_translate_fn(clp->translate_lookup_table,
				 &rfb_pix_fmt, &clp->pix_fmt,
				 fb_ptr, &clp->update_buf[clp->ub_len],
				 clp->fb_tile_pitch, w, n_lines);

	clp->ub_len += n_lines * bytes_per_line;
	y += n_lines;
	h -= n_lines;

	if (h == 0) {	/* rect fitted in buffer, do next one */
	    return 0;
	}

	/* buffer full - flush partial rect and do another nlines */

	if (rfb_send_update_buf(clp) == -1) {
	    return -1;
	}

	fb_ptr = GET_PTR_INTO_FB(clp, x, y);

	n_lines = (UPDATE_BUF_SIZE - clp->ub_len) / bytes_per_line;
	if (n_lines == 0) {
	    pp_log("rfb_send_rect_raw(): Send buffer too small "
		     "for %d bytes per line\n", bytes_per_line);
	    return -1;
	}
    }
}

static int
send_rawvsc_header(rfb_cl_t * clp, int x, int y, int w, int h, u_int8_t flags) {
    char *rect = NULL;
    char *hdr = NULL;
    size_t size1, size2;
    int ret = -1;

#ifdef TIME_MEASURE
    clp->time_pixel += w*h;
#endif

    rect = rfb_create_fb_update_rect_header(x, y, w, h, rfbEncodingRawVSC, &size1);
    if (!rect || size1 == 0) {
    	goto bail;
    }
    
    hdr = rfb_create_raw_vsc_hdr(flags, &size2);
    if (!hdr || size1 == 0) {
    	goto bail;
    }

    if (clp->ub_len + size1 + size2 > UPDATE_BUF_SIZE) {
	if (rfb_send_update_buf(clp) == -1) {
	    goto bail;
	}
    }
    
    memcpy(&clp->update_buf[clp->ub_len], rect, size1);
    clp->ub_len += size1;
    
    memcpy(&clp->update_buf[clp->ub_len], hdr, size2);
    clp->ub_len += size2;
    
    ret = 0;
    
 bail:
    free(rect);
    free(hdr);
    
    return ret;
}

int
rfb_send_rect_rawvsc(rfb_cl_t * clp, int x, int y, int w, int h)
{
    u_int8_t flags = 0;
    int ret = 0;
    rfb_send_rect_softenc_fn_t encode_rect_fn;
    
    if(clp->rawvsc_translate_fn == NULL) {
    	// no translation function found --> send a normal raw encoded rect
    	flags &= ~rfbRawVSCIsVSC;
    	encode_rect_fn = encode_rect_raw;
    }
    else {
    	flags |= rfbRawVSCIsVSC;
    	h = ((h + 15) / 16) * 16;
    	if(clp->rawvsc_use_serverpixelformat) {
    	    flags |= rfbRawVSCUseServersPixelFormat;
    	}
    	else {
    	    flags &= ~rfbRawVSCUseServersPixelFormat;
    	}
#if (__BYTE_ORDER == __BIG_ENDIAN)
    	flags |= rfbRawVSCBigEndian;
#else // (__BYTE_ORDER != __BIG_ENDIAN)
    	flags &= ~rfbRawVSCBigEndian;
#endif // BIG_ENDIAN
    	encode_rect_fn = encode_rect_rawvsc;
    }
    
    if((ret = send_rawvsc_header(clp, x, y, w, h, flags)) != 0) {
    	return ret;
    }
    
    return encode_rect_fn(clp, x, y, w, h);
}
