/* this file includes all functions which deal with the session redirector feature */

#include <stdio.h>
#include <sys/time.h>
#include <time.h>
#include <pp/base.h>
#include <pp/rfb.h>
#include <liberic_net.h>
#include <liberic_pthread.h>
#include "rfb.h"
#include "debug.h"

/*
#undef DEBUGLEVEL
#define DEBUGLEVEL D_BLABLA
*/

#if defined(PP_FEAT_SESSION_REDIRECTOR)

static void rfb_sas_send_new_kvm_session_v(rfb_cl_t * clp, va_list args);
static void rfb_sas_send_kvm_session_closed_v(rfb_cl_t * clp, va_list args);
static void rfb_sas_send_kvm_exclusive_v(rfb_cl_t * clp, va_list args);
static void rfb_sas_print_connection_flags(const char *string, u_int32_t flags);
static void* rfb_sas_input_thread(void *_clp);
static void rfb_sas_enqueue_input_event_v(rfb_cl_t * clp, va_list args);

#endif /* PP_FEAT_SESSION_REDIRECTOR */

u_int32_t rfb_adjust_connection_flags(rfb_cl_t *clp, u_int32_t flags) {

#if defined(PP_FEAT_SESSION_REDIRECTOR)

    rfb_sas_print_connection_flags("Requested connection flags", flags);
    
    // clear the "replayer" flag
    if (flags & rfbConnectionFlagSP) {
    	flags &= ~rfbConnectionFlagSP;
    }

    /* use only local events if KVM-IP is disabled */
#if defined(PP_FEAT_DISABLE_KVM_OVER_IP)
    if (rfb_kvm_ip_disabled()) {
    	flags |= rfbConnectionFlagSasLocalOnly;
    }
#endif /* !defined(PP_FEAT_DISABLE_KVM_OVER_IP) */

    rfb_sas_print_connection_flags("Actual connection flags", flags);

#else /* PP_FEAT_SESSION_REDIRECTOR */

    flags = 0;
    
#endif /* PP_FEAT_SESSION_REDIRECTOR */

    /* save the actual set flags */
    clp->connection_flags = flags;
    return flags;
}


#if defined(PP_FEAT_SESSION_REDIRECTOR)

/* ---- initialize/cleanup routines for SAS connections ---- */

int rfb_sas_init(rfb_cl_t * clp) {
    if (clp->connection_flags & (rfbConnectionFlagSasHW | rfbConnectionFlagSasSW)) {
    	D(D_BLABLA, "initialize SAS for client %p...\n", clp);

    	if ((clp->sas_input_queue = pp_queue_alloc(QUEUE_TYPE_STANDARD)) == NULL) {
	    D(D_ERROR, "%s(): pp_queue_alloc() failed.\n", ___F);
	    return -1;
    	}

    	if (pthread_mutex_init(&clp->sas_input_queue_mtx, &recursive_mtx_attr) ||
    	    pthread_cond_init(&clp->sas_input_queue_cond, NULL)) {
    	    
    	    return -1;
    	}

    	/* start an input processing thread */
    	clp->sas_exit_input_thd = 0;
    	if (eric_pthread_create(&clp->sas_input_thd, 0, 128 * 1024,
				rfb_sas_input_thread, clp) != 0) {
	    D(D_ERROR, "%s(): Cannot create SAS input thread thread.\n", ___F);
	    return -1;
    	}
    	
    	clp->modifier_vector = vector_new(NULL, 3, NULL);
    	clp->unfiltered_keys_vector = vector_new(NULL, 3, NULL);
    	
    	D(D_BLABLA, "SAS initialization for client %p done\n", clp);
    	return 0;
    } else {
    	return 0;
    }
}

void rfb_sas_cleanup(rfb_cl_t * clp) {
    if (clp->connection_flags & (rfbConnectionFlagSasHW | rfbConnectionFlagSasSW)) {
    	D(D_BLABLA, "cleanup SAS for client %p...\n", clp);

    	/* stop input processing thread */
    	MUTEX_LOCK(&clp->sas_input_queue_mtx);
    	clp->sas_exit_input_thd = 1;
    	if (pthread_cond_signal(&clp->sas_input_queue_cond) != 0) {
	    D(D_ERROR, "%s(): pthread_cond_signal() failed\n", ___F);
    	}
    	MUTEX_UNLOCK(&clp->sas_input_queue_mtx);

    	/* wait for input processing thread's death */
    	pthread_join(clp->sas_exit_input_thd, NULL);
    	
    	pp_queue_free(clp->sas_input_queue);
    	pthread_mutex_destroy(&clp->sas_input_queue_mtx);
    	pthread_cond_destroy(&clp->sas_input_queue_cond);
    	
    	vector_delete(clp->modifier_vector);
    	vector_delete(clp->unfiltered_keys_vector);
    	
    	D(D_BLABLA, "SAS cleanup for client %p done\n", clp);
    }
}

/* ---- set and adjust the requested connection type ---- */

static void rfb_sas_print_connection_flags(const char *string, u_int32_t flags) {
    pp_strstream_t txt = PP_STRSTREAM_INITIALIZER;
    if (flags) {
        if (flags & rfbConnectionFlagSasHW) {
    	    pp_strappend(&txt, "SAS_HW ");
        }
        if (flags & rfbConnectionFlagSasSW) {
    	    pp_strappend(&txt, "SAS_SW ");
        }
        if (flags & rfbConnectionFlagSP) {
    	    pp_strappend(&txt, "SP ");
        }
        if (flags & rfbConnectionFlagSasLocalOnly) {
    	    pp_strappend(&txt, "LOCAL ");
        }
        switch (rfbConnectionFlagGetKbd(flags)) {
            case rfbConnectionSasKbdNormal:
            	pp_strappend(&txt, "KBD_NORM ");
            	break;
            case rfbConnectionSasKbdNoKbd:
            	pp_strappend(&txt, "KBD_NO_KBD ");
            	break;
            case rfbConnectionSasKbdFilter:
            	pp_strappend(&txt, "KBD_FILTER ");
            	break;
            case rfbConnectionSasKbdAllFilterPb:
            	pp_strappend(&txt, "KBD_ALL_FILTER_PB ");
            	break;
        }
    } else {
        pp_strappend(&txt, "normal");
    }
    D(D_VERBOSE, "%s: %s\n", string, pp_strstream_buf(&txt));
    pp_strstream_free(&txt);
}

/* ---- send some stuff to all SAS clients ---- */

void rfb_for_all_sas_clients(int channel, void (*callback)(rfb_cl_t *, va_list), ...) {
    va_list    args;
    rfb_cl_t * clp;

    MUTEX_LOCK(&clients_head_mtx);

    for (clp = clients_head; clp; clp = clp->next) {
	if (channel != -1 && clp->video_link != (unsigned int)channel) {
	    continue;
	}
	if (clp->connection_flags & (rfbConnectionFlagSasHW | rfbConnectionFlagSasSW)) {
	    va_start(args, callback);
	    callback(clp, args);
	    va_end(args);
	}
    }

    MUTEX_UNLOCK(&clients_head_mtx);
}

static void rfb_sas_send_open_session(eric_session_int_id_t s, va_list args) {
    rfb_cl_t *clp = va_arg(args, rfb_cl_t *);
    
    const char *user = eric_session_get_user(s);
    const char *ip = eric_session_get_ip_str(s);
    time_t create_time = eric_session_get_creation_time(s);
    int self = clp->session == s;
    pp_rfb_session_data_t **sd = rfb_get_session_data_from_session(s);
    size_t no_kvm = 0;
    int *exclusive = NULL;
    rfb_cl_t **clps = NULL;
    time_t *times = NULL;
    unsigned int i;
    int excl_index = -1;
    
    MUTEX_LOCK(&exclusive_mode_mtx);
    for (i = 0; sd && sd[i]; i++) {
    	no_kvm++;
    	if (exclusive_clp == sd[i]->clp) {
    	    excl_index = i;
    	}
    }
    
    if (no_kvm) {
    	exclusive = malloc(no_kvm * sizeof(int));
    	memset(exclusive, 0, no_kvm * sizeof(int));
    	if (excl_index != -1) {
    	    exclusive[excl_index] = 1;
    	}
    	
    	clps = malloc(no_kvm * sizeof(rfb_cl_t *));
    	memset(clps, 0, no_kvm * sizeof(rfb_cl_t *));
    	for (i = 0; i < no_kvm; i++) {
    	    clps[i] = sd[i]->clp;
    	}
    	
    	times = malloc(no_kvm * sizeof(time_t *));
    	memset(times, 0, no_kvm * sizeof(time_t *));
    	for (i = 0; i < no_kvm; i++) {
    	    times[i] = sd[i]->session_time;
    	}
    }
    MUTEX_UNLOCK(&exclusive_mode_mtx);

    D(D_BLABLA, "Found open session for client %p:\n"
    		"user: %s ip: %s, no_kvm: %u, time: %lu, self: %s\n",
    		clp, user, ip, no_kvm, create_time, self ? "yes" : "no");
    
    /* don't send own session to SAS HW */
    if (! ((clp->connection_flags & rfbConnectionFlagSasHW) && clp->session == s) ) {
    	unsigned int j;
    	
    	rfb_enqueue_sas_existing_session(clp, user, ip, self, create_time, s);
    	for (j = 0; j < no_kvm; j++) {
    	    rfb_enqueue_sas_existing_kvm_session(clp, s, clps[j], exclusive[j], times[j]);
    	}
    }
    
    free(clps);
    free(sd);
    free(exclusive);
    free(times);
}

int rfb_sas_send_open_sessions(rfb_cl_t * clp) {
    /* don't send if local only flag is set */
    if (clp->connection_flags & rfbConnectionFlagSasLocalOnly) {
    	return 0;
    }
    
    D(D_BLABLA, "Sending open sessions to client %p\n", clp);
    
    eric_session_for_all(rfb_sas_send_open_session, clp);
    return 0;
}

void rfb_sas_send_new_kvm_session(rfb_cl_t * new_clp) {
    rfb_for_all_sas_clients(new_clp->video_link, rfb_sas_send_new_kvm_session_v, new_clp);
}

void rfb_sas_send_closed_kvm_session(rfb_cl_t * closed_clp) {
    rfb_for_all_sas_clients(closed_clp->video_link, rfb_sas_send_kvm_session_closed_v, closed_clp);
}

void rfb_sas_send_kvm_exclusive(rfb_cl_t * excl_clp, int exclusive) {
    rfb_for_all_sas_clients(excl_clp->video_link, rfb_sas_send_kvm_exclusive_v, excl_clp, exclusive);
}

void rfb_sas_send_kbd_event(u_int8_t keycode, eric_session_int_id_t session, rfb_cl_t *causing_clp) {
    input_event_t event;
    
    memset(&event, 0, sizeof(event));
    event.type = rfbKeyEvent;
    event.session = session;
    event.clp = causing_clp;
    event.data.keycode = keycode;
    gettimeofday(&event.arrived, NULL);
    
    rfb_for_all_sas_clients(causing_clp ? (int)causing_clp->video_link : -1, rfb_sas_enqueue_input_event_v, &event);
}

void rfb_sas_send_mouse_event(u_int16_t x, u_int16_t y, u_int16_t z,
			      u_int8_t buttons, u_int8_t type,
			      eric_session_int_id_t session,
			      rfb_cl_t *causing_clp) {
    input_event_t event;
    
    memset(&event, 0, sizeof(event));
    event.type = type;
    event.session = session;
    event.clp = causing_clp;
    event.data.mouse_event.x = x;
    event.data.mouse_event.y = y;
    event.data.mouse_event.z = z;
    event.data.mouse_event.button_mask = buttons;
    gettimeofday(&event.arrived, NULL);
    
    rfb_for_all_sas_clients(causing_clp->video_link, rfb_sas_enqueue_input_event_v, &event);
}

void rfb_sas_send_mousesync_event(u_int8_t type, eric_session_int_id_t session, rfb_cl_t *causing_clp) {
    input_event_t event;
    
    memset(&event, 0, sizeof(event));
    event.type = rfbMouseSyncEvent;
    event.session = session;
    event.clp = causing_clp;
    event.data.sync_type = type;
    gettimeofday(&event.arrived, NULL);
    
    rfb_for_all_sas_clients(causing_clp->video_link, rfb_sas_enqueue_input_event_v, &event);
}

/* ---- va functions ---- */

void rfb_sas_login_failure_v(rfb_cl_t * clp, va_list args) {
    const char * user = va_arg(args, const char *);
    const char * ip = va_arg(args, const char *);
    
    /* don't send if local only flag is set */
    if (clp->connection_flags & rfbConnectionFlagSasLocalOnly) {
    	return;
    }

    D(D_BLABLA, "Login failure for client %p:\n"
    		"user: %s, ip: %s\n",
    		clp, user, ip);
    
    /* send to clp */
    if (rfb_enqueue_sas_login_failure_event(clp, user, ip)) {
    	rfb_enqueue_sas_error(clp, rfbSasErrorMissedSessionEvent);
    }
}

void rfb_sas_send_opened_web_session_v(rfb_cl_t * clp, va_list args) {
    eric_session_int_id_t s = va_arg(args, eric_session_int_id_t);
    const char *user = eric_session_get_user(s);
    const char *ip = eric_session_get_ip_str(s);
    time_t create_time = eric_session_get_creation_time(s);
    
    /* don't send if local only flag is set */
    if (clp->connection_flags & rfbConnectionFlagSasLocalOnly) {
    	return;
    }

    D(D_BLABLA, "Session opened for client %p:\n"
    		"user: %s, ip: %s, session: %d\n",
    		clp, user, ip, s);
    
    /* send to clp */
    if (rfb_enqueue_sas_new_session_event(clp, user, ip, create_time, s)) {
    	rfb_enqueue_sas_error(clp, rfbSasErrorMissedSessionEvent);
    }
}

void rfb_sas_send_closed_web_session_v(rfb_cl_t * clp, va_list args) {
    eric_session_int_id_t s = va_arg(args, eric_session_int_id_t);
    
    /* don't send if local only flag is set */
    if (clp->connection_flags & rfbConnectionFlagSasLocalOnly) {
    	return;
    }

    D(D_BLABLA, "Session closed for client %p: session: %d\n", clp, s);
    
    /* send to clp */
    if (rfb_enqueue_sas_generic_event(clp, rfbSasEventUserSessionClosed, s)) {
    	rfb_enqueue_sas_error(clp, rfbSasErrorMissedSessionEvent);
    }
}

static void rfb_sas_send_new_kvm_session_v(rfb_cl_t * clp, va_list args) {
    rfb_cl_t * new_clp = va_arg(args, rfb_cl_t *);
    
    /* don't send if local only flag is set */
    if (clp->connection_flags & rfbConnectionFlagSasLocalOnly) {
    	return;
    }

    if (!new_clp) return;
    
    D(D_BLABLA, "New KVM Session for client %p: session: %d\n",
    		clp, clp->session);
    
    /* send to clp */
    if (rfb_enqueue_sas_new_kvm_session(clp, new_clp->session, new_clp, new_clp->session_time)) {
    	rfb_enqueue_sas_error(clp, rfbSasErrorMissedSessionEvent);
    }
}

static void rfb_sas_send_kvm_session_closed_v(rfb_cl_t * clp, va_list args) {
    rfb_cl_t * closed_clp = va_arg(args, rfb_cl_t *);
    
    /* don't send if local only flag is set */
    if (clp->connection_flags & rfbConnectionFlagSasLocalOnly) {
    	return;
    }

    if (!closed_clp) return;
    
    D(D_BLABLA, "Closed KVM Session for client %p: session: %d\n",
    		clp, clp->session);
    
    /* send to clp */
    if (rfb_enqueue_sas_kvm_session_event(clp,
    	rfbSasEventKvmSessionClosed, closed_clp->session, closed_clp)) {
    	
    	rfb_enqueue_sas_error(clp, rfbSasErrorMissedSessionEvent);
    }
}

static void rfb_sas_send_kvm_exclusive_v(rfb_cl_t * clp, va_list args) {
    rfb_cl_t * excl_clp = va_arg(args, rfb_cl_t *);
    int excl = va_arg(args, int);

    /* don't send if local only flag is set */
    if (clp->connection_flags & rfbConnectionFlagSasLocalOnly) {
    	return;
    }

    if (!excl_clp) return;
    
    D(D_BLABLA, "%s KVM Session exclusive for client %p: session: %d\n",
    		excl ? "Entered" : "Left", clp, clp->session);
    
    /* send to clp */
    if (rfb_enqueue_sas_kvm_session_event(clp,
    	excl ? rfbSasEventKvmExclusiveOn : rfbSasEventKvmExclusiveOff, excl_clp->session, excl_clp)) {
    	
    	rfb_enqueue_sas_error(clp, rfbSasErrorMissedSessionEvent);
    }
}

static int key_pos_in_vector(vector_t *vector, int keycode) {
    unsigned int i;
    
    for (i = 0; i < vector_size(vector); i++) {
        if (vector_get_int(vector, i) == keycode) {
            return i;
        }
    }
    
    return -1;
}

#define IS_FILTER_MODIFIER(key)		\
    ((key) == 54  || /* left Ctrl */	\
     (key) == 55  || /* left Alt */	\
     (key) == 57  || /* right Alt */	\
     (key) == 58  || /* right Ctrl */	\
     (key) == 105 || /* left Win */	\
     (key) == 107)   /* right Win */

#define IS_KEY_TO_FILTER(key)					\
    (((key) >= 1 && (key) <= 10)  ||	/* 1 . .0 */		\
     ((key) >= 15 && (key) <= 25) ||	/* Q ..  */		\
     ((key) >= 29 && (key) <= 39) ||	/* A ..  */		\
     ((key) >= 43 && (key) <= 49) ||	/* Z/Y .. M */		\
     ((key) >= 86 && (key) <= 88) ||	/* 7 .. 9 (num) */	\
     ((key) >= 91 && (key) <= 93) ||	/* 4 .. 6 (num) */	\
     ((key) >= 95 && (key) <= 97) ||	/* 1 .. 3 (num) */	\
     ((key) == 56))			/* space */		\

static int filter_keyboard_event(rfb_cl_t * clp, u_int8_t keycode) {
    int real_code = keycode & 0x7f;
    int pressed = keycode & 0x80;
    int modifier_pressed = 0;
    
    /* check if the key is a modifier (except shift) */
    if (IS_FILTER_MODIFIER(real_code)) {
    	/* do we have this modifier in our list? */
    	int pos = key_pos_in_vector(clp->modifier_vector, real_code);
    	
    	/* add new modifier or remove released modifier */
    	if (pos == -1 && pressed) {
    	    D(D_BLABLA, "Adding new modifier %d\n", real_code);
    	    vector_add(clp->modifier_vector, (void *)real_code);
    	} else if (pos != -1 && !pressed) {
    	    D(D_BLABLA, "Removing modifier %d\n", real_code);
    	    vector_remove(clp->modifier_vector, pos);
    	}
    }
    
    /* is a modifier pressed? */
    modifier_pressed = vector_size(clp->modifier_vector) != 0;
    
    if (IS_KEY_TO_FILTER(real_code)) {
       	/* now it is getting complicated: if a modifier is pressed, we
       	   send the key in any case; but imagine you press Ctrl, press C,
       	   release Ctrl and release C. In this case, the release of C
       	   would not be sent;
       	   way out: we keep all keys which are sent nut normally filtered
       	   in a vector and send the release of keys in it */
       	int pos = key_pos_in_vector(clp->unfiltered_keys_vector, real_code);
       	int ret = 0;
       	
       	if (modifier_pressed) {
       	    if (pressed && pos == -1) {
       	    	vector_add(clp->unfiltered_keys_vector, (void *)real_code);
       	    } else if (!pressed && pos != -1) {
       	    	vector_remove(clp->unfiltered_keys_vector, pos);
       	    }
       	} else {
       	    if (!pressed && pos != -1) {
       	    	vector_remove(clp->unfiltered_keys_vector, pos);
       	    } else if (!(pressed && pos != -1)) {
       	    	ret = 1;
       	    }
       	}
       	return ret;
    } else {
    	return 0;
    }  	
}

static void rfb_sas_enqueue_input_event_v(rfb_cl_t * clp, va_list args) {
    input_event_t *ie = va_arg(args, input_event_t *);
    
    /* don't send if local only flag is set */
    if (clp->connection_flags & rfbConnectionFlagSasLocalOnly && ie->session != 0) {
    	return;
    }

    if (ie->type == rfbKeyEvent) {
    	/* don't send keyboard event if not requested */
    	if (rfbConnectionFlagGetKbd(clp->connection_flags) == rfbConnectionSasKbdNoKbd) {
    	    return;
    	}
    	/* filter alphanumeric keys */
    	if (rfbConnectionFlagGetKbd(clp->connection_flags) == rfbConnectionSasKbdFilter &&
    	    filter_keyboard_event(clp, ie->data.keycode)) {
    	    ie->data.keycode |= 0x7f;
    	}
    }

    MUTEX_LOCK(&clp->sas_input_queue_mtx);
    if (pp_queue_enqueue(clp->sas_input_queue, ie,
			  sizeof(input_event_t), 0)) {
    	rfb_enqueue_sas_error(clp, rfbSasErrorMissedInputEvent);
    	goto bail;
    }
    if (pthread_cond_signal(&clp->sas_input_queue_cond) != 0) {
	D(D_ERROR, "%s(): pthread_cond_signal() failed\n", ___F);
    }
 bail:
    MUTEX_UNLOCK(&clp->sas_input_queue_mtx);
}

void rfb_sas_send_kvm_switch_event_v(rfb_cl_t * clp, va_list args) {
    eric_session_int_id_t session = va_arg(args, eric_session_int_id_t);
    unsigned int channel = va_arg(args, unsigned int);
    unsigned int unit = va_arg(args, unsigned int);
    unsigned int port = va_arg(args, unsigned int);

    /* don't send if local only flag is set */
    if (clp->connection_flags & rfbConnectionFlagSasLocalOnly) {
    	return;
    }

    D(D_BLABLA, "KVM switch event for client %p:\n"
    		"channel: %d, unit: %d, port: %d\n",
    		clp, channel, unit, port);
    
    /* send to clp */
    if (rfb_enqueue_sas_kvm_switch_event(clp, session, channel, unit, port)) {
    	rfb_enqueue_sas_error(clp, rfbSasErrorMissedOtherEvent);
    }
}

/* ---- SAS input event processing thread ---- */

static void rfb_sas_send_input_event(rfb_cl_t *clp, input_event_t *ie) {
    D(D_BLABLA, "Input event for client %p.\n", clp);
    
    /* send to clp */
    if (rfb_enqueue_sas_input_event(clp, ie, ie->session, ie->clp)) {
    	rfb_enqueue_sas_error(clp, rfbSasErrorMissedInputEvent);
    }
}

static void* rfb_sas_input_thread(void *_clp) {
    rfb_cl_t *clp = (rfb_cl_t *) _clp;
    
    D(D_BLABLA, "input processing thread for SAS client %p started.\n", clp);

    while (1) {
	struct timeval now;
	struct timespec timeout;
	pp_queue_entry_t *qe;
    	
	gettimeofday(&now, NULL);
	timeout.tv_sec = now.tv_sec + 10;
	timeout.tv_nsec = now.tv_usec * 1000;
	
	MUTEX_LOCK(&clp->sas_input_queue_mtx);
	if (pp_queue_first(clp->sas_input_queue) == NULL && !clp->sas_exit_input_thd) {
	    pthread_cond_timedwait(&clp->sas_input_queue_cond,
				   &clp->sas_input_queue_mtx,
				   &timeout);
	}
	
	if (clp->sas_exit_input_thd) {
	    MUTEX_UNLOCK(&clp->sas_input_queue_mtx);
	    break;
	}

	if ((qe = pp_queue_dequeue(clp->sas_input_queue)) == NULL) {
	    /* timeout occurred */
	    MUTEX_UNLOCK(&clp->sas_input_queue_mtx);
	    D(D_BLABLA, "No input event in SAS queue\n");
	    continue;
	}
	
	/* take events from queue and send them to the client */
	MUTEX_UNLOCK(&clp->sas_input_queue_mtx);
	while (qe) {
	    /* send the event */
	    input_event_t *ie = (input_event_t *)qe->data;
	    rfb_sas_send_input_event(clp, ie);
	    
	    /* free the event */
	    free(qe->data);
	    free(qe);
	    
	    /* get the next event */
	    MUTEX_LOCK(&clp->sas_input_queue_mtx);
	    qe = pp_queue_dequeue(clp->sas_input_queue);
	    MUTEX_UNLOCK(&clp->sas_input_queue_mtx);
	}
    }
    
    D(D_BLABLA, "input processing thread for SAS client %p finished.\n", clp);
    return NULL;
}

#endif /* PP_FEAT_SESSION_REDIRECTOR */
