/*
 * tableinittctemplate.c - template for initialising lookup tables for
 * truecolour to truecolour translation.
 *
 * This file shouldn't be compiled.  It is included multiple times by
 * translate.c, each time with a different definition of the macro OUT.
 * For each value of OUT, this file defines two functions for initialising
 * lookup tables.  One is for truecolour translation using a single lookup
 * table, the other is for truecolour translation using three separate
 * lookup tables for the red, green and blue values.
 *
 * I know this code isn't nice to read because of all the macros, but
 * efficiency is important here.
 */

/*
 *  Copyright (C) 1999 AT&T Laboratories Cambridge.  All Rights Reserved.
 *
 *  This is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This software is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,
 *  USA.
 */

#if !defined(OUT)
#error "This file shouldn't be compiled."
#error "It is included as part of translate.c"
#endif

#define OUT_T CONCAT3E(u_int,OUT,_t)
#define SwapOUT(x) CONCAT2E(bswap_,OUT(x))
#define rfbInitTrueColourSingleTableOUT \
				CONCAT2E(rfbInitTrueColourSingleTable,OUT)
#define rfbInitTrueColourRGBTablesOUT CONCAT2E(rfbInitTrueColourRGBTables,OUT)
#define rfbInitOneRGBTableOUT CONCAT2E(rfbInitOneRGBTable,OUT)

static void rfbInitOneRGBTableOUT (OUT_T *table, int inMax, int outMax,
				   int outShift, int swap);

/*
 * rfbInitTrueColourSingleTable sets up a single lookup table for truecolour
 * translation.
 */

static void
rfbInitTrueColourSingleTableOUT (char **table,
				 rfbPixelFormat * in,
				 rfbPixelFormat * out)
{
    int i;
    int inRed, inGreen, inBlue, outRed, outGreen, outBlue;
    OUT_T *t;
    int nEntries = 1 << in->bitsPerPixel_8;

    if (*table) {
	free(*table);
    }
    *table = (char *)malloc(nEntries * sizeof(OUT_T));
    if (*table == NULL) {
	pp_log_err("rfbInitTrueColourSingleTableOUT(): malloc()");
	pthread_exit(NULL);
    }
    t = (OUT_T *)*table;

    for (i = 0; i < nEntries; i++) {
	inRed   = (i >> in->redShift_8)   & in->redMax_be16;
	inGreen = (i >> in->greenShift_8) & in->greenMax_be16;
	inBlue  = (i >> in->blueShift_8)  & in->blueMax_be16;

	outRed   = (inRed   * out->redMax_be16   + in->redMax_be16/2)  / in->redMax_be16;
	outGreen = (inGreen * out->greenMax_be16 + in->greenMax_be16/2)/ in->greenMax_be16;
	outBlue  = (inBlue  * out->blueMax_be16  + in->blueMax_be16/2) / in->blueMax_be16;

	t[i] = ((outRed   << out->redShift_8)   |
		(outGreen << out->greenShift_8) |
		(outBlue  << out->blueShift_8));
#if (OUT != 8)
	if (out->bigEndian_8 != in->bigEndian_8) {
	    t[i] = SwapOUT(t[i]);
	}
#endif
    }
}


/*
 * rfbInitTrueColourRGBTables sets up three separate lookup tables for the
 * red, green and blue values.
 */

static void
rfbInitTrueColourRGBTablesOUT (char **table,
			       rfbPixelFormat * in,
			       rfbPixelFormat * out)
{
    OUT_T *redTable;
    OUT_T *greenTable;
    OUT_T *blueTable;

    if (*table) free(*table);
    *table = (char *)malloc((in->redMax_be16+in->greenMax_be16+in->blueMax_be16+3) *
			    sizeof(OUT_T));
    if (*table == NULL) {
	pp_log_err("rfbInitTrueColourRGBTablesOUT(): malloc()");
	pthread_exit(NULL);
    }
    redTable = (OUT_T *)*table;
    greenTable = redTable + in->redMax_be16 + 1;
    blueTable = greenTable + in->greenMax_be16 + 1;

    rfbInitOneRGBTableOUT(redTable, in->redMax_be16, out->redMax_be16,
			  out->redShift_8, out->bigEndian_8 != in->bigEndian_8);
    rfbInitOneRGBTableOUT(greenTable, in->greenMax_be16, out->greenMax_be16,
			  out->greenShift_8, out->bigEndian_8 != in->bigEndian_8);
    rfbInitOneRGBTableOUT(blueTable, in->blueMax_be16, out->blueMax_be16,
			  out->blueShift_8, out->bigEndian_8 != in->bigEndian_8);
}

static void
rfbInitOneRGBTableOUT (OUT_T *table, int inMax, int outMax, int outShift,
		       int swap UNUSED)
{
    int i;
    int nEntries = inMax + 1;

    for (i = 0; i < nEntries; i++) {
	table[i] = ((i * outMax + inMax / 2) / inMax) << outShift;

#if (OUT != 8)
	if (swap) {
	    table[i] = SwapOUT(table[i]);
	}
#endif
    }
}

#undef OUT_T
#undef SwapOUT
#undef rfbInitTrueColourSingleTableOUT
#undef rfbInitTrueColourRGBTablesOUT
#undef rfbInitOneRGBTableOUT
