/*
 * tight_cache.c
 *
 * Routines to implement hash + cache functions for Tight Encoding
 */


#include <stdio.h>
#include <netinet/in.h>
#include <rfb.h>

/* ------------------------------------------------------------------------- *
 * hash functions
 * ------------------------------------------------------------------------- */

typedef  unsigned long int  u4;   /* unsigned 4-byte type */
typedef  unsigned     char  u1;   /* unsigned 1-byte type */
/* The mixing step */
#define mix(a,b,c) \
{ \
  a=a-b;  a=a-c;  a=a^(c>>13); \
  b=b-c;  b=b-a;  b=b^(a<<8);  \
  c=c-a;  c=c-b;  c=c^(b>>13); \
  a=a-b;  a=a-c;  a=a^(c>>12); \
  b=b-c;  b=b-a;  b=b^(a<<16); \
  c=c-a;  c=c-b;  c=c^(b>>5);  \
  a=a-b;  a=a-c;  a=a^(c>>3);  \
  b=b-c;  b=b-a;  b=b^(a<<10); \
  c=c-a;  c=c-b;  c=c^(b>>15); \
}
/* The whole new hash function */
static u4 lookup(register u1 *k, u4 length, u4 initval)
/* the key */
/* the length of the key in bytes */
/* the previous hash, or an arbitrary value */
{
   register u4 a,b,c;  /* the internal state */
   u4          len;    /* how many key bytes still need mixing */
   /* Set up the internal state */
   len = length;
   a = b = 0x9e3779b9;  /* the golden ratio; an arbitrary value */
   c = initval;         /* variable initialization of internal state */
   /*---------------------------------------- handle most of the key */
   while (len >= 12)
   {
      a=a+(k[0]+((u4)k[1]<<8)+((u4)k[2]<<16) +((u4)k[3]<<24));
      b=b+(k[4]+((u4)k[5]<<8)+((u4)k[6]<<16) +((u4)k[7]<<24));
      c=c+(k[8]+((u4)k[9]<<8)+((u4)k[10]<<16)+((u4)k[11]<<24));
      mix(a,b,c);
      k = k+12; len = len-12;
   }
   /*------------------------------------- handle the last 11 bytes */
   c = c+length;
   switch(len)              /* all the case statements fall through */
   {
   case 11: c=c+((u4)k[10]<<24);
   case 10: c=c+((u4)k[9]<<16);
   case 9 : c=c+((u4)k[8]<<8);
      /* the first byte of c is reserved for the length */
   case 8 : b=b+((u4)k[7]<<24);
   case 7 : b=b+((u4)k[6]<<16);
   case 6 : b=b+((u4)k[5]<<8);
   case 5 : b=b+k[4];
   case 4 : a=a+((u4)k[3]<<24);
   case 3 : a=a+((u4)k[2]<<16);
   case 2 : a=a+((u4)k[1]<<8);
   case 1 : a=a+k[0];
     /* case 0: nothing left to add */
   }
   mix(a,b,c);
   /*-------------------------------------------- report the result */
   return c;
}

u_int32_t do_hash_new(u_int8_t *data, int size) {
   return lookup(data, size, 0);
}

u_int32_t do_hash_sdbm(u_int8_t *data, int size) {
   unsigned long hash = 0;

   for (; size > 0; size--, data++)
      hash = *data + (hash << 6) + (hash << 16) - hash;

   return hash;
}

/* ------------------------------------------------------------------------- *
 * cache support functions
 * ------------------------------------------------------------------------- */

static void *
cache_malloc(rfb_cl_t * clp, size_t new_size)
{
    int curr_size;
    void * ret;
    
    if(clp->cache_memory == NULL || clp->cache_memory_next == NULL) return NULL;
    
    curr_size = clp->cache_memory_next - clp->cache_memory;
    if(curr_size + new_size > clp->cache_memory_size) return NULL;
    
    ret = clp->cache_memory_next;
    clp->cache_memory_next += new_size;
    
    return ret;
}

static void
out_cache_list(rfb_cache_list_value_t * cache_list)
{
    rfb_cache_list_value_t * elem;
    int i = 0;

    elem = cache_list;
    while (elem != NULL) {
	printf("%2i - hn=%u hs=%u col=%u c=%i id=%2i  -  e=%p n=%p\n", ++i, 
	       elem->hash_new, elem->hash_sdbm, elem->col, elem->counter, 
	       elem->client_id, elem, elem->next);
	elem = elem->next;
    }
}

void
out_cache_lists(rfb_cache_lists_t * cache_lists)
{
    printf("hash_list: \n");
    out_cache_list(cache_lists->hash_list);
    printf("short_term_cache: \n");
    out_cache_list(cache_lists->short_term_cache);
    printf("long_term_cache: \n");
    out_cache_list(cache_lists->long_term_cache);
}

static void
rfb_init_cache_list_value(rfb_cache_list_value_t * elem, int8_t client_id)
{
    elem->hash_new = 0;
    elem->hash_sdbm = 0;
    elem->col = 0;
    elem->counter = 0;
    elem->client_id = client_id;
}

static rfb_cache_list_value_t *
rfb_init_cache_list(rfb_cl_t * clp, int start, int end)
{
    int i, size_cache_list_value;
    rfb_cache_list_value_t * first;
    rfb_cache_list_value_t * elem;

    size_cache_list_value = sizeof(rfb_cache_list_value_t);

    if ((first = cache_malloc(clp, size_cache_list_value)) == NULL) {
    	return NULL;
    }
    elem = first;
    for (i = start+1; i < end; i++) {
	rfb_init_cache_list_value(elem, i-1);
	if ((elem->next = cache_malloc(clp, size_cache_list_value)) == NULL) {
	    return NULL;
	}
	elem = elem->next;
    }
    rfb_init_cache_list_value(elem, end-1);
    elem->next = NULL;

    return first;
}

int
rfb_init_cache_lists(rfb_cl_t * clp, int new_tight_cache_size)
{
    int i, size_cache_list;
    rfb_cache_lists_t * cache_list;
    size_t memory_size;

    clp->tightCacheSize = new_tight_cache_size;
    size_cache_list = sizeof(rfb_cache_lists_t);
    
    memory_size = sizeof(rfb_cache_lists_t *);		// the pointer to the hash structure
    memory_size += sizeof(rfb_cache_lists_t);		// the hash structure
    memory_size += (RFB_NUM_RECTS_HASH_LIST +
                    RFB_NUM_RECTS_SHORT_TERM_CACHE +
                    RFB_NUM_RECTS_LONG_TERM_CACHE) *
                    sizeof(rfb_cache_list_value_t);	// the cache lists
    memory_size *= clp->tightCacheSize;			// and this for all cache tiles
    
    clp->cache_memory = malloc(memory_size);
    if(clp->cache_memory == NULL) goto fail;
    clp->cache_memory_next = clp->cache_memory;
    clp->cache_memory_size = memory_size;

    if ((clp->cache_lists = cache_malloc(clp, clp->tightCacheSize * sizeof(rfb_cache_lists_t *))) == NULL) {
	return -1;
    }
    for (i = 0; i < clp->tightCacheSize; i++) {
	if ((clp->cache_lists[i] = cache_malloc(clp, size_cache_list)) == NULL) {
	    goto fail;
	}
    	cache_list = clp->cache_lists[i];
	if (!(cache_list->hash_list = rfb_init_cache_list(clp, 0, RFB_NUM_RECTS_HASH_LIST))) {
	    goto fail;
	}
	if (!(cache_list->short_term_cache = rfb_init_cache_list(clp, 0, RFB_NUM_RECTS_SHORT_TERM_CACHE))) {
	    goto fail;
	}
	if (!(cache_list->long_term_cache = rfb_init_cache_list(clp, RFB_NUM_RECTS_SHORT_TERM_CACHE, RFB_NUM_RECTS_SHORT_TERM_CACHE+RFB_NUM_RECTS_LONG_TERM_CACHE))) {
	    goto fail;
	}
    }
    
    return 0;
fail:
    if(clp->cache_memory) {
    	free(clp->cache_memory);
    	clp->cache_memory = NULL;
    	clp->cache_memory_next = NULL;
    	clp->cache_lists = NULL;
    	clp->tightCacheSize = 0;
    	clp->cache_memory_size = 0;
    }
    return -1;
}

void
rfb_deinit_cache_lists(rfb_cl_t * clp)
{
    if(clp->cache_memory) {
    	free(clp->cache_memory);
    	clp->cache_memory = NULL;
    	clp->cache_memory_next = NULL;
    	clp->cache_lists = NULL;
    	clp->tightCacheSize = 0;
    	clp->cache_memory_size = 0;
    }
}

/* ------------------------------------------------------------------------- *
 * cache search functions
 * ------------------------------------------------------------------------- */

rfb_cache_list_value_t *
rfb_search_cache_list_lt(int8_t * client_id, rfb_cache_lists_t * cache_lists, int num, 
                         u_int32_t hash_new, u_int32_t hash_sdbm, u_int32_t col)
{
    int i;
    rfb_cache_list_value_t * elem;
    rfb_cache_list_value_t * first = cache_lists->long_term_cache;
    rfb_cache_list_value_t * second_last = NULL;

    elem = first;
    for (i = 0; i < num; i++) {
    	if ((elem->hash_new == hash_new) && 
    	    (elem->hash_sdbm == hash_sdbm) &&
    	    (elem->col == col)) {
    	    if (elem != first) { // not first in list
    	    	second_last->next = elem->next;
    	    	elem->next = first;
    	    	cache_lists->long_term_cache = elem;
    	    }
    	    *client_id = elem->client_id;
    	    return NULL;
    	}
    	if (elem->next) { // to return the second last elem for insert
    	    second_last = elem;
    	    elem = elem->next;
    	}
    }

    return second_last;
}

rfb_cache_list_value_t *
rfb_search_cache_list_st(int8_t * client_id, rfb_cache_lists_t * cache_lists, int num, 
                         u_int32_t hash_new, u_int32_t hash_sdbm, u_int32_t col, 
                         rfb_cache_list_value_t * lt_2l)
{
    int i;
    rfb_cache_list_value_t * elem;
    rfb_cache_list_value_t * first = cache_lists->short_term_cache;
    rfb_cache_list_value_t * second_last = NULL;

    elem = first;
    for (i = 0; i < num; i++) {
    	if ((elem->hash_new == hash_new) && 
    	    (elem->hash_sdbm == hash_sdbm) &&
    	    (elem->col == col)) {
    	    *client_id = elem->client_id;
	    if (++elem->counter == RFB_RECT_SHORT_2_LONG_TERM_CACHE) {
	    	elem->counter--;
	    	// replace elem in short term cache with last elem of long term cache
	    	if (elem != first) { // not first in list
		    second_last->next = lt_2l->next;
		    second_last->next->next = elem->next;
	    	}
	    	else {
		    cache_lists->short_term_cache = lt_2l->next;
		    cache_lists->short_term_cache->next = elem->next;
		}
		lt_2l->next = NULL;
		// set elem as new first elem of long term cache
		elem->next = cache_lists->long_term_cache;
		cache_lists->long_term_cache = elem;
	    }
	    else {
		if (elem != first) { // not first in list
		    second_last->next = elem->next;
		    elem->next = first;
		    cache_lists->short_term_cache = elem;
		}
	    }
    	    return NULL;
    	}
    	if (elem->next) { // to return the second last elem for insert
	    second_last = elem;
	    elem = elem->next;
    	}
    }

    return second_last;
}

rfb_cache_list_value_t *
rfb_search_cache_list_hl(int8_t * client_id, rfb_cache_lists_t * cache_lists, int num, 
                         u_int32_t hash_new, u_int32_t hash_sdbm, u_int32_t col, 
                         rfb_cache_list_value_t * lt_2l, rfb_cache_list_value_t * st_2l)
{
    int i;
    rfb_cache_list_value_t * elem;
    rfb_cache_list_value_t * first = cache_lists->hash_list;
    rfb_cache_list_value_t * second_last = NULL;

    elem = first;
    for (i = 0; i < num; i++) {
    	if ((elem->hash_new == hash_new) && 
    	    (elem->hash_sdbm == hash_sdbm) &&
    	    (elem->col == col)) {
	    if (++elem->counter < RFB_RECT_SHORT_2_LONG_TERM_CACHE) {
	    	// reorder hash list
	    	if (elem != first) { // not first in list
		    second_last->next = elem->next;
		    st_2l->next->next = cache_lists->hash_list;
	    	}
	    	else {
		    st_2l->next->next = elem->next;
	    	}
	    	cache_lists->hash_list = st_2l->next;
	    	// reorder short term cache
	    	st_2l->next = NULL;
	    	elem->client_id = cache_lists->hash_list->client_id;
	    	elem->next = cache_lists->short_term_cache;
	    	cache_lists->short_term_cache = elem;
	    	// set client_id
	    	*client_id = elem->client_id;
	    }
	    else {
	    	elem->counter--;
	    	// reorder hash list
	    	if (elem != first) { // not first in list
		    second_last->next = elem->next;
		    st_2l->next->next = cache_lists->hash_list;
	    	}
	    	else {
		    st_2l->next->next = elem->next;
	    	}
	    	cache_lists->hash_list = st_2l->next;
	    	// reorder short term cache
	    	st_2l->next = NULL;
	    	lt_2l->next->next = cache_lists->short_term_cache;
	    	cache_lists->short_term_cache = lt_2l->next;
	    	// reorder long term cache
	    	lt_2l->next = NULL;
	    	elem->client_id = cache_lists->hash_list->client_id;
	    	elem->next = cache_lists->long_term_cache;
	    	cache_lists->long_term_cache = elem;
	    	// set client_id
	    	*client_id = elem->client_id;
	    }
    	    return NULL;
    	}
    	if (elem->next) { // to return the second last elem for insert
    	    second_last = elem;
    	    elem = elem->next;
    	}
    }

    return second_last;
}

/* ------------------------------------------------------------------------- *
 * cache insert function
 * ------------------------------------------------------------------------- */

void
rfb_insert_short_term(int8_t * client_id, rfb_cache_lists_t * cache_lists, 
                      u_int32_t hash_new, u_int32_t hash_sdbm, u_int32_t col, 
                      rfb_cache_list_value_t * st_2l, rfb_cache_list_value_t * hl_2l)
{
    // client_id stuff
    hl_2l->next->client_id = st_2l->next->client_id;
    *client_id = hl_2l->next->client_id;
    // reorder hash list
    st_2l->next->next = cache_lists->hash_list;
    cache_lists->hash_list = st_2l->next;
    st_2l->next = NULL;
    // reorder short term cache
    hl_2l->next->next = cache_lists->short_term_cache;
    cache_lists->short_term_cache = hl_2l->next;
    hl_2l->next = NULL;
    // store new values
    cache_lists->short_term_cache->counter = 1;
    cache_lists->short_term_cache->hash_new = hash_new;
    cache_lists->short_term_cache->hash_sdbm = hash_sdbm;
    cache_lists->short_term_cache->col = col;
}
