#include <auth-client.h>
#include <liberic_config.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <libesmtp.h>
#include <unistd.h>
#include <ctype.h>
#include <fcntl.h>
#include <pp/smtp.h>
#include <pp/cfg.h>
#include <pp/vector.h>

static int   tlsinteract (char *buf, int buflen, int rwflag, void *arg);
static int   authinteract (auth_client_request_t request, char **result, int fields, void *arg);
static void  print_recipient_status (smtp_recipient_t recipient, const char *mailbox, void *arg);

/* slightly modified example from libesmtp */
int
pp_smtp_send_mail(mail_data_t * data)
{
    smtp_session_t session;
    smtp_message_t message;
    smtp_recipient_t recipient;
    auth_context_t authctx;
    const smtp_status_t *status;
    struct sigaction sa; 
    char *ip_tmp = NULL;
    char *mail_tmp = NULL;
    char* host_tmp = NULL;
    char from[MAX_OPT_KEY_LEN+1];  

    /* FIXME: (rwa) slave nodes should not call this! */
  
    pp_log("Send Mail to %s, event: %s, subject: %s, host: %s\n", data->to, data->mail_txt, data->subject, data->host);
  
    if (data->mail_txt) {
	asprintf(&mail_tmp, "\r\n%s", data->mail_txt);
    } else {
	mail_tmp = strdup("\r\n");
    }

    if (data->from) {
	snprintf(from, sizeof(from), "%s", data->from);
    } else {
	if (PP_SUCCED(pp_cfg_get(&ip_tmp, "network.ipaddr"))) {
	    snprintf(from, sizeof(from), "root@%s", ip_tmp);
	} else {
	    *from = '\0';
	}
    }

    auth_client_init ();
    session = smtp_create_session ();
    message = smtp_add_message (session);               
               
    sa.sa_handler = SIG_IGN;
    sigemptyset (&sa.sa_mask);
    sa.sa_flags = 0;
    sigaction (SIGPIPE, &sa, NULL);   

    if (data->host) {
	// check for ':'(portnr) in the host string or add it
	if (!strchr(data->host, ':')) {
	    asprintf(&host_tmp, "%s:25", data->host);
	} else {
	    host_tmp = strdup(data->host);
	}
    }
    smtp_set_server(session, host_tmp ? host_tmp : "localhost:25");
  
    authctx = auth_create_context();
  
    auth_set_mechanism_flags(authctx, AUTH_PLUGIN_PLAIN, 0);  
  
    auth_set_interact_cb(authctx, authinteract, NULL);  
  
    smtp_starttls_set_password_cb(tlsinteract, NULL);

    smtp_auth_set_context(session, authctx);

    smtp_set_reverse_path(message, from);

    smtp_set_header(message, "To", NULL, NULL);
  
    smtp_set_header(message, "Date", &data->time_stamp);

    if (*from) smtp_set_header(message, "From", NULL, from);
  
    smtp_set_header (message, "Subject", data->subject);

    /* Set extra headers */
    if (data->extra_hdrs != NULL)
    {
	int hdr_cnt = vector_size(data->extra_hdrs);
	int i;

	for (i = 0; i < hdr_cnt; i++) {
	    mail_header_t *hdr;
	    hdr = (mail_header_t *)vector_get(data->extra_hdrs, i);
	    smtp_set_header (message, hdr->name, hdr->val);
	}
    }
  
    /* Add remaining program arguments as message recipients. */
 
    recipient = smtp_add_recipient (message, data->to);

    smtp_set_message_str(message, mail_tmp);
  
    /* Initiate a connection to SMTP server and transfer the message. */
    if (!smtp_start_session (session)) {
	char buf[128];

	pp_log("%s(): SMTP server problem %s\n", ___F, smtp_strerror(smtp_errno (), buf, sizeof(buf)));
    } else {
	/* Report on the success or otherwise of the mail transfer. */
	status = smtp_message_transfer_status(message);
	smtp_enumerate_recipients(message, print_recipient_status, NULL);
    }
    
    /* Free resources consumed by the program. */
    smtp_destroy_session(session);
    auth_destroy_context(authctx);  

    auth_client_exit();
  
    free(ip_tmp);
    free(mail_tmp);
    free(host_tmp);
  
    return 0;
}
  
/* Callback to print the recipient status */
static void
print_recipient_status (smtp_recipient_t recipient,
			const char *mailbox UNUSED, void *arg UNUSED)
{
    const smtp_status_t *status;

    status = smtp_recipient_status (recipient);  
}

/* Callback to request user/password info.  Not thread safe. */
static int
authinteract (auth_client_request_t request, char **result, int fields,
              void *arg UNUSED)
{
    char prompt[64];
    static char resp[512];
    char *p, *rp;
    int i, n, tty;

    rp = resp;
    for (i = 0; i < fields; i++)
	{
	    n = snprintf (prompt, sizeof prompt, "%s%s: ", request[i].prompt,
			  (request[i].flags & AUTH_CLEARTEXT) ? " (not encrypted)"
			  : "");
	    if (request[i].flags & AUTH_PASS)
		result[i] = getpass (prompt);
	    else
		{
		    tty = open ("/dev/tty", O_RDWR);
		    write (tty, prompt, n);
		    n = read (tty, rp, sizeof resp - (rp - resp));
		    close (tty);
		    p = rp + n;
		    while (isspace (p[-1]))
			p--;
		    *p++ = '\0';
		    result[i] = rp;
		    rp = p;
		}
	}
    return 1;
}

static int
tlsinteract (char *buf, int buflen, int rwflag UNUSED, void *arg UNUSED)
{
    char *pw;
    int len;

    pw = getpass ("certificate password");
    len = strlen (pw);
    if (len + 1 > buflen) return 0;
    strcpy (buf, pw);
    return len;
}
