//=============================================================================
/**
 *      raasip_test.cpp
 *
 *      Date: 11.08.2006 Author Thomas Weber tweb@raritan.com
 *
 */
//=============================================================================

// system includes

// fw includes
#include <pp/um.h>

// local includes
#include "raasip_test.h"


// variables
static const char *testuser = "testRAASIP";
static const char *testuser2 = "testRAASIP2";
static const char *testpass = "myPaS$w0rd";
static const char *adminuser = "admin";
//static const char *admingroup = "Admin";
static const char *unknowngroup = "<Unknown>";
//static const char *nonegroups = "<None>";
static const char *foo = "foo";

// simulate external APIs
#ifdef __cplusplus
extern "C"
{
#endif
void pp_hal_kx2_buzzer(int)
{
}
#ifdef __cplusplus
}   // extern "C"
#endif

#define _CPPUNIT_ASSERT(__cond__) { \
            int __condition__ = __cond__; \
            if(!__condition__) { \
                printf("errno = %d (%s)\n", errno, pp_error_string(errno)); \
            } \
            CPPUNIT_ASSERT(__condition__); \
        }            

CPPUNIT_TEST_SUITE_REGISTRATION( CTestRAASIP );

/*
 * FUNCTION:    ctor
 *
 * ARGUMENTS:
 *
 * RETURN:
 * -------------------------------------------
 *  Test suit ctor
 */
CTestRAASIP::CTestRAASIP()
{
}

/*
 * FUNCTION:    setUp()
 *
 * ARGUMENTS:   none
 *
 * RETURN:      none
 * -------------------------------------------
 *      Put initialization here. This function
 * will be invoked when each test is started.
 */
void
CTestRAASIP::setUp()
{
}


/*
 * FUNCTION:    tearDown()
 *
 * ARGUMENTS:   none
 *
 * RETURN:      none
 * -------------------------------------------
 *  This method will be invoked when each test
 * finishes his job.
 */
void
CTestRAASIP::tearDown()
{
}

static int check_iter(const char *key) {
    pp_cfg_iter_t* iter = NULL;
    int ret = PP_ERR != pp_cfg_iter_create_at_layer(&iter, PP_PROFILE_LOCAL, key)
              && 0 /* NULL */ != pp_cfg_iter_next(iter, NULL, NULL);

    pp_cfg_iter_destroy(iter);
        
    return ret;
}

void CTestRAASIP::Test_Environment() {
    // check if we can initialize um-lib
    CPPUNIT_ASSERT(pp_um_init() != PP_ERR);

    // check if there are already users created
    {
        int local_user_created = check_iter("user[]");
        
        CPPUNIT_ASSERT(!local_user_created);
    }

    // check if there are already groups created
    {
        int local_user_created = check_iter("group[]");
        
        CPPUNIT_ASSERT(!local_user_created);
    }

    // check for default users
    // TODO: this may be OEM dependant!
    {
        vector_t *users = pp_um_get_all_users();
        char *user = NULL;
        
        CPPUNIT_ASSERT(users);
        CPPUNIT_ASSERT(vector_size(users) == 1);
        
        user = (char*)vector_get(users, 0);
        
        CPPUNIT_ASSERT(user);
        CPPUNIT_ASSERT(!strcmp(user, adminuser));
        
        vector_delete(users);
    }
    
    // check for default groups
    // TODO: this may be OEM dependant!
    {
        vector_t *groups = pp_um_get_all_groups();
        char *group = NULL;
        unsigned int groups_sz, u;
        int gid_set[3];
        
        // init gid_set to 0
        memset(gid_set, 0, sizeof(gid_set));
        
        CPPUNIT_ASSERT(groups);
        
        groups_sz = vector_size(groups);
        
        CPPUNIT_ASSERT(groups_sz == 3);
        
        for(u = 0; u < groups_sz; ++u) {
            int gid;
            
            group = (char*)vector_get(groups, u);
        
            CPPUNIT_ASSERT(group);
            CPPUNIT_ASSERT(PP_ERR != (gid = pp_um_group_get_gid(group)));
            CPPUNIT_ASSERT(gid >= 0 && gid <=2);
            
            ++gid_set[gid];
        }
        
        for(u = 0; u < sizeof(gid_set) / sizeof(int); ++u) {
            CPPUNIT_ASSERT(gid_set[u] == 1);
        }
        
        vector_delete(groups);
    }
    
}

void CTestRAASIP::Test_CreateUser() {
    // create a user - this should not cause an error
    CPPUNIT_ASSERT(pp_um_user_create(testuser, NULL, 0, NULL, NULL, 0) != PP_ERR);
    
    // create same user again - errno should be set to PP_UM_ERR_USER_ALREADY_EXISTS
    CPPUNIT_ASSERT(pp_um_user_create(testuser, NULL, 0, NULL, NULL, 0) == PP_ERR);
    CPPUNIT_ASSERT(errno == PP_UM_ERR_USER_ALREADY_EXISTS);
}

void CTestRAASIP::Test_Authenticate() {
    // set password for user
    CPPUNIT_ASSERT(pp_um_user_set_password(testuser, testpass, 
                                           PP_UM_AUTH_NO_FLAGS) == PP_SUC);
    
    // check if user may authenticate with the password set
    CPPUNIT_ASSERT(pp_um_user_authenticate(testuser, testpass, 
                                           PP_UM_AUTH_NO_FLAGS, NULL,
                                           NULL) == PP_SUC);
    
    // ... and only with the password set
    CPPUNIT_ASSERT(pp_um_user_authenticate(testuser, foo, 
                                           PP_UM_AUTH_NO_FLAGS, NULL,
                                           NULL) == PP_ERR);
    
    // TODO: pp_um_user_authenticate_with_ip
}

void CTestRAASIP::Test_GroupAssignment() {
    char *groupname = NULL;
    char individualgroup[63];
  
    snprintf(individualgroup, 63, "%c%s", um_individual_group_prefix, testuser);

    // set individual group
    CPPUNIT_ASSERT(pp_um_user_set_group(testuser, NULL) == PP_SUC);
    
    // test if group naming matches policy
    CPPUNIT_ASSERT(pp_um_user_get_group(testuser, &groupname) == PP_SUC);
    CPPUNIT_ASSERT(groupname);
    CPPUNIT_ASSERT(!strcmp(groupname, individualgroup));
    
    // now set generic group for user
    CPPUNIT_ASSERT(pp_um_user_set_group(testuser, unknowngroup) == PP_SUC);
    
    // assert, individual group vanished
    CPPUNIT_ASSERT(pp_um_group_get_gid(individualgroup) == PP_ERR);
    CPPUNIT_ASSERT(errno = PP_UM_ERR_GROUP_DOESNT_EXIST);
    
    // reset individual group for user
    CPPUNIT_ASSERT(pp_um_user_set_group(testuser, NULL) == PP_SUC);
    
    // test if group naming still matches policy
    CPPUNIT_ASSERT(pp_um_user_get_group(testuser, &groupname) == PP_SUC);
    CPPUNIT_ASSERT(groupname);
    CPPUNIT_ASSERT(!strcmp(groupname, individualgroup));
    
    // assign unknown group to user
    CPPUNIT_ASSERT(pp_um_user_set_group(testuser, foo) == PP_ERR);
    CPPUNIT_ASSERT(errno = PP_UM_ERR_GROUP_DOESNT_EXIST);
}

void CTestRAASIP::Test_TransformUser() {
    char *tmp1 = NULL, *tmp2 = NULL; 
  
    // rename once
    CPPUNIT_ASSERT(pp_um_user_rename(testuser, testuser2) == PP_SUC);

    // rename once more
    CPPUNIT_ASSERT(pp_um_user_rename(testuser, testuser2) == PP_ERR);
    CPPUNIT_ASSERT(errno == PP_UM_ERR_USER_ALREADY_EXISTS ||
                   errno == PP_UM_ERR_USER_DOESNT_EXIST);
    
    // copy once
    CPPUNIT_ASSERT(pp_um_user_copy(testuser2, testuser) == PP_SUC);
    
    // copy once more
    CPPUNIT_ASSERT(pp_um_user_copy(testuser2, testuser) == PP_ERR);
    CPPUNIT_ASSERT(errno == PP_UM_ERR_USER_ALREADY_EXISTS);

    // delete once
    CPPUNIT_ASSERT(pp_um_user_delete(testuser) != PP_ERR);
    CPPUNIT_ASSERT(pp_um_user_delete(testuser2) != PP_ERR);
    
    // delete once more
    CPPUNIT_ASSERT(pp_um_user_delete(testuser) == PP_ERR);
    CPPUNIT_ASSERT(errno == PP_UM_ERR_USER_DOESNT_EXIST);
    CPPUNIT_ASSERT(pp_um_user_delete(testuser2) == PP_ERR);
    CPPUNIT_ASSERT(errno == PP_UM_ERR_USER_DOESNT_EXIST);
    
    // test if copy matches original
    CPPUNIT_ASSERT(pp_um_user_copy(adminuser, testuser) == PP_SUC);
pp_cfg_save(DO_FLUSH); exit(0);
    CPPUNIT_ASSERT(pp_um_user_get_gid(adminuser) == pp_um_user_get_gid(testuser));
    
    CPPUNIT_ASSERT(pp_um_user_get_password(adminuser, &tmp1) != PP_ERR);
    CPPUNIT_ASSERT(tmp1); 
    CPPUNIT_ASSERT(pp_um_user_get_password(testuser, &tmp2) != PP_ERR);
    CPPUNIT_ASSERT(tmp2); 
    CPPUNIT_ASSERT(!strcmp(tmp1, tmp2)); 
    
    // uid may not match!
    CPPUNIT_ASSERT(pp_um_user_get_uid(adminuser) != pp_um_user_get_uid(testuser));
}

/*
void CTestRAASIP::Test_() {
    CPPUNIT_ASSERT();
    CPPUNIT_ASSERT();
    CPPUNIT_ASSERT();
printf("errno = %d (%s)\n", errno, pp_error_string(errno));
}

*/

void CTestRAASIP::Test_Cleanup() {
    CPPUNIT_ASSERT(pp_um_user_delete(testuser) != PP_ERR);
    pp_cfg_save(DO_FLUSH);
}

