/**
 * User manangement routines
 * This library implements RAASIP compatible usermanagement and replaces
 * the old libpp_um (that was based on permission inheritance).
 * 
 * All functions in this library operate (unless otherwise noted)
 * on the config system. That means they operate either on the local
 * flashdisk (fully writable) or on a remote ldap server (mostly
 * readonly).
 *
 * (c) 2006 Peppercon AG, geo@peppercon.de, tweb@raritan.com
 */

#ifndef __PP_UM_H__
#define __PP_UM_H__

#ifdef __cplusplus
extern "C" {
#endif

#include <pp/base.h>
#include <pp/cfg.h>
#include <pp/acl.h>

#ifdef PP_FEAT_AUTH_RADIUS
#include <radlib.h>
#endif

// core functions

#define PP_UM_AUTH_NO_FLAGS                     0
#define PP_UM_AUTH_IGNORE_BLOCKING              (1 << 0)
#define PP_UM_AUTH_PASSWD_AS_MD5_HASH           (1 << 1)
#if defined(PP_FEAT_STRONG_PASSWORDS)
#define PP_UM_AUTH_STRONG_PW_CHECKED            (1 << 6)
#endif /* PP_FEAT_STRONG_PASSWORDS */
#define PP_UM_AUTH_PASSWD_FORCE_CHANGE          (1 << 7)

typedef enum {
    PP_UM_GROUP_GENERIC,
    PP_UM_GROUP_INDIVIDUAL_AVAILABLE,
    PP_UM_GROUP_INDIVIDUAL_TAKEN
} pp_um_group_state_t;

extern const char um_individual_group_prefix;

/**
 * Authenticate a user using the currently active authentication scheme.
 * This can be local, local+AS, AS only or ldap(PEMiX) authentication.
 * Authentication will fail if the device is CC managed.
 * 
 * If authentication returns PP_SUC, the user supplied the correct password
 * and can log in. The group is not explicitly returned. It will be retrieved
 * implicitly when querying for users permissions.
 * 
 * This function does not know anything about session management and thus 
 * can _not_ enforce the single session only option. This option will have
 * to be implemented in the session managment.
 * 
 * This function does implement the user blocking option. If authentication
 * of the same user fails for the specified amount of tries then the user will
 * be blocked for a certain amount of time. An administrator can unblock the
 * currently blocked users through pp_um_user_unblock(). The usermanagement
 * currently does not support IP blacklisting although this could be 
 * implemented here.
 * 
 * This function also implements group time controlled access.
 * FIXME: check how to do this best in peppercon firmware.
 * 
 * This function does _not_ iplement ip access control, please use 
 * pp_um_user_authenticate_with_ip() to use that feature.
 * 
 * @param username    user to authenticate
 * @param password    password supplied by user
 * @param flags       flags to control authentication, 
 *                    e.g. password is already MD5-hash or plain
 * @param user_ip     IP address, the user wants to log on from
 * @param user_ip_len
 * @param server_sid  server session id if authenticated by RADIUS
 * 
 * @return            PP_SUC on if the user is authenticated,
 *                    errno with verbose string else (user unauthenticated,
 *                    internal error, ...)
 */
int pp_um_user_authenticate_with_ip(const char* username,
                                    const char* password,
                                    int flags,
                                    struct sockaddr *user_ip,
                                    socklen_t user_ip_len,
                                    int *server_sid);

/**
 * Authenticate a user. This function does exactly the same as
 * pp_um_user_authenticate_with_ip() except that the ip is given as string of
 * INET format xxx.xxx.xxx.xxx
 * 
 * @see pp_um_user_authenticate_with_ip()
 */
int pp_um_user_authenticate_with_ip_str(const char* username,
                                        const char* password,
                                        int flags,
                                        const char* user_ip_str,
                                        int *server_sid,
					int *gid);

/**
 * Authenticate a user. This function does exactly the same as
 * pp_um_user_authenticate_with_ip() except that the ip ACL restrictions
 * are not checked.
 * 
 * @see pp_um_user_authenticate_with_ip()
 */
int pp_um_user_authenticate(const char* username,
                            const char* password,
                            int flags,
                            int *server_sid,
                            int *gid);

/**
 * Check a permission for a user. Retrieves the users group and checks if the
 * user has the specified permission. Technically permissions are not restricted
 * to the (Y/N) or (none/view/control) permissions defined in RAASIP. Although
 * it is possible to use abritrary permission, it is highly recommended to stick
 * to the RAASIP permission scheme to unify user experience.
 * 
 * Existing permissions and their possible values are configured in the config
 * system and retrieved with 'acl.h'.
 * 
 * @param username           user reference
 * @param permission_name    the name of the permission (e.g. 'port_0')
 * @param permission_value   the desired permission value, 
 *                           (eg. 'view' or 'control')
 * 
 * @return                   PP_SUC if the user has the permission
 *                           errno with verbose string else (no permission,
 *                           internal error, ...)
 */
int pp_um_user_has_permission(const char* username,
                              const char* permission_name,
                              const char* permission_value);

/**
 * Get a permission for a user.
 * Remember to free returned permission_value!
 * 
 * @param username           user reference
 * @param permission_name    the name of the permission (e.g. 'port_0')
 * @param permission_value   return value, not set in PP_ERR case
 * 
 * @return                   PP_SUC if the users permission could be retrived,
 *                           errno with verbose string else (no permission,
 *                           internal error, ...)
 */
int pp_um_user_get_permission(const char* username,
                              const char* permission_name,
                              char** permission_value);

/**
 * Set a permission for a user.
 *
 * @see pp_um_group_set_permission
 * 
 * @param username           user reference
 * @param permission_name    the name of the permission (e.g. 'port_0')
 * @param permission_value   the desired permission value, 
 *                           (eg. 'view' or 'control')
 * 
 * @return                   PP_SUC if the users permission could be set,
 *                           errno with verbose string else (no permission,
 *                           internal error, ...)
 */
int pp_um_user_set_permission(const char* username,
                              const char* permission_name,
                              const char* permission_value);



// convenience functions to prevent useless attempts to change
// unchangeable usersettings (both user- and groupmanagement)

/**
 * Returns whether usermanagement is possible or not. If this function returns 1
 * then users and groups can be edited, created, renamed and deleted. If this
 * function returns 0, users can not be edited. This is currently only true
 * if full LDAP mode is activated (which overrides the local database).
 * 
 * Note: There may be some exceptions in the config system concerning the admin.
 * (Technically the admin is always local and thus will be always writeable).
 * As the only changeable attribute of the admin is the password, this can savely
 * be ignored.
 * 
 * (FIXME: what about the local admin in CC managed mode?)
 * 
 * @return                   1 if usermanagement is possible, 0 else
 */
int pp_um_usermanagement_possible(void);

/**
 * Returns whether changing the local password for a user is reasonable or
 * not. This usually returns 1 if the user authenticates against the local
 * database and 0 if the user authenticates against a remote server and 
 * thus changing the local password will have no effect. For some authentication
 * scenarios (like local) this value is fixed, for others (like RAASIP
 * authentication servers) it will depend on the user.
 * 
 * @param username           name of the user that wants to change his pasword
 * 
 * @return                   1 if passwordchange is reasonable
 */
int pp_um_passwordchange_reasonable(const char* username);

// FIXME: do we need pp_um_groupchange_possible() ?


// Usermanagement, convenience functions to access the config system

/**
 * Create a new user with the specified values.
 * @param username    the user name
 * @param password    the user password
 * @param flags       flags to control password storage, 
 *                    e.g. password is already MD5-hash or plain
 * @param groupname   the user group, need not exist, may be null. In both
 *                    cases the user will be mapped to gropup '<unknown>'
 * @param dialback    dialback number for users that dial in through modem.
 *                    May be null. Convenience parameter only that may be set
 *                    through the config system.
 * @param ipmi_uid    IPMI user id to use; 0 to let the um pick one
 * 
 * @return            PP_SUC if the user could be created, errno else
 */
int pp_um_user_create(const char* username, const char* password, int flags,
                      const char* groupname, const char* dialback, int ipmi_uid);

/**
 * Copy a user account. This does not only copy the user password, group, etc
 * but also the user settings.
 * 
 * @param username_src        the name of the existing user
 * @param username_dest       the name of the new user
 * 
 * @return            PP_SUC if the user could be copied, errno else
 */
int pp_um_user_copy(const char* username_src, const char* username_dest);


/**
 * Rename a user. This does not rename any individual groups related to this user.
 * 
 * @param username_current    the users current name
 * @param username_new        the new username
 * 
 * @return                    PP_SUC if the user could be renamed errno else
 */
int pp_um_user_rename(const char* username_current, const char* username_new);

/**
 * Delete the specified user and all its settings. This does not delete any
 * individual groups related to this users.
 * 
 * @param username            the user account to delete
 * 
 * @return                    PP_SUC if the user could be deleted, errno else
 */
int pp_um_user_delete(const char* username);

/**
 * Enable or disable the specified user. Disabling a user will not delete
 * or hide it, but the user will no longer be able to log in via the web
 * frontend or IPMI.
 *
 * @param username            the user account to enable or disable
 * @param enabled             disable (0) or enable (1) the user
 *
 * @return                    PP_SUC or PP_ERR
 */
int pp_um_user_set_enabled(const char* username, int enabled);

/**
 * Set the local password of a user.
 * This function performs strong password checks if the option is activated.
 * 
 * @param username            the target user 
 * @param password            the new password
 * @param flags               flags to control password storage, 
 *                            e.g. password is already MD5-hash or plain
 * 
 * @return                    PP_SUC if the password could be set, errno else
 */
int pp_um_user_set_password(const char* username, 
                            const char* password, int flags);

/**
 * Set the group of a user.
 * 
 * @param username            the target user
 * @param group               the name of the group to assign
 * 
 * FIXME: does this check the existance of a group ?
 */
int pp_um_user_set_group(const char* username, const char* group);


/**
 * Get a list of all existing usernames. Caller is responsible for
 * freeing the returned vector.
 * 
 * @return                    list of all usernames, NULL on error
 */
vector_t* pp_um_get_all_users(void);

/**
 * Get a list of names of all users assigned to a specified group. Caller is
 * responsible for freeing the returned vector.
 * 
 * @param gid                 ID of the group, users should be assigned to
 *                            (-1 for all groups)
 *
 * @return                    list of all usernames in group, NULL on error
 */
vector_t* pp_um_get_all_users_by_gid(int gid);

/**
 * Get the user id of the specified user. Needed for access to this users
 * configuration settings. Returns -1 (PP_ERR) if the user could not be found
 * (TODO?! or the user has no local writeable configuration.) 
 * 
 * @param username            the username to get the id for
 * @return                    the user id of the specified user, -1 on error
 */
int pp_um_user_get_uid(const char* username);

/**
 * Check whether the specified user is enabled or disabled
 *
 * @param username            the username to get the id for
 * @return                    0 if disabled, 1 if enabled, -1 on error
 */
int pp_um_user_is_enabled(const char* username);

/**
 * Get the plaintext password if it exists. Needed for IPMI authentication.
 * Do not use this in the webfrontend.
 * 
 * @param username            the target user
 * @param password            reference to a pointer that will be modified to
 *                            point to the password string. Password buffer
 *                            must be freed by caller.
 * 
 * @return                    PP_SUC if the password could be read, errno else
 */
int pp_um_user_get_password(const char* username, char** password);

/**
 * Get the group assigned to this user.
 * @param username            the target user
 * @param group               user supplied buffer where the group will be stored
 * 
 * @return                    PP_SUC if the group could be read, errno else
 * 
 * FIXME: group buffer size ?
 * FIXME: errno really necessary ? group should be <unknown> if not existing ...
 */
int pp_um_user_get_group(const char* username, char** group);

/**
 * Get the group id assigned to this user.
 * @param username            the target user
 * 
 * @return                    gid if the group could be read, 
 *                            PP_ERR and errno else
 */
int pp_um_user_get_gid(const char* username);

/**
 * Returns if the user is a root user (belongs to group "admin").
 * 
 * @param is_root             set to 1 if user is root, 0 otherwise
 * @param username            the target user
 * 
 * @return                    PP_SUC if the information could be determined,
 *                            PP_ERR and errno else
 */
int pp_um_user_is_root(int* is_root, const char* username);

/**
 * Returns if the user needs to change it's password (due to password ageing).
 * 
 * @param username            the target user
 * 
 * @return                    PP_SUC if password change is necessary,
 *                            PP_ERR (and errno) else
 */
int pp_um_user_need_pw_change(const char* username);


// Groupmanagement

/**
 * Create a group without any privileges. Privileges must be set individually.
 * 
 * @param groupname           the name of the group to create
 * @param individual          create individual group
 *
 * @return                    PP_SUC if the group could be created, errno else
 */
int pp_um_group_create(const char* groupname, int individual);

/**
 * Rename a group, does not rename the 'group' reference for users.
 * 
 * @param groupname_current   the current name of the group
 * @param_groupname_new       the new name of the group
 * 
 * @return                    PP_SUC if the group could be renamed, errno else
 */
int pp_um_group_rename(const char* groupname_current, const char* groupname_new);

/**
 * Copy a group. Creates a new group and copies all ACLs of the source group.
 * @param groupname_src       the name of the source group
 * @param groupname_dest      the name of the new group
 * 
 * @return                    PP_SUC if the group could be copied, errno else
 */
int pp_um_group_copy(const char* groupname_src, const char* groupname_dest);

/**
 * Delete a group. References to this group for users are not updated.
 * @param groupname           the name of the group to delete.
 * 
 * @return                    PP_SUC if the group could be deleted, errno else
 */
int pp_um_group_delete(const char* groupname);

/**
 * Get a list of all existing groupnames. The caller is responsible for
 * freeing the returned vector.
 * @return                    list of all groupnames, NULL on error
 */
vector_t* pp_um_get_all_groups(void);

/**
 * Set a permission for a group to the specified value.
 */
int pp_um_group_set_permission(const char* groupname, 
                               const char* permission_name,
                               const char* permission_value);

/**
 * Returns if the specified group is an individual group or not.
 *
 * @param groupname             the groupname to get the state for
 * @param username              the username (may be NULL) to get the state for
 *                              if group is individual and taken by 'username',
 *                              return state individual but available
 * @param state                 return value, signals current state of group
 *
 * @return                      PP_SUC if group exists and state could be
 *                              determined, PP_ERR in internal error case
 */
int pp_um_group_is_individual_for_user(const char* groupname, 
                                       const char* username, 
                                       pp_um_group_state_t *state);

/**
 * Count users assigned to specified group. Returns -1 (PP_ERR) on internal
 * error (e.g. if the group could not be found), number of users otherwhise 
 * 
 * @param groupname           the groupname to get the id for
 * @return                    number of users assigned to group, -1 on error
 */
int pp_um_group_count_users(const char* groupname);

/**
 * Get the group id of the specified group. Needed for access to this groups
 * configuration settings. Returns -1 (PP_ERR) if the group could not be found
 * (TODO?! or the user has no local writeable configuration.) 
 * 
 * @param groupname           the groupname to get the id for
 * @return                    the group id of the specified group, -1 on error
 */
int pp_um_group_get_gid(const char* groupname);

/**
 * Check a permission for a group. Technically permissions are not restricted
 * to the (Y/N) or (none/view/control) permissions defined in RAASIP. Although
 * it is possible to use abritrary permission, it is highly recommended to stick
 * to the RAASIP permission scheme to unify user experience.
 * 
 * Existing permissions and their possible values are configured in the config
 * system and retrieved with 'acl.h'.
 * 
 * @param groupname          group reference
 * @param permission_name    the name of the permission (e.g. 'port_0')
 * @param permission_value   the desired permission value, 
 *                           (eg. 'view' or 'control')
 * 
 * @return                   PP_SUC if the group has the permission
 *                           errno with verbose string else (no permission,
 *                           internal error, ...)
 */
int pp_um_group_has_permission(const char* groupname,
                               const char* permission_name,
                               const char* permission_value);

int pp_um_gid_has_permission(  int gid,
                               const char* permission_name,
                               const char* permission_value);

/**
 * Get a permission for a group.
 * Remember to free returned permission_value!
 * 
 * @param groupname          group reference
 * @param permission_name    the name of the permission (e.g. 'port_0')
 * @param permission_value   return value, not set in PP_ERR case
 * 
 * @return                   PP_SUC if the groups permission could be retrived,
 *                           errno with verbose string else (no permission,
 *                           internal error, ...)
 */
int pp_um_group_get_permission(const char* groupname,
                               const char* permission_name,
                               char** permission_value);

/**
 * Get a permission for a group.
 * Remember to free returned permission_value!
 *
 * @param gid                group id
 * @param permission_name    the name of the permission (e.g. 'port_0')
 * @param permission_value   return value, not set in PP_ERR case
 *
 * @return                   PP_SUC if the groups permission could be retrived,
 *                           errno with verbose string else (no permission,
 *                           internal error, ...)
 */
int pp_um_gid_get_permission  (int gid,
                               const char* permission_name,
                               char** permission_value);

/**
 * Check a permission of a group to connect from given IP. 
 * Magic as follows:
 * - access counter is initialized with default group ACL policy
 *   (1 for ACCEPT, 0 for DENY)
 * - each ACCEPT-rule that matches group and ip increments access counter
 * - each DENY-rule that matches group and ip decrements access counter
 * - return PP_SUC for group ACL disabled or access counter > 0,
 *          PP_ERR on error or access counter <= 0
 * 
 * @param groupname          group reference
 * @param ip                 IP address xxx.xxx.xxx.xxx, the ACL is checked for
 * 
 * @return                   PP_SUC if the group has the permission
 *                           PP_ERR (and errno) else
 */
int pp_um_group_is_allowed_from_ip_str(const char* groupname, const char* ip);

/**
 * Check a permission of a user to connect from given IP. 
 * @see pp_um_group_is_allowed_from_ip_str
 */
int pp_um_user_is_allowed_from_ip_str(const char* groupname, const char* ip);

// some unsorted functions ...

/**
 * Initialize the usermanagement library.
 * @return                      PP_SUC or errno
 */
int pp_um_init(void);

/**
 * Cleanup the usermanagement library.
 */
void pp_um_cleanup(void);

/**
 * Flush the local database to disc. This function must be called after
 * all data has been written to make the changes persistant.
 * @returns  PP_SUC or errno
 */
/* rwa: commented out - not implemented */
/*int pp_um_flush(void);*/

/**
 * Returns if and for how long the specified user is blocked.
 */
int pp_um_user_is_blocked(const char* name, int* mins_left);

/**
 * Unblock a blocked user by resetting his remaining block time.
 */
int pp_um_user_unblock(const char* name);

/**
 * Checks, whether LDAP is active for user or not
 *
 * @returns                     PP_SUC if LDAP is active for user, 
 *                              PP_ERR (and errno) if not or error occured
 */
int pp_um_user_ldap_auth(const char *name);

/**
 * Returns if this error is specific to the um (otherwise its
 * a general error.
 *
 * @param error                 the errorcode to check
 * 
 * @return                      1 if um specific error, 0 otherwhise
 */
int pp_um_has_errno(int error);

/**
 * Returns errno base for um errors.
 * 
 * @return                      errno base for um errors
 */
int pp_um_errno_base(void);

// known errors, not complete yet
// Attention: This must match the pp_um_errors array in um.c!
#define PP_UM_ERR_NO_ERROR              0
#define PP_UM_ERR_INTERNAL_ERROR        (pp_um_errno_base() +  0)
#define PP_UM_ERR_AUTH_FAILED           (pp_um_errno_base() +  1)
#define PP_UM_ERR_USER_IS_BLOCKED       (pp_um_errno_base() +  2)
#define PP_UM_ERR_INVALID_USERNAME      (pp_um_errno_base() +  3)
#define PP_UM_ERR_INVALID_GROUPNAME     (pp_um_errno_base() +  4)
#define PP_UM_ERR_NOT_FOR_BUILTIN_USR   (pp_um_errno_base() +  5)
#define PP_UM_ERR_NOT_FOR_BUILTIN_GRP   (pp_um_errno_base() +  6)
#define PP_UM_ERR_INVALID_PERM          (pp_um_errno_base() +  7)
#define PP_UM_ERR_USER_ALREADY_EXISTS	(pp_um_errno_base() +  8)
#define PP_UM_ERR_GROUP_ALREADY_EXISTS	(pp_um_errno_base() +  9)
#define PP_UM_ERR_USER_DOESNT_EXIST 	(pp_um_errno_base() + 10)
#define PP_UM_ERR_GROUP_DOESNT_EXIST	(pp_um_errno_base() + 11)
#define PP_UM_ERR_GROUP_IS_INDIVIDUAL   (pp_um_errno_base() + 12)
#define PP_UM_ERR_DEVICE_IS_CC_MANAGED  (pp_um_errno_base() + 13)
#define PP_UM_ERR_NO_PLAINTEXT_PASSWD   (pp_um_errno_base() + 14)
#define PP_UM_ERR_PASSWORD_DIDNT_CHANGE (pp_um_errno_base() + 15)
#define PP_UM_ERR_TOO_MANY_USERS        (pp_um_errno_base() + 16)
#define PP_UM_ERR_USER_DISABLED         (pp_um_errno_base() + 17)
#define PP_UM_ERR_INVALID_IPMI_UID      (pp_um_errno_base() + 18)
#define PP_UM_ERR_INVALID_GID           (pp_um_errno_base() + 19)
//#define PP_UM_ERR_(pp_um_errno_base() + )

/**
 * encrypts and base64encodes a string
 *
 * @param passwd                plain text password
 *
 * @return                      encrypted encoded password string
 */
char* pp_um_passwd_encrypt(const char* passwd);

#ifdef PP_FEAT_AUTH_RADIUS
/* radius specific functions */

int pp_um_radius_init(void);
void pp_um_radius_close_handles(void);
int pp_um_radius_auth_request(int* is_auth, const char *name, const char *pwd, char **groupname, int *sid);
int pp_um_radius_acct_request(const char *name, int sid, int req_type);
int pp_um_radius_is_active(void);
int pp_um_radius_for_principal_auth(const char * name);
#endif /* PP_FEAT_AUTH_RADIUS */

#ifdef PP_FEAT_IPMI_SERVER
/**
 * Set a permission for a user via BMC (IPMI).
 *
 * IPMI doesn't know groups, so permissions - originally group settings - are
 * treated like user settings when accessing over IPMI. This has side effects,
 * as changing an IPMI "user-setting" (e.g. the Privilege Level) would affect
 * other users (the ones assigned to target-user's group) as well.
 * => Care about that!
 * Magic as follows:
 * If user is the only member of a group (or already has individual group),
 * set pemission and return - the easy one. If user is member of a generic
 * group, already taken by other users, create individual group by copying
 * current group and treat as described.
 *
 * @see pp_um_group_set_permission
 * 
 * @param username           user reference
 * @param permission_name    the name of the permission (e.g. 'port_0')
 * @param permission_value   the desired permission value, 
 *                           (eg. 'view' or 'control')
 * 
 * @return                   PP_SUC if the users permission could be set,
 *                           errno with verbose string else (no permission,
 *                           internal error, ...)
 */
int pp_um_ipmi_user_set_permission(const char* username,
                                   const char* permission_name,
                                   const char* permission_value);
#endif /* PP_FEAT_IPMI_SERVER */

#if defined(PP_FEAT_STRONG_PASSWORDS)
/**
 * Checks whether a the password meets Raritan common security requirements
 * (username dependant).
 * If not NULL, errstr will be set to a message buffer explaining the password
 * policy in error case.
 * NOTE! Remember to free this buffer in calling code!
 * 
 * @param username           user reference
 * @param password           plaintext password
 * @param old_password       plaintext current password
 * @param errstr             return value, is set to constant message buffer if
 *                           not NULL
 *
 * @return                   PP_SUC if the password meets Raritan common
 *                           security requirements, PP_ERR, errno and (if set)
 *                           errstr otherwhise
 */
int pp_um_strong_pw_check_for_user(const char *username, const char *password,
                                   const char *old_password, char **errstr);

/**
 * Adds expired password to password history.
 *
 * @param username           name of user to store password for
 * @param password           password to store in history
 * @param flags              flags characterizing old_password (plain / hash)
 *
 * @return                   PP_SUC on success, PP_ERR and errno otherwhise
 */
int pp_um_strong_pw_add_to_history(const char *username, 
                                   const char *password, int flags);
#endif /* PP_FEAT_STRONG_PASSWORDS */

#ifdef __cplusplus
}
#endif

#endif /* __PP_UM_H__ */

