#include <stdio.h>
#include <stdlib.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <radlib.h>
#include <pp/cfg.h>
#include <pp/um.h>

static const char * auth_mode_key = "auth_mode";
static volatile int session_id = 0;

struct rad_handle *rad_auth_h = NULL;
struct rad_handle *rad_acct_h = NULL;

static pthread_mutex_t radius_mutex = PTHREAD_ERRORCHECK_MUTEX_INITIALIZER_NP;

static void get_nas_ip(struct in_addr *ipv4_addr);

#define noRAD_DEBUG
#ifdef RAD_DEBUG
#define RD(fmt, args...)	{ printf(fmt, ##args); }
#else
#define RD(fmt, args...)
#endif

int
pp_um_radius_init(void)
{
    int ret = -1;
    char *server;
    char *secret;
    u_short auth_port, acct_port;
    u_int i, server_cnt, timeout, retries;

    MUTEX_LOCK(&radius_mutex);
    
    rad_auth_h = rad_auth_open();
  
    if (!rad_auth_h) {
	pp_log("RADIUS auth open failed!\n");
	goto bail;
    }
    rad_acct_h = rad_acct_open();
    if ( !rad_acct_h ) {
	pp_log("RADIUS acc open failed!\n");
	goto bail;
    }
    
    ret = 0;

    pp_cfg_get_uint(&server_cnt, "radius._s_");
    for (i = 0; i < server_cnt; i++) {
	if (PP_SUCCED(pp_cfg_get_nodflt(&server, "radius[%u].server", i))) {
	    pp_cfg_get(&secret, "radius[%u].secret", i);
	    pp_cfg_get_ushort(&auth_port, "radius[%u].auth_port", i);
	    pp_cfg_get_ushort(&acct_port, "radius[%u].acct_port", i);
	    pp_cfg_get_uint(&timeout, "radius[%u].timeout", i);
	    pp_cfg_get_uint(&retries, "radius[%u].retries", i);
	    if (rad_add_server(rad_auth_h, server, auth_port, secret, timeout, retries) < 0) {
		pp_log("Failed to add an auth server\n");
		ret = i + 1;	    
	    } else {
		RD("Added radius auth server: %s, %d, %d, %s, %d, %d\n", server, auth_port, acct_port, secret, timeout, retries);
		if (rad_add_server(rad_acct_h, server, acct_port, secret, timeout, retries) < 0) {
		    pp_log("Failed to add an accounting server\n");
		    ret = i + 1;
		} else {
		    RD("Added radius acct server: %s, %d, %d, %s, %d, %d\n", server, auth_port, acct_port, secret, timeout, retries);
		}
	    }
	    free(server);
	    free(secret);
	}
    }

 bail:
    MUTEX_UNLOCK(&radius_mutex);
    return ret;
}

void
pp_um_radius_close_handles(void)
{
    MUTEX_LOCK(&radius_mutex);   
    if ( rad_auth_h) rad_close(rad_auth_h);
    if ( rad_acct_h) rad_close(rad_acct_h);
    rad_auth_h = NULL;
    rad_acct_h = NULL;
    MUTEX_UNLOCK(&radius_mutex);
}

int
pp_um_radius_auth_request(int* is_auth, const char *name, const char *pwd,
			  char **groupname, int *sid)
{
    const char * fn = ___F;
    int    ret = PP_ERR;
    int    rsp_code  = 0;
    int    attr_type = 0;
    int    chap_enable = 0;
    size_t len = 0;
    
    char   s_id[9];
    char   *filter_id;
    char chap_passwd[17];
 
    const void *data = NULL;
    struct in_addr ipv4_addr;

    assert(is_auth);
    if (groupname) *groupname = NULL;
    
    // do nothing for the super user
    if (!pp_um_radius_for_principal_auth(name)) return PP_ERR;

    // check if need chap password.
    if (PP_ERR == pp_cfg_is_enabled(&chap_enable, "radius_chap_enabled")){
	pp_log("%s(): cfg check failed for <radius_chap_enabled>!\n", fn);
	goto bail;
    }

    MUTEX_LOCK(&radius_mutex);

    if (!rad_acct_h) {
	pp_log("%s(): No RADIUS handle!\n", fn);
	goto bail;
    }
       
    if ( rad_create_request(rad_auth_h,  RAD_ACCESS_REQUEST) < 0 ) {
	pp_log("%s(): Creation of RADIUS request failed!\n", fn);
	goto bail;
    }

    snprintf(s_id, sizeof(s_id), "%08d", ++session_id);
    get_nas_ip(&ipv4_addr);
    if (sid) *sid = session_id;

    if (( rad_put_int(rad_auth_h, RAD_NAS_PORT_TYPE, RAD_VIRTUAL)  < 0 ) ||
	( rad_put_addr(rad_auth_h, RAD_NAS_IP_ADDRESS, ipv4_addr)  < 0 ) ||
	( rad_put_string(rad_auth_h, RAD_USER_NAME, name)  < 0 ) ||
	( rad_put_string(rad_auth_h, RAD_ACCT_SESSION_ID, s_id)  < 0 )) {
	pp_log("%s(): Adding RADIUS attributes failed\n", fn);
	goto bail;
    }

    if (chap_enable) {
	// CHAP password encryption
	rad_get_chap_challenge(rad_auth_h, pwd, chap_passwd);
	if ( rad_put_attr(rad_auth_h, RAD_CHAP_PASSWORD, chap_passwd, sizeof(chap_passwd)) < 0 ) {
            pp_log("%s(): Adding RADIUS attribute RAD_CHAP_PASSWORD failed\n", fn);
            goto bail;
        }
    } else {
	if ( rad_put_string(rad_auth_h, RAD_USER_PASSWORD, pwd) < 0 ) {
	    pp_log("%s(): Adding RADIUS attribute RAD_USER_PASSWORD failed\n", fn);
	    goto bail;
	}
    }

    if ( (rsp_code = rad_send_request(rad_auth_h)) < 0 ) {
        pp_log("%s(): RADIUS ACCESS request failed\n", fn);
	goto bail;
    } else {
	RD("RADIUS ACCESS Request return with %d\n", rsp_code);
	*is_auth = (rsp_code == RAD_ACCESS_ACCEPT) ? 1 : 0;
    }

    // try to extract Raritan group from RADIUS attributes
    // (only if requested by callee)
    if (groupname)
    {
        while ( (attr_type = rad_get_attr(rad_auth_h, &data, &len)) > 0 ){
            /* retrieve group name from response */
            if(*groupname == NULL && attr_type == RAD_FILTER_ID && len > 10){
                filter_id = (char *) malloc(len+1);
                memcpy(filter_id, data, len);
                filter_id[len] = 0;

                char *ptr, *ptr1;
                if((ptr = strstr(filter_id, "Raritan:G{")) != NULL){
                    *groupname = (char *) malloc(len+1);
                    ptr = ptr + 10; // point to beginning of groupname
                    if((ptr = strtok_r(ptr, "}", &ptr1)) != NULL) {
                        *groupname = strdup(ptr);
                    }
                }
                free(filter_id); 
            }
        }

        if ( attr_type < 0 ) {
            pp_log("%s(): Getting RADIUS attributes failed!\n", fn);
        }
    }

    ret = PP_SUC;
    
 bail:
    MUTEX_UNLOCK(&radius_mutex);
    return ret;
}

int
pp_um_radius_acct_request(const char *name, int sid, int req_type)
{
    const char * fn = ___F;
    char * auth_mode;
    char s_id[9];
    struct in_addr ipv4_addr;
    int rsp_code  = 0;
    int attr_type = 0;
    const void *data = NULL;
    size_t len = 0;
    int ret = PP_ERR;

    /* silently return if authmode != radius */
    pp_cfg_get(&auth_mode, "auth_mode");
    if (strcmp(auth_mode, "radius")) {
	free(auth_mode);
	return PP_SUC;
    }
    free(auth_mode);

    // do nothing for the super user
    if (!pp_um_radius_for_principal_auth(name)) return PP_SUC;

    MUTEX_LOCK(&radius_mutex);

    if (!rad_acct_h) {
	pp_log("%s(): No RADIUS handle!\n", fn);
	goto bail;
    }
       
    if ( rad_create_request(rad_acct_h, RAD_ACCOUNTING_REQUEST) < 0 ) {
	pp_log("%s(): Creation of RADIUS request failed!\n", fn);
	goto bail;
    }

    snprintf(s_id, sizeof(s_id), "%08d", sid);
    get_nas_ip(&ipv4_addr);
    
    if (rad_put_int(rad_acct_h, RAD_ACCT_STATUS_TYPE, req_type) < 0
	|| rad_put_int(rad_acct_h, RAD_NAS_PORT_TYPE, RAD_VIRTUAL) < 0
	|| rad_put_int(rad_acct_h, RAD_NAS_PORT, 0)  < 0
	|| rad_put_addr(rad_acct_h, RAD_NAS_IP_ADDRESS, ipv4_addr) < 0
	|| rad_put_string(rad_acct_h, RAD_USER_NAME, name) < 0
	|| rad_put_string(rad_acct_h, RAD_ACCT_SESSION_ID, s_id) < 0) {
	pp_log("%s(): Adding RADIUS attributes failed\n", fn);
	goto bail;
    }

    if ((rsp_code = rad_send_request(rad_acct_h)) < 0) {
        pp_log("%s(): RADIUS ACCOUNTING request failed\n", fn);
	goto bail;
    } else {
	RD("RADIUS ACCOUNTING Request return with %d\n", rsp_code);
    }

    while ((attr_type = rad_get_attr(rad_acct_h, &data, &len)) > 0);

    if (attr_type < 0) pp_log("%s(): Getting RADIUS attributes failed!\n", fn);

    ret = PP_SUC;

 bail:
    MUTEX_UNLOCK(&radius_mutex);
    return ret;
}

int
pp_um_radius_is_active(void)
{
    char *auth_mode = NULL;
    int ret = 0;

    if (PP_FAILED(pp_cfg_get(&auth_mode, auth_mode_key))) {
	pp_log("%s(): could not determine auth mode\n", ___F);
	goto bail;
    }

    if (!pp_strcmp_safe(auth_mode, "radius")) {
	ret = 1;
    }
    
 bail:
    RD("%s() ret=%d\n", ___F, ret);
    if (auth_mode) free(auth_mode);
    return ret;
}

int 
pp_um_radius_for_principal_auth(const char * name)
{
    int use_radius = 1, is_root = 0;
    assert(name);

    if (PP_FAILED(pp_um_user_is_root(&is_root, name))) {
	goto bail;
    }

    use_radius = !is_root;    
    
 bail:
    RD("%s() use_radius=%d\n", ___F, use_radius);
    return use_radius;
}

static void
get_nas_ip(struct in_addr *ipv4_addr)
{
    char *if_name;
    char ipv4_string[INET_ADDRSTRLEN];

    pp_cfg_get(&if_name, "network.eth_interface");
    pp_get_linklocal_ipv4_address(if_name, ipv4_string, sizeof(ipv4_string));
    (*ipv4_addr).s_addr = inet_addr(ipv4_string);
    free(if_name);
}
