/**
 * User manangement routines
 * This library implements RAASIP compatible usermanagement and replaces
 * the old libpp_um (that was based on permission inheritance).
 * 
 * All functions in this library operate (unless otherwise noted)
 * on the config system. That means they operate either on the local
 * flashdisk (fully writable) or on a remote ldap server (mostly
 * readonly).
 *
 * (c) 2006 Peppercon AG, geo@peppercon.de, tweb@peppercon.de
 */

// system includes
#include <arpa/inet.h> // inet_addr

// fw includes
#include <pp/um.h>
#include <pp/xdefs.h>

// local includes
#include "um_intern.h"
#include "um_cfg_keys.h"

const char um_individual_group_prefix = '@';

static pp_hash_t *um_gid_by_groupname = NULL;
static pthread_mutex_t um_group_mtx = PTHREAD_RECURSIVE_MUTEX_INITIALIZER_NP;

static int next_free_gid = 3;

static inline int get_next_free_gid(void) {
    return next_free_gid++;
}

int um_group_init() {
    assert(um_gid_by_groupname == NULL);
    
    um_gid_by_groupname = pp_hash_create(100);
    um_load_all_groups();

    return PP_SUC;
}

void um_group_cleanup(void) {
    assert(um_gid_by_groupname);

    pp_hash_delete(um_gid_by_groupname);
    um_gid_by_groupname = NULL;
}

int pp_um_group_create(const char* groupname, int is_individual) {
    int ret = PP_SUC;
    int gid = -1, *gid_p;
    
    MUTEX_LOCK(&um_group_mtx);
    
    assert(groupname);
    assert(um_gid_by_groupname);
    
    /* individual check */
    if(is_individual != (*groupname == um_individual_group_prefix)) {
        errno = PP_UM_ERR_INVALID_GROUPNAME;
        ret = PP_ERR;
        goto bail;
    }
    
    /* new group unique? */
    if (pp_hash_get_entry(um_gid_by_groupname, groupname) != NULL) {
        errno = PP_UM_ERR_GROUP_ALREADY_EXISTS;
        ret = PP_ERR;
        goto bail;
    }
    
    gid = get_next_free_gid();
    assert(gid > 0);

    gid_p = (int*)malloc(sizeof(int));
    *gid_p = gid;
    pp_hash_set_entry(um_gid_by_groupname, groupname, gid_p, free);
    
    /* set groupname */
    ret = pp_cfg_set(groupname, grp_name_key, gid);
    
 bail:
    if(ret == PP_ERR && gid > 0) {
        /* something's wrong, do not create group -> cover the tracks */
        pp_cfg_remove(grp_key, gid);
        pp_hash_delete_entry(um_gid_by_groupname, groupname);
    }
 
    MUTEX_UNLOCK(&um_group_mtx);
    return ret;
}

int pp_um_group_rename(const char* groupname_current, 
                       const char* groupname_new) {
    int ret = PP_SUC;
    int gid, *gid_p;
    
    assert(groupname_current);
    assert(groupname_new);
    assert(um_gid_by_groupname);
    
    /* no rename if current and new name are identical */
    if (!strcmp(groupname_current, groupname_new)) {
        return PP_SUC;
    }

    MUTEX_LOCK(&um_group_mtx);
    
    /* old name exists? */
    if (PP_ERR == (gid = pp_um_group_get_gid(groupname_current))) {
        /* errno is set by pp_um_group_get_gid */
        ret = PP_ERR;
        goto bail;
    }

    /* new name unique? */
    if (pp_hash_get_entry(um_gid_by_groupname, groupname_new) != NULL) {
        errno = PP_UM_ERR_GROUP_ALREADY_EXISTS;
        ret = PP_ERR;
        goto bail;
    }
    
    /* builtin group? */
    if (um_group_is_renameable(gid) == PP_ERR) {
        errno = PP_UM_ERR_NOT_FOR_BUILTIN_GRP;
        ret = PP_ERR;
        goto bail;
    }
    
    pp_cfg_set(groupname_new, grp_name_key, gid);

    pp_hash_delete_entry(um_gid_by_groupname, groupname_current);
    gid_p = (int*)malloc(sizeof(int));
    *gid_p = gid;
    pp_hash_set_entry(um_gid_by_groupname, groupname_new, gid_p, free);
    
 bail:
    MUTEX_UNLOCK(&um_group_mtx);
    
    return ret;
}

int pp_um_group_copy(const char* groupname_src, const char* groupname_dest) {
    int ret = PP_SUC;
    char gid_buffer[10];
    int gid_src, gid_dest, *gid_p;
    
    assert(groupname_src);
    assert(groupname_dest);
    assert(um_gid_by_groupname);
    
    MUTEX_LOCK(&um_group_mtx);
    
    /* old name exists? */
    if (PP_ERR == (gid_src = pp_um_group_get_gid(groupname_src))) {
        /* errno is set by pp_um_group_get_gid */
        ret = PP_ERR;
        goto bail;
    }

    /* new name unique? */
    if (pp_hash_get_entry(um_gid_by_groupname, groupname_dest) != NULL ||
        !strcmp(groupname_src, groupname_dest)) {
        errno = PP_UM_ERR_GROUP_ALREADY_EXISTS;
        ret = PP_ERR;
        goto bail;
    }
    
    gid_dest = get_next_free_gid();
    snprintf(gid_buffer, 9, "%d", gid_dest);
    if(PP_ERR == (ret = pp_cfg_copy(gid_buffer, grp_key, gid_src))) {
        /* todo errno?? */
        goto bail;
    }

    pp_cfg_set(groupname_dest, grp_name_key, gid_dest);
    gid_p = (int*)malloc(sizeof(int));
    *gid_p = gid_dest;
    pp_hash_set_entry(um_gid_by_groupname, groupname_dest, gid_p, free);
    
 bail:
    MUTEX_UNLOCK(&um_group_mtx);
    
    return ret;
}

int pp_um_group_delete(const char* groupname) {
    int gid;
    int ret = PP_ERR;
    
    assert(um_gid_by_groupname);

    MUTEX_LOCK(&um_group_mtx);

    /* name exists? */
    if (PP_ERR == (gid = pp_um_group_get_gid(groupname))) {
        /* errno is set by pp_um_group_get_gid */
        goto bail;
    }

    if(gid < 0) {
        errno = PP_UM_ERR_INVALID_GROUPNAME;
    } else if(um_group_is_deleteable(gid) == PP_ERR) {
        errno = PP_UM_ERR_NOT_FOR_BUILTIN_GRP;
    } else {
        if(PP_SUC == (ret = pp_cfg_remove(grp_key, gid))) {
            pp_hash_delete_entry(um_gid_by_groupname, groupname);
            pp_cfg_save(DO_FLUSH);
        }
    }
    
 bail:
    MUTEX_UNLOCK(&um_group_mtx);
    return ret;
}

vector_t* pp_um_get_all_groups() {
    vector_t* groups = NULL;
    const char *group;
    
    assert(um_gid_by_groupname);
    
    MUTEX_LOCK(&um_group_mtx);
    
/* FIXME: CC integration */
/* FIXME: LDAP */
    groups = vector_new(NULL, 10, free);
    
    for(group = pp_hash_get_first_key(um_gid_by_groupname); group;
        group = pp_hash_get_next_key(um_gid_by_groupname)) {
        vector_add(groups, strdup(group));
    }
    
    if(vector_size(groups)) {
        vector_qsort(groups, str_compare);
    } else {
        vector_delete(groups);
        groups = NULL;
    }

    MUTEX_UNLOCK(&um_group_mtx);
    return groups;
}

int um_group_validate_remote(const char* groupname )
{
    int gid;

#if defined(PP_FEAT_HAS_NONE_GROUP)
    if(groupname == NULL){
	/* no group name, assign group <NONE> */
	gid = 2;
    } else
#endif /* PP_FEAT_HAS_NONE_GROUP */
    if(PP_ERR == (gid = pp_um_group_get_gid(groupname))) {
        /* remote_group doesn't exist on device, assign group <Unknown> */
        gid = 1;
    }

    return gid;
}

int pp_um_group_set_permission(const char* groupname, 
                               const char* permission_name,
                               const char* permission_value) {
    int gid;
    
    if(PP_ERR == (gid = pp_um_group_get_gid(groupname))) {
        /* errno is set by pp_um_group_get_gid */
        return PP_ERR;
    }

    return um_group_set_permission(gid, permission_name, permission_value);
}

int um_group_set_permission(int gid, 
                            const char* permission_name,
                            const char* permission_value) {
    int ret = PP_ERR;
    
    if(gid < 0) {
        /* should not happen */
        assert(gid >= 0);
        errno = PP_UM_ERR_INTERNAL_ERROR;
    } else if(um_group_is_modifiable(gid) == PP_ERR) {
        errno = PP_UM_ERR_NOT_FOR_BUILTIN_GRP;
    } else {
        pp_acl_t *acl = NULL;

        if(pp_acl_objects_has_perm(permission_name, permission_value)) {
            ret = pp_cfg_set(permission_value, grp_acl_key, 
                             gid, permission_name);
        } else {
            errno = PP_UM_ERR_INVALID_PERM;
        }

        pp_profile_release_acl(acl);
    }
    
    return ret;
}

int pp_um_group_is_individual_for_user(const char* groupname, 
                                       const char* username, 
                                       pp_um_group_state_t *state) {
    int individual, ret = PP_SUC, gid;
    
    MUTEX_LOCK(&um_group_mtx);

    if(PP_ERR == (gid = pp_um_group_get_gid(groupname))) {
        /* errno is set by pp_um_group_get_gid */
        ret = PP_ERR;
        goto bail;
    }
        
    individual = *groupname == um_individual_group_prefix;
    
    if(!individual) {
        /* the easy one */
        *state = PP_UM_GROUP_GENERIC;
    } else {
        vector_t *users = pp_um_get_all_users();
        int users_sz = vector_size(users), i;
        
        for(i = 0; i < users_sz; ++i) {
            char *user = (char*)vector_get(users, i);
            int group_id = pp_um_user_get_gid(user);
            if(gid == group_id && (!username || strcmp(username, user))) {
                /* user's group is searched one, so group is taken.
                   if username specified and matches user, group is taken by
                   exactly this user and thus available */
                *state = PP_UM_GROUP_INDIVIDUAL_TAKEN;
                break;
            }
        }
        
        if(i >= users_sz) {
            /* we ran trough users and didn't find anything */
            *state = PP_UM_GROUP_INDIVIDUAL_AVAILABLE;
        }
        
        vector_delete(users);
    }
    
 bail:
    MUTEX_UNLOCK(&um_group_mtx);
    return ret;
}

int pp_um_group_count_users(const char* groupname) {
    vector_t *users;
    int gid, users_sz, i, users_in_group = 0;
    
/* FIXME: CC integration */
/* FIXME: LDAP */
    if(!groupname) {
        errno = PP_UM_ERR_INTERNAL_ERROR;
        return PP_ERR;
    }
    
    if(PP_ERR == (gid = pp_um_group_get_gid(groupname))) {
        /* errno is set by pp_um_group_get_gid */
        return PP_ERR;
    }
    
    users = pp_um_get_all_users();
    users_sz = vector_size(users);
        
    for(i = 0; i < users_sz; ++i) {
        int group_id = pp_um_user_get_gid((char*)vector_get(users, i));
        if(gid == group_id) {
            users_in_group++;
        }
    }
    
    return users_in_group;
}

int pp_um_group_get_gid(const char* groupname) {
    int *gid;
    
    assert(um_gid_by_groupname);

/* FIXME: CC integration */
/* FIXME: LDAP */
    if(!groupname || !*groupname) {
        errno = PP_UM_ERR_INTERNAL_ERROR;
        return PP_ERR;
    }
    
    if(NULL != (gid = (int*)pp_hash_get_entry(um_gid_by_groupname, 
                                              groupname))) {
        return *gid;
    }
    
    errno = PP_UM_ERR_GROUP_DOESNT_EXIST;
    
    return PP_ERR;
}

void um_load_all_groups() {
    pp_profile_type_t prof[] = { PP_PROFILE_LOCAL, PP_PROFILE_LOCAL_CODE };
    pp_cfg_iter_t* iter;
    const char *gidstr;
    char *groupname;
    unsigned int i;
    int gid;

    assert(um_gid_by_groupname);

    for (i = 0; i < sizeof(prof)/sizeof(pp_profile_type_t); ++i) {
	if (PP_ERR != pp_cfg_iter_create_at_layer(&iter, prof[i], grp_vector)) {
	    while (pp_cfg_iter_next(iter, &gidstr, NULL)) {
                int *gid_p;
                
		gid = pp_strtol(gidstr, -1, 10, NULL);
		if (gid < 0) {
		    pp_log("invalid group id '%s'\n", gidstr);
		    continue;
		}
		
                /* do not get at layer, fallback for default group(s) */
		if (PP_ERR == pp_cfg_get_nodflt(&groupname, grp_name_key, gid)){
                    pp_log("group name not set for group '%d'\n", gid);
                    continue;
                }

		// check wether already there
		if (pp_hash_get_entry(um_gid_by_groupname, groupname) != NULL) {
		    free(groupname);
		    continue;
		}

		gid_p = (int*)malloc(sizeof(int));
                *gid_p = gid;
                pp_hash_set_entry(um_gid_by_groupname, groupname, gid_p, free);

                if (gid >= next_free_gid) {
                    next_free_gid = gid + 1;
                }

		free(groupname);
	    }
	    pp_cfg_iter_destroy(iter);
	}
    }
}

int pp_um_group_has_permission(const char* groupname,
                               const char* permission_name,
                               const char* permission_value) {
    int gid;
    
    if(PP_ERR == (gid = pp_um_group_get_gid(groupname))) {
        /* errno is set by pp_um_group_get_gid */
        return PP_ERR;
    }

    return um_group_has_permission(gid, permission_name, permission_value);
}

int pp_um_gid_has_permission(  int gid,
                               const char* permission_name,
                               const char* permission_value) {

    return um_group_has_permission(gid, permission_name, permission_value);
}

int pp_um_group_get_permission(const char* groupname,
                               const char* permission_name,
                               char** permission_value) {
    int gid;
    
    if (!groupname || !permission_name || !permission_value) {
        errno = PP_UM_ERR_INTERNAL_ERROR;
        return PP_ERR;
    }

    assert(groupname);
    assert(permission_name);
    assert(permission_value);
    
    if(PP_ERR == (gid = pp_um_group_get_gid(groupname))) {
        /* errno is set by pp_um_group_get_gid */
        return PP_ERR;
    }

    return um_group_get_permission(gid, permission_name, permission_value);
}

int pp_um_gid_get_permission  (int gid,
                               const char* permission_name,
                               char** permission_value) {

    if (!permission_name || !permission_value) {
        errno = PP_UM_ERR_INTERNAL_ERROR;
        return PP_ERR;
    }

    if (gid < 0) {
	errno = PP_UM_ERR_INVALID_GID;
	return PP_ERR;
    }

    return um_group_get_permission(gid, permission_name, permission_value);
}

int um_group_has_permission(int gid,
                            const char* permission_name,
                            const char* permission_value) {
    int ret = PP_ERR;
    
/* FIXME: TODO RAASIP! */
#if 0 // tmp RAASIP
#warning TODO LDAP/local decision really necessary?
    if (pp_um_ldap_for_principal(name)) {
        pp_acl_t* acl;
	if(uid==-1) goto bail;
	if (PP_FAILED(pp_cfg_get_acl(&acl, "user[%u]", uid))) goto bail;
	ret = pp_acl_accumulate_all_permissions(acl, NULL, object, perm);

        pp_profile_release_acl(acl);
    } else {
	ret = pp_um_P rincip al_ac cumulate_pe rmission(name, object, perm, PP_UM_ACCUMULATE_ALL);
    }
#endif // 0     
    if(!gid && pp_acl_objects_has_perm(permission_name, permission_value)) {
        /* Admin group, allow all */
        ret = PP_SUC;
    } else if(permission_name && *permission_name == '\0') {
        /* acl-object is "", this is anyone */
        ret = PP_SUC;
    } else {
        pp_acl_t *acl = NULL;
        if (PP_SUCCED(pp_cfg_get_acl(&acl, "group[%u]", gid))) {
            pp_acl_entry_t* ae;
            char *real_permission_name;
            if ((real_permission_name = pp_acl_lookup_real_permission_name(permission_name))) {
                /* for all acl entries that match the object and permission */
                for (ae = pp_hash_get_first_entry(acl->acl); ae != NULL;
                     ae = pp_hash_get_next_entry(acl->acl)) {
                    if (!strcmp(ae->object->name, real_permission_name) &&
                        !strcmp(ae->perm->name, permission_value)) {
                        ret = PP_SUC;
                        break;
                    }
                }
                free(real_permission_name);
            }
            pp_profile_release_acl(acl);
        }
    }
    
    return ret;
}

int um_group_get_permission(int gid,
                            const char* permission_name,
                            char** permission_value) {
    int ret;
    char *val = NULL;
    
/* FIXME: CC integration */
/* FIXME: LDAP */

    if(!gid) {
        /* Admin group, return highest permission possible */
        
/* FIXME: type should be enum! */
        char *type = NULL;
        
        if(PP_ERR != (ret = pp_acl_objects_get_type(&type, permission_name))) {
            assert(type);

            if(!strcmp(type, PP_CD_PERMTYPE_ADMIN_STR) || 
               !strcmp(type, PP_CD_PERMTYPE_USER_STR)) {
                /* user or admin perm -> return "yes" */
                *permission_value = strdup(pp_acl_raasip_yes_str);
#if defined(PRODUCT_XX01IP_ANY) || defined(PP_FEAT_EXTENDED_PORT_ACLS)
            } else if(!strcmp(type, PP_CD_PERMTYPE_PORT_STR)) {
                /* port perm, return "control" */
                *permission_value = strdup(pp_acl_raasip_control_str);
#endif /* PRODUCT_XX01IP_ANY || PP_FEAT_EXTENDED_PORT_ACLS */
#if defined(PP_FEAT_EXTENDED_PORT_ACLS)
            } else if(!strcmp(type, PP_CD_PERMTYPE_VM_STR)) {
                /* vm perm, return "readwrite" */
                *permission_value = strdup(pp_acl_raasip_readwrite_str);
#endif /* PP_FEAT_EXTENDED_PORT_ACLS */
#ifdef PP_FEAT_IPMI_SERVER
            } else if(!strcmp(type, PP_CD_PERMTYPE_PRIVLVL_STR)) {
                /* IPMI privilege level, return "oem" (includes admin perm!) */
                *permission_value = strdup(PP_CD_IPMIPRIVLEVEL_OEM_STR);
#endif /* PP_FEAT_IPMI_SERVER */
            } else {
                /* should not happen */
                pp_log("invalid permission type %s\n", type);
                assert(0);
            }
            
            free(type);
        }
    } else if(PP_ERR != (ret = pp_cfg_get_nodflt(&val, grp_acl_key, 
                                                 gid, permission_name))) {
        *permission_value = val;
    }
    
    return ret;
}

int pp_um_group_is_allowed_from_ip_str(const char* groupname, const char* ip) {
    int targid;
    
    if(PP_ERR == (targid = pp_um_group_get_gid(groupname))) {
        /* errno is set by pp_um_group_get_gid */
        return PP_ERR;
    }
    
    return um_group_is_allowed_from_ip_str(targid, ip);
}

int um_group_is_allowed_from_ip_str(int targid, const char* ip) {
    int ret = PP_SUC;
    int enabled;
    
    pp_cfg_is_enabled(&enabled, "security.group_acl.enabled");
    
    if(enabled) {
        /* group acl enabled, collect data */
        u_int32_t s_addr;
        int access_cnt, i, rule_cnt;
        char *default_policy = NULL;
        
/*
        if(!user_ip || user_ip_len < sizeof(struct sockaddr_in) ||
           user_ip->sa_family != AF_INET) {
            // TODO errno?
            return PP_ERR;
        }
        s_addr = (u_int32_t)((struct sockaddr_in*)user_ip)->sin_addr.s_addr;
*/
        s_addr = (u_int32_t)ntohl(inet_addr(ip));
        
        pp_cfg_get(&default_policy, "security.group_acl.default_policy");
        /* initialize access_cnt to 1 if default_policy is ACCEPT, 0 else */
        access_cnt = !strcmp(default_policy, PP_CD_IPFW_POLICY_ACCEPT_STR);
        
        pp_cfg_get_int(&rule_cnt, "security.group_acl.ruleset._s_");

        for(i = 0; i < rule_cnt; ++i) {
            /* check all rules if they affect targid */
            int gid;
            
            if(PP_ERR != pp_cfg_get_int_nodflt(&gid,
                             "security.group_acl.ruleset[%u].gid", i) &&
               (gid == targid || gid == -1 /* all groups */)) {
                char *start_ip = NULL;
                char *stop_ip = NULL;
                u_int32_t s_start, s_stop;
                
                /* get rule's ip range */
                pp_cfg_get(&start_ip,
                           "security.group_acl.ruleset[%u].start_ip", i);
                s_start = (u_int32_t)ntohl(inet_addr(start_ip));
                pp_cfg_get(&stop_ip, 
                           "security.group_acl.ruleset[%u].stop_ip", i);
                s_stop = (u_int32_t)ntohl(inet_addr(stop_ip));
                
                /* check if rule matches given IP */
                if(s_addr >= s_start && s_addr <= s_stop) {
                    char *policy = NULL;
                    
                    pp_cfg_get(&policy,
                               "security.group_acl.ruleset[%u].policy", i);
                    
                    /* increase access_cnt if policy is ACCEPT,
                       decrease if policy is DENY */
                    access_cnt += !strcmp(policy,
                                          PP_CD_IPFW_POLICY_ACCEPT_STR)
                                  ? 1 : -1;
                    
                    free(policy);
                }
                
                free(start_ip);
                free(stop_ip);
            }
        }
        
        ret = access_cnt > 0 ? PP_SUC : PP_ERR;
    
        free(default_policy);
    }
    
    return ret;
}

