/**
 * User manangement routines for access from BMC (IPMI)
 * This library implements RAASIP compatible usermanagement and replaces
 * the old libpp_um (that was based on permission inheritance).
 * 
 * All functions in this library operate (unless otherwise noted)
 * on the config system. That means they operate either on the local
 * flashdisk (fully writable) or on a remote ldap server (mostly
 * readonly).
 *
 * (c) 2006 Peppercon AG, geo@peppercon.de, tweb@peppercon.de
 */

// system includes

// fw includes
#include <pp/um.h>

// local includes
#include "um_intern.h"
#include "um_cfg_keys.h"

int pp_um_ipmi_user_set_permission(const char* username,
                                   const char* permission_name,
                                   const char* permission_value) {
    int gid, ret = PP_ERR;
    char *groupname = NULL;
    pp_um_group_state_t state;
    
    if(PP_ERR == (gid = pp_um_user_get_gid(username)) ||
       PP_ERR == pp_cfg_get_nodflt(&groupname, grp_name_key, gid) ||
       PP_ERR == (ret = pp_um_group_is_individual_for_user(groupname, username,
                                                           &state))) {
        // todo errno?
        ret = PP_ERR;
        goto bail;
    }
    
    if(state == PP_UM_GROUP_GENERIC) {
        /* group is generic, check if current user is the only one in group */
        vector_t *users;
        int users_sz;
        
        if(NULL == (users = pp_um_get_all_users_by_gid(gid))) {
            ret = PP_ERR;
            goto bail;
        }
        
        users_sz = vector_size(users);
        vector_delete(users);
        
        assert(users_sz > 0); // at least current user!
        
        if(users_sz > 1) {
            /* we know, the current user is member of its group assigned.
               if users_sz > 1, he is not the only one */

            char groupbuf[65];
        
            snprintf(groupbuf, sizeof(groupbuf), "%c%s",
                     um_individual_group_prefix, username);
             
            /* create new individual group of specified name */
            while(1) {
                if(PP_SUC == pp_um_group_copy(groupname, groupbuf)) {
                    break;
                } else if (errno != PP_UM_ERR_GROUP_ALREADY_EXISTS || 
                           sizeof(groupbuf) < strlen(groupbuf) + 1) {
                    /* maximum size of groupname reached, break */
                    errno = PP_UM_ERR_INTERNAL_ERROR;
                    goto bail;
                } else {
                    /* group of that name exists, append '_' to groupname
                       and try again */
                    snprintf(groupbuf, sizeof(groupbuf), "%s_", groupbuf);
                }
            }
            
            /* set gid to new group */
            if(PP_ERR == (gid = pp_um_group_get_gid(groupbuf)) ||
               PP_ERR == pp_um_user_set_group(username, groupbuf)) {
                assert(0); // may not happen, we just created that group!
                errno = PP_UM_ERR_INTERNAL_ERROR;
                ret = PP_ERR;
                goto bail;
            }
        }
    }

 bail:
    free(groupname);
    if(ret == PP_ERR) {
        return PP_ERR;
    }
    return um_group_set_permission(gid, permission_name, permission_value);
}
