/**
 * Strong password management routines
 * See common security requirements document
 *
 * (c) 2006 Peppercon AG - a Raritan company, tweb@raritan.com
 */

// system includes
#include <ctype.h>
#include <openssl/md5.h>
#include <openssl/evp.h>

// fw includes
#include <pp/um.h>
#include <pp/intl.h>
#include <liberic_config.h>
#include <liberic_notify.h>

// local includes
#include "um_intern.h"
#include "um_cfg_keys.h"

static const char* strong_pw_len_min_key = "security.strong_pw.length.min";
static const char* strong_pw_len_max_key = "security.strong_pw.length.max";
static const char* strong_pw_char_lower_key =
                       "security.strong_pw.charset.lower";
static const char* strong_pw_char_upper_key =
                       "security.strong_pw.charset.upper";
static const char* strong_pw_char_numeric_key =
                       "security.strong_pw.charset.numeric";
static const char* strong_pw_char_special_key =
                       "security.strong_pw.charset.special";

int pp_um_strong_pw_check_for_user(const char *username, const char *password,
                                   const char *old_password, char **errstr) {
    int usr_do_check = 1;
    int cfg_do_check;
    
    pp_cfg_is_enabled(&cfg_do_check, strong_pw_enabled_key);
    
#if defined(PP_FEAT_STRONG_PASSWORDS_ONLY_ROOT)
    pp_um_user_is_root(&usr_do_check, username);
#endif
        
    if(cfg_do_check && usr_do_check) {
        return um_strong_pw_check_for_user_core(username, password,
                                                old_password, errstr);
    }
    
    /* no need to check for strong password if not enabled */
    return PP_SUC;
}
    
#define RETERR(_fmt_, _args_...) if(errstr) { \
                                     snprintf(buf, 1024, _(_fmt_), ##_args_); \
                                     *errstr = strdup(buf); \
                                 } \
                                 return PP_ERR

int um_strong_pw_check_for_user_core(const char *username, const char *password, 
                                     const char *old_password, char **errstr) {
    int has_upper = 0, has_lower = 0, has_numeric = 0, has_special = 0;
    int upper, lower, numeric, special, history_sz, history_max, i, uid;
    u_int max_len, min_len, pw_len, match_len, user_len, u, cnt;
    const char *c, *h;
    char buf[1024];
    char md5_hash[MD5_DIGEST_LENGTH];
    char md5_hash_str[MD5_DIGEST_LENGTH * 2 + 1];

    assert(username);

    /* check emptyness */
    if (!password) {
        RETERR("The password must not be empty.");
    }
    
    /* check length */
    pw_len = strlen(password);
    pp_cfg_get_uint(&min_len, strong_pw_len_min_key);
    if (pw_len < min_len) {
        RETERR("The password is too short. Minimum length is %d characters.",
               min_len);
    }
    pp_cfg_get_uint(&max_len, strong_pw_len_max_key);
    if (pw_len > max_len) {
        RETERR("The password is too long. Maximum length is %d characters.",
               max_len);
    }
    
    /* check enforced characters */
    for (c = password; *c; c++) {
	if (isupper(*c)) {
	    has_upper = 1;
	} else if (islower(*c)) {
	    has_lower = 1;
	} else if (isdigit(*c)) {
	    has_numeric = 1;
	} else if (ispunct(*c) || *c == ' ') {
	    has_special = 1;
	} else {
            RETERR("The password must not contain control characters.");
	}
    }
    pp_cfg_is_enabled(&lower, strong_pw_char_lower_key);
    if(lower && !has_lower) {
        RETERR("The password has to contain at least one lower character.");
    }
    pp_cfg_is_enabled(&upper, strong_pw_char_upper_key);
    if(upper && !has_upper) {
        RETERR("The password has to contain at least one upper character.");
    }
    pp_cfg_is_enabled(&numeric, strong_pw_char_numeric_key);
    if(numeric && !has_numeric) {
        RETERR("The password has to contain at least one numeric character.");
    }
    pp_cfg_is_enabled(&special, strong_pw_char_special_key);
    if(special && !has_special) {
        RETERR("The password has to contain at least one printable special character.");
    }

    /* check username matching */
    user_len = strlen(username);
    /* check for at least pw_len or user_len, max 4 continous characters */
    match_len = pw_len < user_len ? (pw_len < 4 ? pw_len : 4)
                                  : (user_len < 4 ? user_len : 4);
    for(u = 0, c = username; u < user_len - match_len; ++u, ++c) {
        u_int p;
        const char *d;
        
        for(p = 0, d = password; p < pw_len - match_len; ++p, ++d) {
            if(!strncmp(c, d, match_len)) {
                RETERR("The password and username may match in at least four countinous characters.");
            }
        }
    }
    
    /* check password history */
    if (PP_ERR == (uid = pp_um_user_get_uid(username))) {
        RETERR(pp_error_string(errno)); // ERRNO is set
    }

    pp_cfg_get_int(&history_sz, usr_history_sz_key, uid);
    pp_cfg_get_int(&history_max, strong_pw_history_key);
    if(history_max < history_sz) {
        /* in case we stored more passwords than we currently want to check */
        history_sz = history_max;
    }
    
    MD5(password, strlen(password), md5_hash);
    pp_md5_hash_to_str(md5_hash, md5_hash_str);
    
    for(i = 0; i < history_sz; ++i) {
        char *hash;
        
        pp_cfg_get(&hash, usr_history_key, uid, i);
        
        if(!pp_strcmp_safe(hash, md5_hash_str)) {
            RETERR("The password already is in history.");
        }
    }
    
    /* check significant changes */
    if (old_password) {
        /* old password may be empty, e.g. if changed by admin */

        for(c = password, h = old_password, cnt = 0; *c && *h; ++c, ++h) {
            if(*c != *h) {
                /* char differs */
                ++cnt;
            }
        }
        
        if(cnt < 4) {
            RETERR("The password has to differ from previous in at least 4 characters.");
        }
    }

    return PP_SUC;
}

int um_user_strong_pw_notify_cb(pp_cfg_chg_ctx_t *ctx UNUSED) {
    eric_notify_post_event("Strong password settings changed.", 
                           "security", PP_NOTIFY_EVENT_GENERIC);
    
    return PP_SUC;
}

int um_user_strong_pw_got_enabled_cb(pp_cfg_chg_ctx_t *ctx) {
    int ret = PP_SUC;
    int need_pw_change = 0;
    
    /**
     * figure out, if strong password support changed from disabled to enabled
     */
    if(ctx && ctx->key_comp_vec) {
        pp_cfg_tx_t *tx;
        int i, kcv_sz = vector_size(ctx->key_comp_vec);
        
        /* FIXME: this is a dirty hack...
           find workaround to bypass TX on config get! */
        tx = pp_cfg_tx_get_thread_ctx();
        pp_cfg_tx_release_thread_ctx(NULL);
               
        for(i = 0; !need_pw_change && (i < kcv_sz); ++i) {
            vector_t *key_comps = (vector_t*)vector_get(ctx->key_comp_vec, i);
            int enabled;
            char cfg_key[MAX_OPT_KEY_LEN];
            
            pp_cfg_get_key_from_key_comps(key_comps, cfg_key, MAX_OPT_KEY_LEN);
            
            if(pp_cfg_is_enabled(&enabled, cfg_key) != PP_ERR && !enabled) {
                /* config key changed from disabled to enabled */
                need_pw_change = 1;
            }
        }
        
        /* set tx again! */
        pp_cfg_tx_set_thread_ctx(tx);
    }
    
    if(need_pw_change) {
        ret = um_user_strong_pw_cb(ctx);
    }
    
    return ret;
}

int um_user_strong_pw_cb(pp_cfg_chg_ctx_t *ctx UNUSED) {
    /* force all users to change pw on next login */
    
    vector_t *uids = um_get_all_uids_by_gid(-1);
    u_int u, uids_sz = vector_size(uids);
    
    for(u = 0; u < uids_sz; ++u) {
        pp_cfg_set_enabled(1, usr_need_pw_change_key, vector_get_int(uids, u));
    }        
    
    vector_delete(uids);
    
    return PP_SUC;
}

int pp_um_strong_pw_add_to_history(const char *username, 
                                   const char *password, int flags) {
    int history_sz, history_max, uid, src, dst, ret = PP_SUC;
    
    assert(username);
    assert(password);
    
    /* name exists? */
    if (PP_ERR == (uid = pp_um_user_get_uid(username))) {
        return PP_ERR;
    }

    pp_cfg_get_int(&history_max, strong_pw_history_key);
    pp_cfg_get_int(&history_sz, usr_history_sz_key, uid);
    
    /* increase history by one (new pw) and check, if size is still valid */
    if(++history_sz > history_max) {
        history_sz = history_max;
    }
    
    /* push history one entry backwards.
       overwrite last entry (sz - 1), if history is already full */
    for(dst = history_sz - 1, src = dst - 1; src >= 0; --src, --dst) {
        char *hash;
        
        pp_cfg_get(&hash, usr_history_key, uid, src);
        pp_cfg_set(hash, usr_history_key, uid, dst);
        
        free(hash);
    }
    
    /* now set the given password */
    if(flags & PP_UM_AUTH_PASSWD_AS_MD5_HASH) {
        /* we already got the hash */
        pp_cfg_set(password, usr_history_key, uid, 0);
    } else {
        /* create the hash and set it */
        char md5_hash[MD5_DIGEST_LENGTH];
        char md5_hash_str[MD5_DIGEST_LENGTH * 2 + 1];
        
        MD5(password, strlen(password), md5_hash);
        pp_md5_hash_to_str(md5_hash, md5_hash_str);

        pp_cfg_set(md5_hash_str, usr_history_key, uid, 0);
    }
    
    pp_cfg_set_int(history_sz, usr_history_sz_key, uid);
    
    return ret;
}


