/**
 * libpp_vsc: ddc.c
 *
 * Implements DDC/EDID support to deliver monitor
 * information to the video card
 *
 * Currently supported is DDC2B, EDID 1.3
 */

#include <ctype.h>
#include <fcntl.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>

#include <pp/base.h>
#include <pp/vsc.h>
#include <pp/i2c.h>
#include <liberic_pthread.h>

#include "context.h"
#include "ddc.h"
#include "debug.h"

#define DEVICE_NAME		"/dev/i2c-ddc"
#define CHAR2VAL(c)		((c) - 'A' + 1)

/* the EDID structure */
#define MANUFACTURER_ID(c1, c2, c3)	(cpu_to_be16_const((CHAR2VAL(c1) << 10) | (CHAR2VAL(c2) <<  5) | CHAR2VAL(c3)))
#define PRODUCT_ID(x)			(cpu_to_le16_const(x))
#define SERIAL(x)			(cpu_to_le32_const(x))
#define MANUFACTURE_WEEK(x)		(x)
#define MANUFACTURE_YEAR(x)		((u_int8_t)((x) - 1990))

typedef struct vendor_product_s {
    u_int16_t	manufacturer_id;
    u_int16_t	product_code;
    u_int32_t	serial;
    u_int8_t	manufacture_week;
    u_int8_t	manufacture_year;
} __attribute__ ((__packed__)) vendor_product_t ;

typedef struct version_s  {
    u_int8_t	version;
    u_int8_t	revision;
} __attribute__ ((__packed__)) version_t;

#define SIGNAL_LEVEL_0700_0300			0
#define SIGNAL_LEVEL_0714_0286			1
#define SIGNAL_LEVEL_1000_0400			2
#define SIGNAL_LEVEL_0700_0000			3
#define DFP_10_COMPATIBLE			1
#define GAMMA(x)				(((x) * 100) - 100)
#define GAMMA_UNDEFINED				0xFF
#define DISPLAY_TYPE_MONO			0
#define DISPLAY_TYPE_RGB			1
#define DISPLAY_TYPE_NONRGB			2

typedef struct display_s {
    BITFIELD7(	u_int8_t,						
		sync_serration_dfp:	1,     /* serration = hsync during vsync */ 
 	        sync_green:		1,	/* ad == 1 -> 0 */		
		sync_composite:		1,	/* ad == 1 -> 0 */ 
		sync_separate:		1,	/* ad == 1 -> 0 */ 
    		setup:			1,	/* ad == 1 -> 0 */			
		signal_level:		2,	/* ad == 1 -> 0 */			
		ad:			1);
    u_int8_t	max_image_size_h;		/* in cm, or 0 */			
    u_int8_t	max_image_size_v;		/* in cm, or 0 */			
    u_int8_t	gamma;
    BITFIELD7(	u_int8_t,
		default_gtf_supported:	1,
		has_preferred_timing:	1,
		srgb:			1,
		display_type:		2,
		active_off:		1,
		suspend:		1,
		standby:		1);
		
} __attribute__ ((__packed__)) display_t;

#define COLOR_HI(x)	(((u_int16_t)((x + 0.0005) * 1024) & 0x03FC) >> 2)
#define COLOR_LO(x,s)	(((u_int16_t)((x + 0.0005) * 1024) & 0x0003) << s)

#define COLORVAL_RED	1.000
#define COLORVAL_GREEN	1.000
#define COLORVAL_BLUE	1.000
#define COLORVAL_WHITE	1.000

#define COLORLO_SHIFT_RED_X	6
#define COLORLO_SHIFT_RED_Y	4
#define COLORLO_SHIFT_GREEN_X	2
#define COLORLO_SHIFT_GREEN_Y	0
#define COLORLO_SHIFT_BLUE_X	6
#define COLORLO_SHIFT_BLUE_Y	4
#define COLORLO_SHIFT_WHITE_X	2
#define COLORLO_SHIFT_WHITE_Y	0

typedef struct color_s {
    BITFIELD4(	u_int8_t,
		gy_low:		2,
		gx_low:		2,
		ry_low:		2,
		rx_low:		2);
    BITFIELD4(	u_int8_t,
		wy_low:		2,
		wx_low:		2,
		by_low:		2,
		bx_low:		2);
    u_int8_t	red_x;
    u_int8_t	red_y;
    u_int8_t    green_x;
    u_int8_t	green_y;
    u_int8_t    blue_x;
    u_int8_t	blue_y;
    u_int8_t    white_x;
    u_int8_t	white_y;
} __attribute__ ((__packed__)) color_t;

typedef struct established_timings_s {
    BITFIELD8(	u_int8_t,			
		support_800x600at60:	1,
		support_800x600at56:	1,	
		support_640x480at75:	1,	
		support_640x480at72:	1,	
		support_640x480at67:	1,	
		support_640x480at60:	1,	
		support_720x400at88:	1,	
		support_720x400at70:	1
		);
    BITFIELD8(	u_int8_t,			
		support_1280x1024at75:	1,
		support_1024x768at75:	1,	
		support_1024x768at70:	1,	
		support_1024x768at60:	1,	
		support_1024x768at87i:	1,	
		support_832x624at75:	1,	
		support_800x600at75:	1,	
		support_800x600at72:	1	
		);
    BITFIELD2(	u_int8_t,
		support_reserved:	7,
		support_1152x870at75:	1	
		);
} __attribute__ ((__packed__)) established_timings_t;

#define STANDARD_ASPECT_16to10		0
#define STANDARD_ASPECT_4to3		1
#define STANDARD_ASPECT_5to4		2
#define STANDARD_ASPECT_15to9		3
// #define STANDARD_TIMING_UNDEFINED	0x01

#define ACTIVE_H_CALC(x)                ((x / 8) - 31)
#define REFRESH_CALC(x)                 (x - 60)
#define STANDARD_ACTIVE_H_UNUSED        0x01
#define STANDARD_ASPECT_UNUSED          0x00
#define STANDARD_REFRESH_UNUSED         0x01
typedef struct standard_timing_s {
    u_int8_t	active_h;
    BITFIELD2(	u_int8_t,		
		refresh:	6,	
		aspect:		2	
		);
} __attribute__ ((__packed__)) standard_timing_t;

#define TIMING_CLOCK(x)				(cpu_to_le16_const((x) / 10))
#define TIMING_12_LOW(x)			((x) & 0xFF)
#define TIMING_12_HI(x)				(((x) & 0x0F00) >> 4)
#define TIMING_6_LOW(x)				((x) & 0x0F)
#define TIMING_6_HI(x)				(((x) & 0x30) >> 4)
#define TIMING_NON_INTERLACED			1
#define TIMING_INTERLACED			1
#define TIMING_STEREO_NORMAL			0
/* FIXME: add other stereo stuff when its needed */

#define TIMING_SYNC_ANALOG_COMPOSITE		0
#define TIMING_SYNC_BI_ANALOG_COMPOSITE		1
#define TIMING_SYNC_DIGITAL_COMPOSITE		2
#define TIMING_SYNC_DIGITAL_SEPARATE		3

#define TIMING_SYNC_ANALOG_NO_SERRATE		0
#define TIMING_SYNC_ANALOG_SERRATE		1
#define TIMING_SYNC_ANALOG_SYNC_ON_GREEN	0
#define TIMING_SYNC_ANALOG_SYNC_ON_RGB		1
#define TIMING_SYNC_DIGITALC_NO_SERRATE		0
#define TIMING_SYNC_DIGITALC_SERRATE		1
#define TIMING_SYNC_DIGITALC_HSYNC_POL_LOW	0
#define TIMING_SYNC_DIGITALC_HSYNC_POL_HI	1
#define TIMING_SYNC_DIGITALS_VSYNC_POL_LOW	0
#define TIMING_SYNC_DIGITALS_VSYNC_POL_HI	1
#define TIMING_SYNC_DIGITALS_HSYNC_POL_LOW	0
#define TIMING_SYNC_DIGITALS_HSYNC_POL_HI	1

typedef struct detailed_timing_s {
    u_int16_t	clock;			/* kHZ / 10 */
    u_int8_t	active_h_low;
    u_int8_t	blank_h_low;
    BITFIELD2(	u_int8_t,			
		active_h_hi:		4,
		blank_h_hi:		4
		);
    u_int8_t	active_v_low;
    u_int8_t    blank_v_low;
    BITFIELD2(	u_int8_t,			
		active_v_hi:		4,
		blank_v_hi:		4
		);
    u_int8_t	sync_offset_h_low;
    u_int8_t	sync_width_h_low;
    BITFIELD2(	u_int8_t,			
		sync_width_v_low:	4,
		sync_offset_v_low:	4	
		);
    BITFIELD4(	u_int8_t,			
		sync_width_v_hi:	2,
		sync_width_h_hi:	2,	
		sync_offset_v_hi:	2,	
		sync_offset_h_hi:	2
		);
    u_int8_t	image_size_h_low;
    u_int8_t	image_size_v_low;
    BITFIELD2(	u_int8_t,			
		image_size_v_hi:	4,
		image_size_h_hi:	4	
		);
    u_int8_t	border_h;
    u_int8_t	border_v;
    BITFIELD6(	u_int8_t,			
		stereo2:		1,
		sync_rgb_comp_pol:	1,	
		sync_serrate_pol:	1,	
		sync_type:		2,	
		stereo1:		2,	
		interlace:		1	
		);
} __attribute__ ((__packed__)) detailed_timing_t;

#define MONITOR_DESC_NO_SECONDARY_TF	0x00
#define MONITOR_DESC_SECONDARY_TF	0x02

#define MONITOR_DESC_PAD_SEC_TF		0x00
#define MONITOR_DESC_PAD_NO_SEC_TF	0x0A

typedef struct monitor_desc_range_s {
    u_int8_t	min_v;	/* Hz */
    u_int8_t	max_v; 
    u_int8_t	min_h;	/* kHz */
    u_int8_t	max_h;
    u_int8_t    clock;	/* MHz / 10 */

    /* FIXME: secondary generalized timing formula */
    u_int8_t	gtf_supported;
    u_int8_t	gtf_pad;
    u_int8_t	gtf_start_freq;
    u_int8_t	gtf_c2;
    u_int16_t	gtf_m;
    u_int8_t	gtf_k;
    u_int8_t	gtf_j2; 
} __attribute__ ((__packed__)) monitor_desc_range_t;

#define MONITOR_WHITEPOINT_UNUSED	0x00
/* FIXME: add whitepoint macros analog to color stuff above */
typedef struct monitor_desc_whitepoint_s {
    u_int8_t	index;
    u_int8_t	xy_low;
    u_int8_t	x;
    u_int8_t	y;
    u_int8_t	gamma;
} __attribute__ ((__packed__)) monitor_desc_whitepoint_t;

#define MONITOR_COLOR_PAD1		0x0A
#define MONITOR_COLOR_PAD2		0x20
typedef struct monitor_desc_color_s {
    monitor_desc_whitepoint_t white[2];
    u_int8_t	pad;
    u_int8_t	pad2[2];
} __attribute__ ((__packed__)) monitor_desc_color_t;


#define MONITOR_DESC_TIMING_PAD		0x0A
typedef struct monitor_desc_timing_s {
    standard_timing_t	timing[6];
    u_int8_t		pad;		/* must be 0x0A */
} __attribute__ ((__packed__)) monitor_desc_timing_t;

#define MONITOR_DESC_SERIAL		0xFF
#define MONITOR_DESC_STRING		0xFE
#define MONITOR_DESC_RANGELIMITS	0xFD
#define MONITOR_DESC_NAME		0xFC
#define MONITOR_DESC_COLOR		0xFB
#define MONITOR_DESC_TIMING		0xFA
/* 0xF9 - 0x11 currently undefined */
#define MONITOR_DESC_UNUSED		0x10
/* 0x00 - 0x0F manufacturer specified */

typedef union monitor_data_u {
    char serial[13];
    char string[13];
    monitor_desc_range_t range;
    char name[13];
    monitor_desc_color_t color;
    monitor_desc_timing_t timing;    
} __attribute__ ((__packed__)) monitor_data_t;

typedef struct monitor_descriptor_s {
    u_int16_t		flag1;	/* always 0 in monitor descriptor */
    u_int8_t		flag2;	/* always 0 in monitor descriptor */
    u_int8_t		type;
    u_int8_t		flag3;	/* always 0 in monitor descriptor */
    monitor_data_t	data;
} __attribute__ ((__packed__)) monitor_descriptor_t;

typedef union timing_monitor_u {
    detailed_timing_t timing;
    monitor_descriptor_t monitor;
} __attribute__ ((__packed__)) timing_monitor_t;

typedef struct edid_s {
    u_int8_t			header[8];
    vendor_product_t		vendor_product;
    version_t			version;
    display_t			display;
    color_t			color;
    established_timings_t	established_timings;
    standard_timing_t		standard_timings[8];
    timing_monitor_t		timing_monitor[4];
    u_int8_t			extension_count;
    u_int8_t			checksum;
} __attribute__ ((__packed__)) edid_t;

static edid_t edid_template = {
    header:		{ 0x00,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0x00 },
    vendor_product:	{
	manufacturer_id:	MANUFACTURER_ID('P','E','P'),
	product_code:		PRODUCT_ID(0x0001),
	serial:			SERIAL(0x00000002),
	manufacture_week:	MANUFACTURE_WEEK(1),
	manufacture_year:	MANUFACTURE_YEAR(2005),
    },
    version:		{
	version:		1,
	revision:		3,
    },
    display:		{
	ad:			1,
	signal_level:		SIGNAL_LEVEL_0700_0300, /* ??? */
	setup:			0,
	sync_separate:		0,
	sync_composite:		0,
	sync_green:		0,
	sync_serration_dfp:	0,
	max_image_size_h:	34,
	max_image_size_v:	27,
	gamma:			GAMMA_UNDEFINED,
	standby:		0,
	suspend:		0,
	active_off:		0,
	display_type:		DISPLAY_TYPE_RGB,
	srgb:			0,
	has_preferred_timing:	1,
	default_gtf_supported:	0
    },
    color:		{
	rx_low:			COLOR_LO(COLORVAL_RED, COLORLO_SHIFT_RED_X),
	ry_low:			COLOR_LO(COLORVAL_RED, COLORLO_SHIFT_RED_Y),
	gx_low:			COLOR_LO(COLORVAL_GREEN, COLORLO_SHIFT_GREEN_X),
	gy_low:			COLOR_LO(COLORVAL_GREEN, COLORLO_SHIFT_GREEN_Y),
	bx_low:			COLOR_LO(COLORVAL_BLUE, COLORLO_SHIFT_BLUE_X),
	by_low:			COLOR_LO(COLORVAL_BLUE, COLORLO_SHIFT_BLUE_Y),
	wx_low:			COLOR_LO(COLORVAL_WHITE, COLORLO_SHIFT_WHITE_X),
	wy_low:			COLOR_LO(COLORVAL_WHITE, COLORLO_SHIFT_WHITE_Y),
	red_x:			COLOR_HI(COLORVAL_RED),
	red_y:			COLOR_HI(COLORVAL_RED),
	green_x:		COLOR_HI(COLORVAL_GREEN),
	green_y:		COLOR_HI(COLORVAL_GREEN),
	blue_x:			COLOR_HI(COLORVAL_BLUE),
	blue_y:			COLOR_HI(COLORVAL_BLUE),
	white_x:		COLOR_HI(COLORVAL_WHITE),
        white_y:		COLOR_HI(COLORVAL_WHITE)
    },
    established_timings:	{
	support_720x400at70:	1,
	support_720x400at88:	0,
	support_640x480at60:	1,
	support_640x480at67:	1,
	support_640x480at72:	1,
	support_640x480at75:	1,
	support_800x600at56:	1,
	support_800x600at60:	1,
	support_800x600at72:	0,
	support_800x600at75:	0,
	support_832x624at75:	0,
	support_1024x768at87i:	0,
	support_1024x768at60:	0,
	support_1024x768at70:	0,
	support_1024x768at75:	0,
	support_1280x1024at75:	0,
	support_1152x870at75:	0,
	support_reserved:	7
    },
    standard_timings:	{
	{ active_h: ACTIVE_H_CALC(1600), aspect: STANDARD_ASPECT_4to3, refresh: REFRESH_CALC(60) },
	{ active_h: ACTIVE_H_CALC(1280), aspect: STANDARD_ASPECT_5to4, refresh: REFRESH_CALC(60) },
	{ active_h: ACTIVE_H_CALC(1280), aspect: STANDARD_ASPECT_4to3, refresh: REFRESH_CALC(60) },
	{ active_h: ACTIVE_H_CALC(1152), aspect: STANDARD_ASPECT_4to3, refresh: REFRESH_CALC(60) },

	{ active_h: ACTIVE_H_CALC(1024), aspect: STANDARD_ASPECT_4to3, refresh: REFRESH_CALC(60) },
	{ active_h: STANDARD_ACTIVE_H_UNUSED, aspect: STANDARD_ASPECT_UNUSED, refresh: STANDARD_REFRESH_UNUSED },
	{ active_h: STANDARD_ACTIVE_H_UNUSED, aspect: STANDARD_ASPECT_UNUSED, refresh: STANDARD_REFRESH_UNUSED },
	{ active_h: STANDARD_ACTIVE_H_UNUSED, aspect: STANDARD_ASPECT_UNUSED, refresh: STANDARD_REFRESH_UNUSED }
    },

#define TOLOW(x) (x & 0xff)
#define TOHI(x) ((x & 0xff00) >> 8)
    
    timing_monitor: {
	{
	    timing: {
		clock: cpu_to_le16_const(10800),
		active_h_low: TOLOW(256),
		active_h_hi: TOHI(256),

		blank_h_low: TOLOW(1432),
		blank_h_hi: TOHI(1432),

		blank_v_low: TOLOW(1066),
		blank_v_hi: TOHI(1066),

		active_v_low: TOLOW(0),
		active_v_hi: TOHI(0),

		sync_offset_h_low: TOLOW(48),
		sync_offset_h_hi: TOHI(48),

		sync_width_h_low: TOLOW(112),
		sync_width_h_hi: TOHI(112),

		sync_width_v_low: TOLOW(3),
		sync_width_v_hi: TOHI(3),

		sync_offset_v_low: TOLOW(1),
		sync_offset_v_hi: TOHI(1),
		
		image_size_h_low: TOLOW(376),
		image_size_h_hi: TOHI(376),
		
		image_size_v_low: TOLOW(301),
		image_size_v_hi: TOHI(301),
		
		border_h: 0,
		border_v: 0,
		stereo2: 0,
		sync_rgb_comp_pol: 0,
		sync_serrate_pol: 0,
		sync_type: 0,
		stereo1: 0,
		interlace: 0,
	    }
	},
	
	{
	    timing: {
		clock: cpu_to_le16_const(25170),
		active_h_low: TOLOW(128),
		active_h_hi: TOHI(128),
		
		blank_h_low: TOLOW(672),
		blank_h_hi: TOHI(672),
		
		blank_v_low: TOLOW(355),
		blank_v_hi: TOHI(355),
		
		active_v_hi: TOHI(94),
		active_v_low: TOLOW(94),

		sync_offset_h_low: TOLOW(16),
		sync_offset_h_hi: TOHI(16),

		sync_width_h_hi: TOHI(608),
		sync_width_h_low: TOLOW(608),

		sync_width_v_low: TOLOW(2),
		sync_width_v_hi: TOHI(2),

		sync_offset_v_low: TOLOW(5),
		sync_offset_v_hi: TOHI(5),
		
		image_size_h_hi: TOHI(376),
		image_size_h_low: TOLOW(376),
		
		image_size_v_low: TOLOW(301),
		image_size_v_hi: TOHI(301),
		
		border_h: 0,
		border_v: 0,
		stereo2: 0,
		sync_rgb_comp_pol: 0,
		sync_serrate_pol: 0,
		sync_type: 0,
		stereo1: 0,
		interlace: 0,
	    }
	},
	{
	    monitor: {
		type: MONITOR_DESC_RANGELIMITS,
		flag1: cpu_to_le16_const(0), flag2:0, flag3:0,
		data: {
		    range: {
			min_v: 56,
			max_v: 76,
			min_h: 31,
			max_h: 83,
			clock: 15,
			gtf_supported: 0,
			gtf_pad: 0,
			gtf_start_freq: 0,
			gtf_c2: 0,
			gtf_m: cpu_to_le16_const(0),
			gtf_k: 0,
			gtf_j2: 0
		    }
		}
	    }
	},
	{
	    monitor: {
		type: MONITOR_DESC_NAME,
		flag1: cpu_to_le16_const(0), flag2:0, flag3:0,
		data: {
		    name: "KIRA\n      "
		}
	    }
	},
    },
    extension_count:	0,
    checksum:		0
};

/* DDC data structure, per context */
struct ddc_data_s {
    int		fd;
    edid_t	edid;
};
typedef struct ddc_data_s ddc_data_t;


/* function declarations */
static void initialize_edid(edid_t *edid);
#ifdef PP_FEAT_DDC_DEVICE
static int set_edid(vsc_context_t *context, ddc_data_t *data);
#endif
static u_int8_t calc_checksum(edid_t *edid);

#ifdef PP_FEAT_UPDATE_EDID
static int send_edid_to_eeprom(vsc_context_t *context UNUSED, ddc_data_t *data,
			       char* i2c_devname, unsigned char i2c_addr);
#endif

/*
 * external interface
 */

int
ddc_init(vsc_context_t *context)
{
    int ret = -1;
   
    context->ddc_data = malloc(sizeof(ddc_data_t));

    initialize_edid(&context->ddc_data->edid);

#ifdef PP_FEAT_DDC_DEVICE
    if ((context->ddc_data->fd = open(DEVICE_NAME, O_RDWR)) == -1) {
	goto bail_free;
    }

    set_edid(context, context->ddc_data);
#endif

#ifdef PP_FEAT_UPDATE_EDID
    send_edid_to_eeprom(context, context->ddc_data,
			"/dev/i2c-faraday-5", 0x50);
#endif
    
    D(D_NOTICE, "DDC initialized successfully (%s)\n", DEVICE_NAME);
    ret = 0;
    
    // shut compiler up
    goto bail_free;

 bail_free:
    free(context->ddc_data);
    
    return ret;
}

void
ddc_cleanup(vsc_context_t *context)
{
#ifdef PP_FEAT_DDC_DEVICE
    close(context->ddc_data->fd);
#endif
    
    free(context->ddc_data);
}


/* helper functions for edid access and initialization */
static void
initialize_edid(edid_t *edid)
{
    memcpy(edid, &edid_template, sizeof(edid_t));
    
    calc_checksum(edid);
}

static u_int8_t
calc_checksum(edid_t *edid)
{
    u_int8_t *edid_array = (u_int8_t*) edid;
    u_char i, sum = 0;
    
    for (i = 0; i < sizeof(edid_t) - 1; i++) {
	sum += edid_array[i];
    }

    edid->checksum = 256 - sum;

    return edid->checksum;
}

#ifdef PP_FEAT_DDC_DEVICE
static int
set_edid(vsc_context_t *context UNUSED, ddc_data_t *data)
{
    int ret = -1;
    int wret;

    if ((wret = write(data->fd, &data->edid, sizeof(edid_t))) != sizeof(edid_t)) {
	D(D_ERROR, "writing DDC EDID to kernel failed (write ret: %d, edit_t size: %d\n", wret, sizeof(edid_t));
	goto bail;
    }

    D(D_NOTICE, "DDC EDID set to kernel\n");
    ret = 0;
    
 bail:
    return ret;
}
#endif

#ifdef PP_FEAT_UPDATE_EDID
static int
send_edid_to_eeprom(vsc_context_t *context UNUSED, ddc_data_t *data,
		    char* i2c_devname, unsigned char i2c_addr)
{
    u_char handle;
    int error, ret = PP_ERR;
    u_int i;

    handle = pp_i2c_open_device(i2c_devname, &error);
    if (error == PP_I2C_ERROR) {
	D(D_ERROR, "Error opening i2c device '%s'\n", i2c_devname);
	return ret;
    }

    for (i = 0; i < sizeof(edid_t); i++) {
	u_char buf;
	u_char *edid_data = (u_char*)&data->edid;
	
	memcpy(&buf, &edid_data[i], 1);
	
	pp_i2c_tx_byte_data(handle, i2c_addr, i, buf, &error);
	if (error == PP_I2C_ERROR) {
	    D(D_ERROR, "Error bursting data from to '%s' from offset %d\n", i2c_devname, i);
	    goto bail;
	}

	/* wait maximum page write cycle time */
	usleep(10000); 
    }

    ret = PP_SUC;
    
 bail:
    pp_i2c_close(handle);
    return ret;
}
#endif

#ifdef PP_FEAT_DDC
void pp_vsc_get_ddc_edid(unsigned char edid[128]) {
    initialize_edid((edid_t *)edid);
}
#endif
