/**
 * \file vsc.h
 *
 * This is the interface to the video sampling controller FPGA.
 * It handles video mode/settings related operations.
 *
 * Note: This API supports several VSC channels at once. This
 *       is not completely implemented yet.
 * 
 * (c) 2004 Peppercon AG
 */

#ifndef _PP_VSC_H
#define _PP_VSC_H

#include <sys/types.h>

#include <lara.h>
#include <liberic_session.h>
#include <liberic_config.h>
#include <pp/base.h>
#include <pp/vsc_types_external.h>
#include <pp/cfg.h>

#define MAX_CUSTOM_VIDEO_MODES 10


typedef enum {
    PP_DONT_BLANK_SCREEN,
    PP_BLANK_SCREEN
} pp_blank_screen_t;


/**
 * Event types for vsc mode changes
 */
typedef enum {
    VSC_MODE_CHANGE_VALID,       // video mode recognized and activated
    VSC_MODE_CHANGE_UNKNOWN,     // unknown video mode, not displayed
    VSC_MODE_CHANGE_NO_SIGNAL,   // no incoming signal, (usually powerdown)
    VSC_MODE_CHANGE_TO_UNSTABLE  // begin of mode change, not displayed
} pp_vsc_mode_change_t;

/**
 * Event struct for video mode changes
 * @see vsc_mode_change_t
 */
typedef struct {
    pp_vsc_mode_change_t type;
    
    // additional information for type=VSC_MODE_CHANGE_VALID
    unsigned int res_x;        // horizontal resolution
    unsigned int res_y;        // vertical resolution
    // additional information for type=VSC_MODE_CHANGE_VALID and !PP_FEATURE_VSC_PANEL_INPUT
    unsigned int refresh;         // vertical refresh rate
    unsigned char is_defaultmode; // set to 1 if this is a default mode, 0 else
} pp_vsc_event_mode_change_t;

/**
 * Event types for the auto adjust process
 */
typedef enum {
    VSC_AA_SUCCESS,           // auto adjust is done successfully
    VSC_AA_IN_PROGRESS,       // this event type is called repeatedly until state changes
    VSC_AA_INTERRUPTED,       // auto adjust interrupted
    VSC_AA_FAILED             // auto adjust failed
} pp_vsc_aa_progress_t;

/**
 * Event structure for vsc auto adjustment
 */
typedef struct {
    pp_vsc_aa_progress_t type;
    // no additional data
} pp_vsc_event_aa_progress_t;


/**
 * describes the type of the vsc event
 */
typedef enum {
    VSC_EVENT_MODE_CHANGE,     // video mode change event
    VSC_EVENT_AA_ADJUSTMENT    // auto adjustment event
} pp_vsc_event_type_t;

/**
 * vsc event, sent to the encoder using a callback
 * on various vsc state changes
 *
 * @see vsc_event_type_t
 */
typedef struct {
    pp_vsc_event_type_t type; // the type of the event

    // actual event, depending on the type
    union {
        pp_vsc_event_mode_change_t mode_change;
        pp_vsc_event_aa_progress_t aa_progress;
    } event;
} pp_vsc_event_t;


/**
 * encoder descriptor
 *
 * Defines the callbacks that an encoder has to
 * implement to connect to the vsc. Callbacks in
 * this structure must not be NULL.
 */
typedef struct {
    /**
     * Notify the encoder of any VSC event that occured.
     * 
     * @param video_link  The channel (vsc id) that sent this event
     * @param event       A ptr to the event description. Invalid after the call returns.
     *                    Please copy if necessary
     * @see vsc_event_t
     */
    void (*vsc_event)(unsigned char video_link, pp_vsc_event_t* event);

    /**
     * Return if any session is active (someone is logged in to the device or not)
     * 
     * @param video_link  The channel (vsc id) that sent this request
     * @param id          FIXME
     * @return            1 if at least on session is active, else 0
     */
    int (*encoder_is_active)(unsigned char video_link, eric_session_int_id_t id);
} pp_vsc_encoder_desc_t;


/**
 * Initialize the video sampling controller and
 * its controlling library.
 * @return  0  on success
 * @return -1  on error
 */
int pp_vsc_init(void);

/**
 * Cleanup the vsc and release all hardware.
 */
void pp_vsc_cleanup(void);

/**
 * Initiate a vsc reset.
 * @param video_link  The vsc channel (vsc id) that will be reset.
 * @param type        Set to VSC_RESET_SOFTWARE_ONLY or 
 *                    to VSC_RESET_PROGRAM_FPGA
 */
void pp_vsc_reset(unsigned char video_link, vsc_reset_type_t type);

/**
 * Connect an encoder implementation to the vsc. This
 * function may only be called once to initially add encoder
 * callbacks. Several encoders can be registered in parallel
 * by calling this function multiple times. Encoders cannot
 * be disconnected.
 * 
 * Note: currently only one encoder is implemented. support
 *       for multiple encoders can be added transparently when
 *       needed.
 * 
 * @param encoder  encoder decription structure containing
 *                 the information and callbacks of the actual encoder
 * @return         PP_SUC if successful
 *                 PP_ERR on error
 */
int pp_vsc_connect_encoder(pp_vsc_encoder_desc_t* encoder);


/**
 * Check if the specified video_link is correctly
 * initialized and ready to use.
 *
 * @param video_link  vsci id (channel) to check, in case its a single
 *                    ported hardware, it is always 0
 * @return            1 if active, else 0
 */
pp_bool_t  pp_vsc_is_active(unsigned char video_link);

/**
 * Set local video state.
 * 
 * @param video_link  vsc id (channel) to control, in case its a single
 *                    ported hardware, it is always 0
 * @param state       1 = enable local video, 0 = disable local video
 * @return            0 if successful
 */
int pp_vsc_set_local_video(unsigned char video_link, unsigned char state);

/**
 * Get the current format of the framebuffer.
 * 
 * @param video_link  vsc id (channel) to use, in case its a single
 *                    ported hardware, it is always 0
 * @param fb          framebuffer format structure, supplied by user.
 * @return            0 if successful
 *                    -1 if error
 */
int pp_vsc_get_fb_format(unsigned char video_link, fb_format_info_t *fb);

/**
 * Get the current h- and v-freq.
 * 
 * @param video_link  vsc id (channel) to use, in case its a single
 *                    ported hardware, it is always 0
 * @param hf          horizontal (line) freq in Hz
 * @param vf          vertical (frame) freq in Hz
 * @return            0 if successful
 *                    -1 if error
 */
int pp_vsc_get_freqs(unsigned char video_link, int *hf, int *vf);

/**
 * Returns the video input signal state. Possible return
 * values are on, off, unknown.
 * 
 * @param video_link   vsc id (channel) to use, in case its a single
 *                     ported hardware, it is always 0
 * @return             video input signal state
 */
video_signal_state_t pp_vsc_has_signal(unsigned char video_link);

/**
 * Returns the video output signal state
 * 
 * Result is OUTPUT_VALID if the VSC is programmed,
 * the input signal is active and the video mode is known 
 *
 * @param video_link  vsc id (channel) to use, in case its a single
 *                    ported hardware, it is always 0
 * @return            video output signal state
 */
vsc_output_state_t pp_vsc_output_state(unsigned char video_link);

/**
 * Create a test picture.
 * 
 * @param video_link  vsc id (channel) to use, in case its a single
 *                    ported hardware, it is always 0
 * @param show        a 1 shows the test picture, 0 reverts to original picture
 */
void pp_vsc_testpic(unsigned char video_link, unsigned char show);

/**
 * Returns if the VSC is able to use a display enable signal
 * to determine the active video area when using panel input.
 * 
 * @param video_link  vsc id (channel) to use, in case its a single
 *                    ported hardware, it is always 0
 * @return	      0 when display_enable is not available
 *		      1 when display_enable IS available
 *		      -1 when its not possible to determine this (yet)
 */
int pp_vsc_have_display_enable(unsigned char video_link);

#ifdef PP_FEAT_VIDEO_CUSTOM_MODES
void pp_vsc_update_custom_modes(unsigned char video_link);
#endif /* PP_FEAT_VIDEO_CUSTOM_MODES */

#ifdef PP_FEAT_DDC
void pp_vsc_get_ddc_edid(unsigned char edid[128]);
#endif

/* -------------------------- video settings ------------------------- */
int pp_vsc_set_video_setting(unsigned char video_link, unsigned char setting, unsigned short value);
int pp_vsc_get_video_settings(unsigned char video_link, video_settings_t * settings);
int pp_vsc_save_video_settings(unsigned char video_link, vsc_setting_type_t types);
int pp_vsc_cancel_video_settings(unsigned char video_link, vsc_setting_type_t types);
int pp_vsc_reset_video_settings(unsigned char video_link, vsc_setting_type_t types);
int pp_vsc_trigger_autoadjust(unsigned char video_link);

/**
 * Puts the VSC lib in an auto-adjustment like mode to only show the
 * image rescan error while sampling complete frames
 * Used to debug the noise level of the analog video input
 * 
 * @param video_link	vsc id (channel) to use, in case its a single
 *			ported hardware, it is always 0
 *
 * @return		PP_SUC when successfully toggled
 *			PP_ERR when VSC is not yet initialized
 */
int pp_vsc_debug_ire_toggle(unsigned char video_link);

#endif /* _PP_VSC_H */
