/*********************************************************************
 * libpp_vsc: init.c
 *
 * contains the vsc initialization and some functions
 * available to the 'outside world'
 *
 ********************************************************************/

#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include <liberic_pthread.h>
#include <pp/hal_common.h>
#include <pp/cfg.h>
#include <pp/grab.h>
#include <pp/vsc.h>
#include <lara.h>

#include "adc.h"
#include "adc_io.h"
#include "ddc.h"
#include "debug.h"
#include "main.h"
#include "mode.h"
#include "settings.h"
#include "timer.h"
#include "vsc_funcs.h"
#include "vsc_io.h"

#define TIMER_TICK_DEFAULT_FAST_MS	2
#define TIMER_TICK_DEFAULT_MS		100

static const char * vsc_misc_options[] = {
    "video.diff_tolerance",
    "video.vsc.sun_mode",
    NULL /* must be present and the last */
};

#ifdef PP_FEAT_VIDEO_CUSTOM_MODES
static const char * vsc_custom_modes_options[] = {
    "video.tft_custom",
    NULL /* must be present and the last */
};
#endif

static int initialized = 0;
static u_int current_num_video_links = 0;

static pthread_mutexattr_t errcheck_mtx_attr;

static void init_local_video(vsc_context_t *context);
static int reconf_misc_ch(pp_cfg_chg_ctx_t *ctx);
#ifdef PP_FEAT_VIDEO_CUSTOM_MODES
static int reconf_custom_modes_ch(pp_cfg_chg_ctx_t *ctx);
#endif
static void propchange_handler(pp_propchange_listener_t * listener, 
			       unsigned short prop, unsigned short prop_flags);

PP_DECLARE_PROPCHANGE_LISTENER(propchange_listener, propchange_handler);


int
pp_vsc_init(void)
{

    vsc_context_t *context = NULL;

    u_int   i;
    u_int  i2c_addr_mask=0xa,
	   mux_channel_mask=0xc;
    int num_vscs;

    int ret = -1;
   
    num_vscs = pp_hal_common_get_vsc_cnt();
    if (num_vscs <= 0) {
	printf("pp_hal_common_get_vsc_cnt returns as count of vscs: %d, aborting\n", num_vscs);
	return -1;
    }
    current_num_video_links = (u_int)num_vscs;
    assert( current_num_video_links > 0 );
 
#if !defined(PP_FEAT_VSC_PANEL_INPUT)
    if ((ret = adc_io_init())) {
    	pp_log("adc_io_init failed\n");
    	goto error;
    }
#endif

#if 0
    /* set FPGA JTAG pins (debug) */
    pp_gpio_set(PP_GPIO_DEV_IBM, PP_IBM_GPIO_MASK(8),
                PP_IBM_GPIO_MASK(2) | PP_IBM_GPIO_MASK(4),  0);
#endif

    vsc_contexts = (vsc_context_t *)malloc(sizeof(vsc_context_t) * current_num_video_links);
    memset(vsc_contexts, 0, sizeof(vsc_context_t) * current_num_video_links);

    for( i=0; i<current_num_video_links; i++ ){
	vsc_contexts[i].adc_i2c_addr = 0x4c + ((i2c_addr_mask >> i) & 0x1);
	vsc_contexts[i].adc_i2c_mux_channel = (mux_channel_mask >> i) & 0x1;
#ifdef LARA_KACY
	vsc_contexts[i].adc_i2c_mux_addr= 0x70; 
#else
	vsc_contexts[i].adc_i2c_mux_addr= 0x0; 
#endif 
    }

    pthread_mutexattr_init(&errcheck_mtx_attr);
    pthread_mutexattr_settype(&errcheck_mtx_attr, PTHREAD_MUTEX_ERRORCHECK);  
    
    if (mode_init() != 0) {
	D(D_ERROR, "mode_init failed\n");
	goto error;
    }

    for( i=0; i<current_num_video_links; i++){

	// check if VSC device is there 
	if (pp_grab_fd((u_char)i)  == -1) {
	    D(D_ERROR, "VSC device node not found, grabber not initialized?, video_link=%d\n", i);
	    goto error;
	}
	
	context = &vsc_contexts[i];

	// initialize context and controller channel
	context->id = i;
	context->fd = pp_grab_fd( (u_char)i );
	if (vsc_initialize(context, VSC_DEFAULT_INIT_METHOD) == PP_TRUE) {
	    context->thread_run = 1;
	    context->sync_state = INITIAL;
	    context->input_sync_signals = UNDETERMINED;
	    context->sync_signals = UNDETERMINED;
	    context->kvm_unit = -1;
	    context->kvm_port = -1;
	    /* TODO: check whether PP_FALSE works with 1601ip */
	    context->kvm_port_changed = PP_FALSE;
	    context->force_irq = PP_FALSE;
	    context->force_mode_setup = PP_FALSE;
	    context->current_mode = NULL;
	    context->adc_sync_mode = UNDEFINED_SYNC;
	    context->mode_valid = PP_FALSE;
	    context->resume_reset = 0;
	    context->reset_request = RESET_REQUEST_NONE;
	    context->feats = vsc_features_default;
	    
	    init_local_video(context);
		    
	    // initialize mutexes
	    pthread_mutex_init(&context->kvm_port_mtx, &errcheck_mtx_attr);
		    
	    // initialize hardware and each part
#if !defined(PP_FEAT_VSC_PANEL_INPUT)
	    if (adc_init(context) != 0) {
		    D(D_ERROR, "adc_init failed, video_link=%d\n", i);
		    goto error;
	    }
#endif
	    if (vsc_init(context) != 0) {
		    D(D_ERROR, "vsc_init failed, video_link=%d\n", i);
		    goto error;
	    }
	    if (timer_init(context, TIMER_TICK_DEFAULT_MS, TIMER_TICK_DEFAULT_FAST_MS) != 0) {
		D(D_ERROR, "timer_init failed, video_link=%d\n", i);
		goto error;
	    }

#ifdef PP_FEAT_DDC
	    if (ddc_init(context) != 0) {
		D(D_ERROR, "ddc_init failed, video_link=%d\n", i);
		goto error;    
	    }
#endif
	    if (eric_pthread_create(&context->thread, 0, 64 * 1024, vsc_thread_func, context)) {
		    D(D_ERROR, "Cannot create thread, video_link=%d\n", i);
		goto error;
	    }
	}
    }

    /* add change handlers for settings */
    for (i = 0; vsc_misc_options[i] != NULL; ++i) {
	pp_cfg_add_change_listener(reconf_misc_ch, vsc_misc_options[i]);
    }
#ifdef PP_FEAT_VIDEO_CUSTOM_MODES
    for (i = 0; vsc_custom_modes_options[i] != NULL; ++i) {
	pp_cfg_add_change_listener(reconf_custom_modes_ch, vsc_custom_modes_options[i]);
    }
#endif

    pp_propchange_add(&propchange_listener, PP_PROP_KVM_PORT_SWITCHED);
    
    initialized = 1;

    ret = 0;

 error:
    return ret;
}

void
pp_vsc_cleanup(void)
{
    u_int i;

    vsc_context_t *context = NULL;

    pp_propchange_remove_all(&propchange_listener);

    /* add change handlers for settings */
    for (i = 0; vsc_misc_options[i] != NULL; ++i) {
	pp_cfg_rem_change_listener(reconf_misc_ch, vsc_misc_options[i]);
    }
#ifdef PP_FEAT_VIDEO_CUSTOM_MODES
    for (i = 0; vsc_custom_modes_options[i] != NULL; ++i) {
	pp_cfg_rem_change_listener(reconf_custom_modes_ch, vsc_custom_modes_options[i]);
    }
#endif

    for( i=0; i<current_num_video_links; i++ ){
	context = &vsc_contexts[i];

	if (context->vsc_active == PP_TRUE) {
	    context->thread_run = 0;
	    pthread_join(context->thread, NULL);

#ifdef PP_FEAT_DDC
	    ddc_cleanup(context);
#endif
	    pthread_mutex_destroy(&context->kvm_port_mtx);
	}
    }

    pthread_mutexattr_destroy(&errcheck_mtx_attr);

    mode_cleanup();

#if !defined(PP_FEAT_VSC_PANEL_INPUT)
    adc_io_cleanup();
#endif

    free(vsc_contexts);

    initialized = 0;
}

void
pp_vsc_reset(unsigned char video_link, vsc_reset_type_t type)
{
    vsc_contexts[video_link].reset_request = (type == VSC_RESET_PROGRAM_FPGA) ? RESET_REQUEST_HARD : RESET_REQUEST_SOFT;
}

int
pp_vsc_connect_encoder(pp_vsc_encoder_desc_t* encoder) {
    /* note: currently only one encoder implemented */
    assert(vsc_encoder_hook == NULL);
    assert(encoder != NULL);
    
    // register
    vsc_encoder_hook = encoder;
    return PP_SUC;
}

static int
reconf_misc_ch(pp_cfg_chg_ctx_t *ctx UNUSED)
{
    vsc_context_t *context = NULL;
    u_int i;
	
    for( i=0; i<current_num_video_links; i++ ){
	context = &vsc_contexts[i];
	D(D_BLABLA, "Reconfigure VSC subsystem video_link=%d\n", i );
	if (context->vsc_active == PP_TRUE) {
	    vsc_update_diff_thresholds();
	    vsc_update_sun_mode(context);
	}
    }

    return PP_SUC;
}

#ifdef PP_FEAT_VIDEO_CUSTOM_MODES
static int
reconf_custom_modes_ch(pp_cfg_chg_ctx_t *ctx UNUSED)
{
    u_int i;
    D(D_BLABLA, "Reconfigure VSC subsystem (custom modes)\n");
    for(i=0; i<current_num_video_links; i++){
	pp_vsc_update_custom_modes(i);
    }

    return PP_SUC;
}
#endif /* PP_FEAT_VIDEO_CUSTOM_MODES */

static void
propchange_handler(pp_propchange_listener_t * listener UNUSED, 
		   unsigned short prop, unsigned short prop_flags UNUSED)
{
    u_int i;
    vsc_context_t *context = NULL;

    for( i=0; i<current_num_video_links; i++ ){
	context = &vsc_contexts[i];
	switch (prop) {
	    case PP_PROP_KVM_PORT_SWITCHED:
		if (context->vsc_active == PP_TRUE) {
		    D(D_BLABLA, "ERIC_KVM_PORT_SWITCHED, video_link=%d\n", i);
		    MUTEX_LOCK(&context->kvm_port_mtx);
		    context->kvm_port_changed = PP_TRUE;
		    MUTEX_UNLOCK(&context->kvm_port_mtx);
		}
	    break;
	    default:
		pp_log("%s(): Unhandled property %hu received.\n", ___F, prop);
	    break;
	}
    }
}

int
pp_vsc_set_local_video(unsigned char video_link, unsigned char state)
{
    vsc_context_t *context = &vsc_contexts[video_link];

    context->local_video_state = state;

    if (context->vsc_active == PP_TRUE) {
        vsc_tx_masked(context, VSC_REG_MR,
                      context->local_video_state ? ~MR_VIDDISABLE : MR_VIDDISABLE,
                      MR_VIDDISABLE);
    }

    return 0;
}

int
pp_vsc_get_fb_format(unsigned char video_link, fb_format_info_t *fb)
{
    vsc_context_t *context = &vsc_contexts[video_link];
    
    assert(fb);
    
    if (context->mode_valid != PP_TRUE)	{
	return -1;
    }
    
    mode_get_format(context, fb);
    return 0;
}

int
pp_vsc_get_freqs(unsigned char video_link, int *hf, int *vf)
{
    vsc_context_t *context = &vsc_contexts[video_link];

    if (context->mode_valid != PP_TRUE) return -1;

    if (hf) *hf = (int)context->newFreqH * 10;
    if (vf) *vf = (int)(context->newFreqV + 5) / 10;
    return 0;
}

pp_bool_t
pp_vsc_is_active(unsigned char video_link)
{
    return vsc_contexts[video_link].vsc_active;
}

video_signal_state_t
pp_vsc_has_signal(unsigned char video_link)
{
    return vsc_has_signal(&vsc_contexts[video_link]);
}

vsc_output_state_t
pp_vsc_output_state(unsigned char video_link)
{
    return vsc_output_state(&vsc_contexts[video_link]);
}

void
pp_vsc_testpic(unsigned char video_link, unsigned char show)
{
    vsc_context_t *context = &vsc_contexts[video_link];

    if (show) {
	vsc_tx_masked(context, VSC_REG_MR,MR_TEST_PIC, MR_TEST_PIC);
    } else {
	vsc_tx_masked(context, VSC_REG_MR, 0, MR_TEST_PIC);
    }
}

int
pp_vsc_have_display_enable(unsigned char video_link)
{
    vsc_context_t *context = &vsc_contexts[video_link];
   
    return context->feats.display_enable;
}

#ifdef PP_FEAT_VIDEO_CUSTOM_MODES
void
pp_vsc_update_custom_modes(unsigned char video_link)
{
    vsc_context_t *context = &vsc_contexts[video_link];

    if (context->vsc_active == PP_TRUE) {
	invalidate_custom_modes();
    }    
}
#endif /* PP_FEAT_VIDEO_CUSTOM_MODES */

int
pp_vsc_debug_ire_toggle(unsigned char video_link)
{
    vsc_context_t *context = &vsc_contexts[video_link];

    if (context->vsc_active == PP_TRUE) {
	context->aa_data.ire_request = PP_TRUE;
	return PP_SUC;
    }
    
    return PP_ERR;
}

/* internal functions */

static void
init_local_video(vsc_context_t *context)
{
    assert(context);

/* FIXME: enable_local_video for RIPC */
#if !defined(PRODUCT_RIPCKIMXN)
    u_char video_link = context->id;
    int video_enabled = 0;

    pp_cfg_is_enabled(&video_enabled, "video.enable_local");
    pp_vsc_set_local_video(video_link, video_enabled ? 1 : 0);
#else
    (void)context;
#endif
}

