 /*********************************************************************
 * libpp_vsc: main.c
 *
 * vsc main state machine and handling functions
 *
 ********************************************************************/

#include <unistd.h>
#include <sys/ioctl.h>
#include <pp/base.h>
#include <pp/grab.h>
#include <liberic_misc.h>
#include <pp/rfb.h>
#include <liberic_session.h>
#include <pp/vsc.h>
#include <lara.h>
#include <pp/kvm.h>
#include <pp/intl.h>

#include "adc.h"
#include "context.h"
#include "debug.h"
#include "main.h"
#include "mode.h"
#include "settings.h"
#include "timer.h"
#include "vsc_funcs.h"
#include "vsc_io.h"

#define NO_SIGNAL_MESSAGE	_("No signal")

static const unsigned int UNSTABLE_WAIT_MS		= 500;   /* time to wait until sync gets stable */
static const unsigned int UNSTABLE_INSTABLE_MS		= 1000;  /* time to accept ongoing sync IRQs */
static const unsigned int POWERDOWN_WAIT_MS		= 1000;  /* when no signal, time until idle */
static const unsigned int UNKNOWN_MODE_WAIT_MS		= 2000;  /* when unknown mode, time until idle */
static const unsigned int AUTO_AA_WAIT_MS		= 10000; /* after getting a stable mode, time to start auto adjust */
static const unsigned int VFIFO_RESET_WAIT_MS		= 20;    /* wait for VFIFO reset acknowledge from VSC */
static const unsigned int PRESTABLE_WAIT_MS		= 20;    /* after setting sync input, wait a bit to settle */
static const unsigned int POSTSTABLE_WAIT_MS		= 20;    /* wait to re-enable sync irq input */
static const unsigned int RESET_NO_SIGNAL_WAIT_MS	= 1000;  /* time for 'no signal' when VSC stays in RESET state */

/* encoder hook, set by init.c */
pp_vsc_encoder_desc_t* vsc_encoder_hook = NULL;

/* private prototypes */
static void handle_sync_state(vsc_context_t *context);
#if !defined(PP_FEAT_VSC_PANEL_INPUT) && !defined(PRODUCT_ERICXP) && !defined(PRODUCT_ERICG4)
static void handle_kvm_port_change(vsc_context_t *context);
#endif
static void handle_sync_input_change(vsc_context_t *context);
static void handle_diff_mask(vsc_context_t *context);

static void handle_initial_state(vsc_context_t *context);
static void handle_reset_state(vsc_context_t *context);
static void handle_reset_ram_state(vsc_context_t *context);
static void handle_idle_state(vsc_context_t *context);
static void handle_powerdown_state(vsc_context_t *context);
static void handle_unknown_state(vsc_context_t *context);
static void handle_unstable_state(vsc_context_t *context);
static void handle_vfifo_reset_state(vsc_context_t *context);
static void handle_prestable_state(vsc_context_t *context);
static void handle_stable_state(vsc_context_t *context);
static void handle_poststable_state(vsc_context_t *context);

static void switch_to_unstable(vsc_context_t *context, pp_blank_screen_t bs);
static void switch_to_powerdown(vsc_context_t *context);
static void switch_to_idle(vsc_context_t *context);
static void switch_to_initial(vsc_context_t *context, int do_program_fpga);
static void switch_to_reset(vsc_context_t *context);
static void switch_to_reset_ram(vsc_context_t *context);
static void switch_to_final(vsc_context_t *context);
static void switch_to_vfifo_reset(vsc_context_t *context);
static void switch_to_prestable(vsc_context_t *context);
static void switch_to_stable(vsc_context_t *context);
static void switch_to_poststable(vsc_context_t *context);
static void switch_to_aa(vsc_context_t *context, pp_bool_t auto_aa);

static void auto_aa_check_session(eric_session_int_id_t s, va_list args);
static void reinitialize_vsc(vsc_context_t *context);

#ifdef PP_FEAT_VSC_PANEL_INPUT
static void reinitialize_vsc_digital(vsc_context_t *context);
#else
static void reinitialize_vsc_analog(vsc_context_t *context);
#endif

int
vsc_init(vsc_context_t *context UNUSED)
{
    /* actual configuraton is done in INITIAL state
       to repeat it manually when VSC is reset */
    
    return 0;
}

void*
vsc_thread_func(void* arg)
{
    vsc_context_t *context = (vsc_context_t*) arg;

    if (nice(2) == -1) {
	D(D_ERROR, "setting nice level failed");
    }  
    while (context->thread_run) {	
	timer_tick(context,
		   (context->sync_state == AUTOADJUST) ? TICK_FAST : TICK_STANDARD);

	// initial and reset states have priority,
	// we have to (re)initialize the vsc in this case
	if (context->sync_state == INITIAL ||
	    context->sync_state == RESET   ||
	    context->sync_state == RESET_RAM) {
	    
	    handle_sync_state(context);
	    continue;
	}
#if !defined(PP_FEAT_VSC_PANEL_INPUT) && !defined(PRODUCT_ERICXP) && !defined(PRODUCT_ERICG4)
	handle_kvm_port_change(context);
#endif
	handle_sync_input_change(context);
	handle_diff_mask(context);	
	handle_sync_state(context);
    }

    return NULL;
}

static void
handle_sync_state(vsc_context_t *context)
{
    pp_bool_t force_irq = PP_FALSE;

    // initial state has priority
    if (context->sync_state != INITIAL &&
	context->sync_state != RESET   &&
	context->sync_state != RESET_RAM) {
	
	// check for IRQ from VSC or a virtual one from some function
	if (context->force_irq) {
	    force_irq = PP_TRUE;
	    context->force_irq = PP_FALSE;
	}
	
	if (force_irq || vsc_get_sync_irq(context, PP_FALSE)) {
	    if (context->ignore_irq == PP_FALSE) {
		switch_to_unstable(context, PP_BLANK_SCREEN);
	    }
	}	
    }
   
    switch (context->sync_state) {
      case INITIAL:
	  handle_initial_state(context);
	  break;
      case RESET:
	  handle_reset_state(context);
	  break;
      case RESET_RAM:
	  handle_reset_ram_state(context);
	  break;
      case IDLE:
	  handle_idle_state(context);
	  break;
      case AUTOADJUST:
#ifdef PP_FEAT_VSC_PANEL_INPUT
	  if (context->feats.display_enable) {
	      context->sync_state = IDLE;
	  }
#endif
	  if (context->aa_data.ire_request) {
	      context->aa_data.ire_process = PP_FALSE;
	      context->aa_data.ire_request = PP_FALSE;
	  }
	  vsc_aa_advance(context);
	  if (context->aa_data.state == AA_IDLE) {
	      context->sync_state = IDLE;
	      
	      // auto aa is done, maybe someone wants to start action
	      if (context->auto_aa) {
		  DCH(D_VERBOSE, "Sending 'auto aa done' propchange\n");
		  pp_propchange_enqueue(PP_PROP_AUTO_AUTO_ADJUSTMENT_DONE,
				        context->id);
	      }
	  }
	  break;
      case UNSTABLE:
	  handle_unstable_state(context);
	  break;
      case VFIFO_RESET:
	  handle_vfifo_reset_state(context);
	  break;
      case PRESTABLE:
	  handle_prestable_state(context);
	  break;
      case STABLE:
	  handle_stable_state(context);
	  break;
      case POSTSTABLE:
	  handle_poststable_state(context);
	  break;
      case POWERDOWN:
	  handle_powerdown_state(context);
	  break;
      case UNKNOWN:
	  handle_unknown_state(context);
	  break;
      case FINAL:
	  /* cleanup? */
	  context->thread_run = PP_FALSE;
	  break;
      default:
	  DCH(D_ERROR, "Unknown vsc sync state %d\n", context->sync_state);
	  context->sync_state = INITIAL;
	  break;
    }
}

#if !defined(PP_FEAT_VSC_PANEL_INPUT) && !defined(PRODUCT_ERICXP) && !defined(PRODUCT_ERICG4)
static void
handle_kvm_port_change(vsc_context_t *context)
{
    MUTEX_LOCK(&context->kvm_port_mtx);
    if (context->kvm_port_changed == PP_TRUE) {
	u_char kvm_unit;
	u_short kvm_port;
	int video_link = context->id; 
	if (video_link < 0) goto bail;
	if (PP_FAILED(pp_kvm_get_unit_port_for_video_link(video_link, &kvm_unit, &kvm_port))) {
	    DCH(D_ERROR, "Could not get current kvm unit and port, not switching\n");
	    goto bail;
	}
	context->kvm_unit = kvm_unit;
	context->kvm_port = kvm_port;
	
	set_kvm_local_settings(context);

	context->kvm_port_changed = PP_FALSE;

	switch_to_unstable(context, PP_DONT_BLANK_SCREEN);
    }
bail:
    MUTEX_UNLOCK(&context->kvm_port_mtx);
}
#endif /* !PP_FEAT_VSC_PANEL_INPUT */

static void
handle_sync_input_change(vsc_context_t *context)
{
#if !defined(PP_FEAT_VSC_PANEL_INPUT)
    adc_set_sync_input(context);
#endif
    
    if (sync_detection(context) == SYNC_CHANGED) {
#if !defined(PP_FEAT_VSC_PANEL_INPUT)
	sync_configure_input(context);
#endif
	switch_to_unstable(context, PP_BLANK_SCREEN);
    }
}

static void
handle_diff_mask(vsc_context_t *context)
{
    if (timer_active(context, DIFF_THRESHOLD_TIMER) &&
	timer_expired(context, DIFF_THRESHOLD_TIMER)) {

	DCH(D_VERBOSE, "Diff Threshold reactivated\n");
	vsc_set_diff_threshold(context);
	timer_deactivate(context, DIFF_THRESHOLD_TIMER);
    }
}

static void
handle_initial_state(vsc_context_t *context)
{    
    reinitialize_vsc(context);

    if (vsc_rx_reg(context, VSC_REG_SR) == SR_RESET) {
	timer_set(context, RESET_NO_SIGNAL_TIMER, RESET_NO_SIGNAL_WAIT_MS);
	switch_to_reset(context);
    }  else {
	switch_to_reset_ram(context);
    }   
}

static void
handle_reset_state(vsc_context_t *context)
{
    if (vsc_rx_reg(context, VSC_REG_SR) == SR_RESET) {
	if (timer_active(context, RESET_NO_SIGNAL_TIMER) &&
	    timer_expired(context, RESET_NO_SIGNAL_TIMER)) {

	    /* VSC does not come back from reset, so it has
	       no input clock */
	    pp_vsc_event_t event;
	    DCH(D_VERBOSE, "Reset active a long time -> No signal\n");
	    if (vsc_encoder_hook && vsc_encoder_hook->vsc_event) {
		event.type = VSC_EVENT_MODE_CHANGE;
		event.event.mode_change.type = VSC_MODE_CHANGE_NO_SIGNAL;
		vsc_encoder_hook->vsc_event(context->id, &event);
	    }

	    timer_deactivate(context, RESET_NO_SIGNAL_TIMER);
	}	
	return;
    }

    switch_to_reset_ram(context);
}

static void
handle_reset_ram_state(vsc_context_t *context)
{
    if (vsc_rx_reg(context, VSC_REG_SR) != SR_RESET) {
	switch_to_unstable(context, PP_BLANK_SCREEN);
    }
}

static void
handle_unstable_state(vsc_context_t *context)
{
    if (timer_expired(context, UNSTABLE_TIMER)) {
	vsc_sync_irq_state(context, PP_FALSE);	
	
	DCH(D_VERBOSE, "Sync stable, preparing\n");
	switch_to_vfifo_reset(context);
    }
}

static void
handle_vfifo_reset_state(vsc_context_t *context)
{
    u_char done = 0;

    if (ioctl(context->fd, PPIOCVSCGETADCRESOK, &done) == -1) {
	if (errno != EAGAIN) {
	    DCH(D_ERROR, "Error during ioctl(PPIOCVSCGETADCRESOK): %s\n",
	      strerror(errno));
	}
    }

    /* VFIFO reset done either be signaled from VSC
       or assumed after a certain time (on KIRA R01) */
       
    if (done) {
	DCH(D_VERBOSE, "VFIFO reset done (ack from VSC)\n");
	switch_to_prestable(context);
    } else if (timer_expired(context, VFIFO_RESET_TIMER)) {
	DCH(D_VERBOSE, "VFIFO reset done (timer)\n");
	switch_to_prestable(context);
    }
}

static void
handle_prestable_state(vsc_context_t *context)
{
    if (context->input_sync_signals == V || context->input_sync_signals == NONE) {
	context->sync_signals = context->input_sync_signals;
	switch_to_powerdown(context);
	goto bail;
    }

    if (timer_expired(context, PRESTABLE_TIMER)) {
	switch_to_stable(context);
    }

bail:
    // rearm 'unstable instable' timer
    timer_deactivate(context, UNSTABLE_INSTABLE_TIMER);
}

static void
handle_stable_state(vsc_context_t *context)
{
    mode_det_res_t mode_det_res;
    
    if (sync_measurement(context) != 0) {
	switch_to_powerdown(context);
	goto bail;
    }

    mode_det_res = mode_detection(context);
    if (mode_det_res == MODE_DET_OK) {
	mode_setup(context, context->current_mode);
#if !defined(PP_FEAT_VSC_PANEL_INPUT)
	if (context->current_mode != context->last_mode) {
	    timer_set(context, AUTO_AA_TIMER, AUTO_AA_WAIT_MS);
	}
#endif

	/* if we suspended because of resetting, now is
	   the right time to start the grabber again */
	while (context->resume_reset) {
	    pp_grab_suspend(context->id, GRAB_RESUME);
	    context->resume_reset--;
	}
	switch_to_poststable(context);
    } else if (mode_det_res == MODE_DET_FAILED) {
	/* set the nearest mode in comp sync to have a stable pll
	   but don't announce and use it */
	if (context->adc_sync_mode == COMPOSITE_SYNC) {
	    mode_setup(context, context->nearest_mode);
	}
	context->sync_state = UNKNOWN;
	timer_set(context, UNKNOWN_MODE_TIMER, UNKNOWN_MODE_WAIT_MS);
    } else if (mode_det_res == MODE_DET_TRY_AGAIN) {
	switch_to_unstable(context, PP_BLANK_SCREEN);
    }
    
    if (!context->feats.extd_sync_switch) {
	/* we have a stable PLL now, use it as VSC sync input */
	vsc_set_sync_input(context, HSYNC_ADC, VSYNC_ADC);
    }

 bail:
    // rearm 'unstable instable' timer
    timer_deactivate(context, UNSTABLE_INSTABLE_TIMER);
     
    return;
}

static void
handle_poststable_state(vsc_context_t *context)
{
    if (timer_active(context, POSTSTABLE_TIMER) &&
	timer_expired(context,POSTSTABLE_TIMER)) {
	
	timer_deactivate(context, POSTSTABLE_TIMER);

	/* re-enable sync irq */
	vsc_sync_irq_state(context, PP_TRUE);
	switch_to_idle(context);
    }
}

static void
handle_idle_state(vsc_context_t *context)
{   
    // setup mode, requested on settings cancelation
    if (context->force_mode_setup) {
	context->force_mode_setup = FALSE;
	mode_setup(context, context->current_mode);
    }

    // put into auto adjustment mode when requested
    if (context->aa_data.request ||
	context->aa_data.ire_request) {
	
	switch_to_aa(context, PP_FALSE);
	
	context->aa_data.request = PP_FALSE;
	context->aa_data.ire_request = PP_FALSE;
    }

    // auto aa mode
    if (timer_active(context, AUTO_AA_TIMER) &&
	timer_expired(context, AUTO_AA_TIMER)) {
	
	timer_deactivate(context, AUTO_AA_TIMER);
	/* AUTO_AA deactivated because it nerved */
	//switch_to_aa(context, PP_TRUE);
    }

    // go back to inital state when resetting, possibly reprogramming fpga
    if (context->reset_request != RESET_REQUEST_NONE) {
	switch_to_initial(context, (context->reset_request == RESET_REQUEST_HARD) ?
			  VSC_DEFAULT_INIT_METHOD : VSC_DONT_PROGRAM_FPGA);
	context->reset_request = RESET_REQUEST_NONE;
    }
}

static void
handle_powerdown_state(vsc_context_t *context)
{
    pp_vsc_event_t event;
    
    if (timer_expired(context, POWERDOWN_TIMER)) {
        if (vsc_encoder_hook && vsc_encoder_hook->vsc_event) {
            event.type = VSC_EVENT_MODE_CHANGE;
            event.event.mode_change.type = VSC_MODE_CHANGE_NO_SIGNAL;
            vsc_encoder_hook->vsc_event(context->id, &event);
        }
        DCH(D_NOTICE, "Powerdown\n");
        switch_to_idle(context);
    }
}

static void
handle_unknown_state(vsc_context_t *context)
{
    pp_vsc_event_t event;
    
    if (timer_expired(context, UNKNOWN_MODE_TIMER)) {
        if (vsc_encoder_hook && vsc_encoder_hook->vsc_event) {
            event.type = VSC_EVENT_MODE_CHANGE;
            event.event.mode_change.type = VSC_MODE_CHANGE_UNKNOWN;
            vsc_encoder_hook->vsc_event(context->id, &event);
        }
	DCH(D_NOTICE, "Unknown video mode\n");

	vsc_sync_irq_state(context, PP_TRUE);
	switch_to_idle(context);
    }
}

static void
switch_to_unstable(vsc_context_t *context, pp_blank_screen_t bs)
{
    pp_vsc_event_t event;
    
#if !defined(PP_FEAT_VSC_PANEL_INPUT)
    vsc_aa_stop(context);   
#endif

    if (!context->feats.extd_sync_switch &&
	!timer_active(context, UNSTABLE_INSTABLE_TIMER)) {

	DCH(D_VERBOSE, "Starting UNSTABLE_INSTABLE timer\n");
	    
	timer_set(context, UNSTABLE_INSTABLE_TIMER, UNSTABLE_INSTABLE_MS);
    }
    
    if (context->feats.extd_sync_switch ||
	!timer_expired(context, UNSTABLE_INSTABLE_TIMER)) {

	if (bs == PP_BLANK_SCREEN) {
	    event.type = VSC_EVENT_MODE_CHANGE;
	    event.event.mode_change.type = VSC_MODE_CHANGE_TO_UNSTABLE;
	    if (vsc_encoder_hook && vsc_encoder_hook->vsc_event) {
		vsc_encoder_hook->vsc_event(context->id, &event);
	    }
	}

	DCH(D_VERBOSE, "Switching to UNSTABLE state\n");
	
	context->sync_state = UNSTABLE;
	timer_set(context, UNSTABLE_TIMER, UNSTABLE_WAIT_MS);
    } else {
	DCH(D_VERBOSE, "Avoid switching to UNSTABLE state\n");
    }
}

static void
switch_to_powerdown(vsc_context_t *context)
{
#if !defined(PP_FEAT_VSC_PANEL_INPUT)
    vsc_aa_stop(context);
#endif
    DCH(D_VERBOSE, "Switching to POWERDOWN state\n");
    context->sync_state = POWERDOWN;
    timer_set(context, POWERDOWN_TIMER, POWERDOWN_WAIT_MS);

    vsc_sync_irq_state(context, PP_TRUE);
}

static void
switch_to_idle(vsc_context_t *context)
{
    DCH(D_VERBOSE, "Switching to IDLE state\n");
    context->sync_state = IDLE;
}

static void
switch_to_initial(vsc_context_t *context, int do_program_fpga)
{  
    pp_grab_suspend(context->id, GRAB_SUSPEND);
    context->resume_reset++;

    if (vsc_initialize(context, do_program_fpga) == PP_FALSE) {
	switch_to_final(context);
	return;
    }

    if (do_program_fpga) {
    } else {
	
	context->vsc_active = PP_TRUE;
	if (ioctl(context->fd, PPIOCVSCACTIVATE, NULL) == -1) {
	    if (errno != EAGAIN) {
		DCH(D_ERROR, "Error during ioctl(PPIOCVSCACTIVATE): %s\n", strerror(errno));
	    }
	}
    }

    DCH(D_VERBOSE, "Switching to INITIAL state\n");
    context->sync_state = INITIAL;
}

static void
switch_to_reset(vsc_context_t *context)
{
    DCH(D_VERBOSE, "Switching to RESET state\n");
    context->sync_state = RESET;
}

static void
switch_to_reset_ram(vsc_context_t *context)
{
    if (context->feats.sdram_initword) {
	vsc_init_sdram(context);
	DCH(D_VERBOSE, "Switching to RESET_RAM state\n");
	context->sync_state = RESET_RAM;
    } else {
	switch_to_unstable(context, PP_BLANK_SCREEN);
    }
}

static void
switch_to_final(vsc_context_t *context)
{
    DCH(D_VERBOSE, "Switching to FINAL state\n");
    context->sync_state = FINAL;
}

static void
switch_to_vfifo_reset(vsc_context_t *context)
{
    DCH(D_VERBOSE, "Switching to VFIFO_RESET state\n");
    context->sync_state = VFIFO_RESET;

    DCH(D_VERBOSE, "Resetting VSC Video FIFO\n");   
    vsc_tx_masked(context, VSC_REG_MR, MR_VFIFO_RESET, MR_VFIFO_RESET);
    usleep(50000);
    vsc_tx_masked(context, VSC_REG_MR, 0, MR_VFIFO_RESET);

    timer_set(context, VFIFO_RESET_TIMER, VFIFO_RESET_WAIT_MS);
}

static void
switch_to_prestable(vsc_context_t *context)
{
    DCH(D_VERBOSE, "Switching to PRESTABLE state\n");
    context->sync_state = PRESTABLE;

    // set input for mode detection
#ifdef PP_FEAT_VSC_PANEL_INPUT
    if (context->feats.extd_sync_switch) {
	vsc_set_sync_reg_input(context, HSYNC_ADC, VSYNC_ADC);
    }
#else
    if (context->adc_sync_mode == COMPOSITE_SYNC) {
	/* use the original hsync because pll may
	   not yet be locked, but since this is a composite
	   one avoid getting sync changes all the time */
	if (context->feats.extd_sync_switch) {
	    vsc_set_sync_reg_input(context, HSYNC_BUF, VSYNC_ADC);
	} else {
	    vsc_set_sync_input(context, HSYNC_BUF, VSYNC_ADC);
	}
    } else {
	/* sync for measurements from buffer */
	if (context->feats.extd_sync_switch) {
	    vsc_set_sync_reg_input(context, HSYNC_BUF, VSYNC_BUF);
	} else {
	    vsc_set_sync_input(context, HSYNC_BUF, VSYNC_BUF);
	}
	
    }
#endif

    timer_set(context, PRESTABLE_TIMER, PRESTABLE_WAIT_MS);
}

static void
switch_to_stable(vsc_context_t *context)
{
    DCH(D_VERBOSE, "Switching to STABLE state\n");
    context->sync_state = STABLE;
}

static void
switch_to_poststable(vsc_context_t *context)
{
    DCH(D_VERBOSE, "Switching to POSTSTABLE state\n");
    context->sync_state = POSTSTABLE;

    timer_set(context, POSTSTABLE_TIMER, POSTSTABLE_WAIT_MS);
}

static void
switch_to_aa(vsc_context_t *context, pp_bool_t auto_aa)
{    
    if (auto_aa == PP_FALSE) {	
	if (context->aa_data.ire_request == PP_TRUE) {
	    DCH(D_VERBOSE, "Toggling ImageRescanError output\n");
	    context->aa_data.ire_process = PP_TRUE;	    
	} else {
	    DCH(D_VERBOSE, "Switching to AUTOADJUST state\n");	    
	}
	context->auto_aa = PP_FALSE;
	context->sync_state = AUTOADJUST;
	return;
    }

    // for auto aa we first check if there is an RFB session active
    context->auto_aa = PP_TRUE;
    eric_session_for_all(auto_aa_check_session, context);

    if (context->auto_aa == PP_TRUE) {
	DCH(D_VERBOSE, "Switching to AUTOADJUST state automatically\n");
	context->sync_state = AUTOADJUST;
	return;
    }
}

static void
auto_aa_check_session(eric_session_int_id_t s, va_list args)
{
    vsc_context_t *context;
    
    context = va_arg(args, vsc_context_t*);

    if (vsc_encoder_hook && vsc_encoder_hook->encoder_is_active) {
        if (vsc_encoder_hook->encoder_is_active(context->id, s)) context->auto_aa = PP_FALSE;
    }
}

/* reprogram the VSC registers after a reprogramming
   or a master reset */
static void
reinitialize_vsc(vsc_context_t *context)
{
    DCH(D_VERBOSE, "Initializing VSC\n");

    vsc_check_version_features(context);
    
#ifdef PP_FEAT_VSC_PANEL_INPUT
    reinitialize_vsc_digital(context);
#else
    reinitialize_vsc_analog(context);
#endif

    /* enable sync irq */
    vsc_tx_masked(context, VSC_REG_IMR, IMR_SFC, IMR_SFC);

    /* enable extended sync switching on newer VSCs,
       always set sample input to ADC */
    if (context->feats.extd_sync_switch) {
	vsc_tx_masked(context, VSC_REG_MR, MR_SYNCSWITCH, MR_SYNCSWITCH);
	vsc_set_sync_sample_input(context, HSYNC_ADC, VSYNC_ADC);
#ifdef PP_FEAT_VSC_PANEL_INPUT
	vsc_set_sync_irq_input(context, HSYNC_ADC, VSYNC_ADC);
#else
	vsc_set_sync_irq_input(context, HSYNC_BUF, VSYNC_BUF);
#endif
    } else {
	/* set initial sync input */
#ifdef PP_FEAT_VSC_PANEL_INPUT
	vsc_set_sync_input(context, HSYNC_ADC, VSYNC_ADC);
#else
	/* no stable PLL yet, better use the incoming signals */
	vsc_set_sync_input(context, HSYNC_BUF, VSYNC_BUF);
#endif
    }
       
    vsc_sync_irq_state(context, PP_TRUE);
    vsc_update_sync_deltas(context);
    vsc_update_sun_mode(context);
    
    vsc_set_diff_threshold(context);
   
    /* we rely on the next mode setup for triggering the grab
       refresh, don't do it explicitly */
    pp_grab_suspend(context->id, GRAB_RESUME);
}

#ifdef PP_FEAT_VSC_PANEL_INPUT
static void
reinitialize_vsc_digital(vsc_context_t *context)
{
    /* workaround for the first buggy KIRA100 R01 chips */
    if (!context->feats.display_enable) {
#ifdef PP_FEAT_VSC_SINGLE_PIXEL
	vsc_tx_masked(context, VSC_REG_MR, MR_MODE(MR_MODE_ADC1X), MR_MODE_MASK);
#else
	vsc_tx_masked(context, VSC_REG_MR, MR_MODE(MR_MODE_ADC2X), MR_MODE_MASK);
#endif
    } else {

#ifdef PP_FEAT_VSC_SINGLE_PIXEL
	vsc_tx_masked(context, VSC_REG_MR, MR_MODE(MR_MODE_DVI1X), MR_MODE_MASK);
#else
	vsc_tx_masked(context, VSC_REG_MR, MR_MODE(MR_MODE_DVI2X), MR_MODE_MASK);
#endif

	/* setup display enable delay, amount depends on the panel
	   output type (direct on eRIC2, DVI receiver on msi and amd, ...) */
#if defined (PRODUCT_ERIC2)
	vsc_tx_masked(context, VSC_REG_MR, DE_DELAY(3), DE_DELAY_MASK);
#elif defined(KIRA_KIMSMIG4)
	vsc_tx_masked(context, VSC_REG_MR, DE_DELAY(4), DE_DELAY_MASK);
#else
	vsc_tx_masked(context, VSC_REG_MR, DE_DELAY(2), DE_DELAY_MASK);
#endif
	
	vsc_sync_check_reset(context);
    }
}
#else /* !PP_FEAT_VSC_PANEL_INPUT */
static void
reinitialize_vsc_analog(vsc_context_t *context)
{
# ifdef PP_FEAT_VSC_SINGLE_PIXEL
    vsc_tx_masked(context, VSC_REG_MR, MR_MODE(MR_MODE_ADC1X), MR_MODE_MASK);
# else
    vsc_tx_masked(context, VSC_REG_MR, MR_MODE(MR_MODE_ADC2X), MR_MODE_MASK);
# endif
    vsc_tx_masked(context, VSC_REG_MR, context->local_video_state ? 0 : MR_VIDDISABLE, MR_VIDDISABLE);

    if (set_global_settings(context)) {
	DCH(D_ERROR, "set_global_settings failed, continuing anyway\n");
    }
}
#endif /* !PP_FEAT_VSC_PANEL_INPUT */
